
##############################################################################
#
# Copyright (c) 2003-2017 by The University of Queensland
# http://www.uq.edu.au
#
# Primary Business: Queensland, Australia
# Licensed under the Apache License, version 2.0
# http://www.apache.org/licenses/LICENSE-2.0
#
# Development until 2012 by Earth Systems Science Computational Center (ESSCC)
# Development 2012-2013 by School of Earth Sciences
# Development from 2014 by Centre for Geoscience Computing (GeoComp)
#
##############################################################################

from __future__ import print_function, division

__copyright__="""Copyright (c) 2003-2017 by The University of Queensland
http://www.uq.edu.au
Primary Business: Queensland, Australia"""
__license__="""Licensed under the Apache License, version 2.0
http://www.apache.org/licenses/LICENSE-2.0"""
__url__="https://launchpad.net/escript-finley"

"""
basic tests for util.py

:remark: use see `test_util`

:var __author__: name of author
:var __copyright__: copyrights
:var __license__: licence agreement
:var __url__: url entry point on documentation
:var __version__: version
:var __date__: date of the version
"""

__author__="Lutz Gross, l.gross@uq.edu.au"

import esys.escriptcore.utestselect as unittest
import numpy
import os
from esys.escript import *
from esys import escript

try:
     ESCRIPT_WORKDIR=os.environ['ESCRIPT_WORKDIR']
except KeyError:
     ESCRIPT_WORKDIR='.'


class Test_util_values(unittest.TestCase):
    """
    Please do not define tests in this class.
    The idea is to make it easier to isolate groups of tests, which
    will be harder if there are tests in the base class
    
    Sets of values could be generated as follows:
    def z2(levels, row):
    if levels==1:
      v=[]
      for j in range(row):
          v.append(random.triangular(-1., 1.0))
      return tuple(v)
    else:
      v=[]
      for j in range(row):
          v.append(z2(levels-1,row))
      return tuple(v)
      
    z2(2,2)
    z2(3,3)
    z2(4,4)
    
    However, some of the larger arrays are modified afterwards to ensure
    that they contain at least one 0 for NaN checking purposes
    """

    RES_TOL=1.e-7 # RES_TOLerance to compare results
    DIFF_TOL=1.e-7 # RES_TOLerance to derivatices

    def makeTagged(self, fs, v1, v2):
        """
        Helper routine which makes a DataTagged containing the two values
        (v1 as default and v2 as tag 1).
        """
        d=Data(v1, fs)
        d.setTaggedValue(1, v2)
        return d
    
    def get_scalar_inputL(self, cplx):
        if not cplx:
            v1=33489.58300735649
        else:
            v1=(9824.950442887115-46215.48181304759j)
        return (v1,v1)

    def get_python_inputL(self, rank, cplx, no_first_arg_negative=False, no_zeros=False):
        if cplx:
            if rank==0:
                return (-3960.4642291825876+70893.2937559552j,)
            if rank==1:
                return ((59731.62346830329-82535.93558513945j), (97413.28611726397-71163.21858234056j))
            if rank==2:
                return (((22662.9306818+10450.1928396j), (-5876.588037489+31316.069231294925j)),
                        ((-48775.88574400305+21535.3746218j), (36479.6003269178-39287.306108j)))
            if rank==3:
                return ((((-50847.97960-9062.27j), (-63529.6088+55171.4663j), (215.4-10185.37j)),
                         ((40080.58380528418-34586.4289827135j), (-27276.009276950237-61855.16687539085j), (-34721.25780748475+82851.2788274762j)),
                         ((43010.5425+18456.0305j), (-8655.17674+4404.157j), (23658.850518231033-20372.09007648639j))),
                        (((43347.843141+58591.16j), (25836.3146-5995.971j), (-28969.5+31065.035j)), 
                         ((-31942.46+61141.334j), (-84655.794+49812.188j), (63001.6-44321.65j)), 
                         ((-19368.317+29114.8j), (-84938.09+83917.34j), (30356.3+48962.755j))), 
                        (((-50756.5-28925.102j), (3348.16+79354.6j), (30159.52+6881.75j)), 
                         ((59025.953-39.831j), (-237.942-53.93j), (9308.8+36247.003775959376j)), 
                         ((12068.153033533716-68009.47095735556j), (7693.602018284684-23713.22567007532j), (44993.89718529736-3807.1609666917066j))))
            if rank==4:   
                return (((((36743.82175242095+26778.737326529765j), (-29754.182129388544+25393.077634889414j), (-7835.209676340906+49819.26015705921j), (33964.789417447566-32778.45444590585j)),
                          ((-44579.370809357875-68703.24361188836j), (-149.28320431357133-64698.83715098185j), (38295.13314270778+58425.02740389345j), (-59088.20438563965-81586.99049191552j)),
                          ((22892.612062358283-24715.70868029048j), (13367.867311961876+20860.578306432217j), (-47859.40778589482+71392.8351064374j), (-2304.287041885458+49294.11306179231j)),
                          ((-62640.841478880255+36871.688232087705j), (10655.923749286507-23128.560897714342j), (-4131.801825297647+4443.2840182007785j), (-27903.670838787744-3117.482981970883j))),
            (((57034.38962108706+38044.26403584313j), (-4462.844355084104+18211.587917962242j), (4313.178312608696-12868.233657189892j), (46374.72869675713-11812.40480331378j)),
             ((322.189068738895+47179.67709271595j), (72322.13544947651-3662.5467369358375j), (-9231.467186395414+76230.40517770775j), (-41704.75682501451+32558.11893996234j)),
             ((79487.23328596324-7366.42862253316j), (10481.924314965669-15120.283604848068j), (26960.83136881994-41071.566704782505j), (-24115.39190283895-7711.336401151741j)),
             ((22474.95612418659+15623.165463453799j), (25929.087385707462+38175.58304877044j), (-8177.232611463696+21287.640323926986j), (34229.957465479616+34000.54107560321j))),
            (((26772.169008338446-23226.644157592586j), (44355.462946645705+38435.617150983846j), (82270.23681598579+57228.460214922394j), (-61649.67759149854-29496.646828970726j)),
             ((33208.12540969731-42543.11381431012j), (-37368.19634070844+36983.91078521421j), (-53479.67762639474+69192.57351520905j), (-16987.119998626833-24907.621827720854j)),
             ((58265.74226484515+21428.598358311836j), (-25451.773790957814+34641.52936103524j), (49737.44278424142+46668.70872917027j), (-19763.07777821306-55742.60412333406j)),
             ((39696.62704009203+11792.880723760507j), (45040.78931593459+1353.15524889082j), (38919.41458390256+5018.601358823813j), (24700.01773615746+18638.158415414437j))),
            (((-27357.697301503198+67993.98776577965j), (43358.65691551873-60981.94828910699j), (-36961.845280498295+73544.32798356535j), (-7005.460088561973+56718.32085849555j)),
             ((-45794.83454061549-24789.54243897248j), (-35553.771747041705+281.20812003471656j), (741.8214864204201-48926.27892674535j), (15258.748935604614+5750.3134710095765j)),
             ((13578.214948777008+49023.37779316875j), (-52997.87500887418-28291.632747669166j), (34523.01328904568+70373.31760549065j), (-8549.195292188699+38710.00933198735j)),
             ((-8724.746595711767+39671.57400059246j), (54220.76682697828+52787.45089736888j), (-2929.7928072663053-14645.947666259133j), (6967.3562262647465+20575.621108094565j)))), 
            ((((41255.3128726088-4735.07598620026j), (17330.015550087788+39190.403383224395j), (-2409.2939922655205+33755.22845153985j), (-84016.4244828014-28839.63999275923j)),
              ((-7567.298951515288+24481.103774662595j), (-25149.082523503632-8583.669927251554j), (47788.23750305794+37492.05579481191j), (-19575.167208193656+21113.148224316566j)),
              ((-8076.55304084401+75096.61990916729j), (42138.65038370111-27451.77036345769j), (17599.47893995483-6322.617589222951j), (43000.60101023788+54422.76834543564j)),
              ((-24961.480874041634-2122.019874702237j), (6425.889470030932+5622.557974648473j), (-65254.39721134174+59644.00375157643j), (-26617.634404672892-53495.819302484895j))),
            (((-54060.398876227686+55992.77686553825j), (-29817.626727209106-29704.79980981775j), (-6767.670138070098+10024.257522469488j), (-30133.80524111932+40328.89815642918j)),
             ((50000.78584499907-5894.192721755971j), (-3021.8049413580593-70247.81603731646j), (695.299152508218+58895.55560166443j), (-16680.2648353032-87022.82137596376j)),
             ((14249.594063942364-1274.074993916045j), (12234.307608613788-18357.139710977237j), (32290.96562094186+38793.36224883628j), (67656.21258170376-9760.573070193976j)),
             ((-45601.023706816115-74319.21757213926j), (55876.22452146437-61120.98256716965j), (58027.41230717529+10786.688485100865j), (-3495.6762613661413-46756.553429873304j))),
            (((92894.48804954227+23902.59394418483j), (1353.988548653011-92567.68243719656j), (65397.064463537616+10248.124624835778j), (567.8160836168245+78786.59184031615j)),
             ((-40282.49291521364+79825.32258981027j), (-40310.88688269715+455.4179428850184j), (-11166.391823961778+2764.918200738073j), (78697.6368921528-685.3372273494751j)),
             ((70892.98538412598+78843.94735644812j), (-64882.52692417733+80256.3932249126j), (26045.841803168907-12175.162753092736j), (-58159.5440761434-67887.15729778618j)),
             ((8775.14598500292+16795.25423101084j), (13155.934674011907+16737.126752117358j), (15975.852541323125+14513.776915618466j), (-25215.109458495746+8530.276446405784j))),
            (((25795.767870170166-31056.851297134257j), (4037.894742402088-83.2035600123636j), (-19413.455650262404+40693.39115961014j), (-63145.75424957367+34735.00168222326j)),
             ((17181.9690394505+18815.796074907106j), (45370.782381827026+55332.61058358025j), (-15112.046450780603-88000.14177105221j), (81609.38175982882+21862.010369853655j)),
             ((-55560.45495276697-8487.570486260593j), (-1240.3046453718998-36806.93746782728j), (2314.7751094353007+66709.2996375249j), (73913.798692593+17715.330241595133j)),
             ((-6756.64963479065-17680.06798381319j), (-68433.87475450675-1521.61274877697j), (-38429.70821819296-59684.70774465629j), (679.6723316871648-5141.593014362268j)))),
            ((((6009.97586310716+60604.30942478338j), (32396.109723949165-43790.651419414105j), (28779.550340574773+51904.28569017858j), (22898.47954312162+24374.281930383077j)),
              ((43337.67939273335-5975.004215923647j), (-20668.774830910013-76079.20355770885j), (28645.123677151292+25672.909247014104j), (-31717.682104715685+9732.221717787586j)),
              ((53661.13842896042+72923.84155787138j), (35385.072072520015-6307.198139944798j), (30479.544507534592+62234.1254985712j), (-1034.7905621764367+40286.709890159545j)),
              ((-81367.55312016135+16928.505901155673j), (-7815.888755622378+6045.680730129257j), (23493.410721616994+56525.86301109548j), (64379.29795689413-41596.85817228346j))),
            (((10077.706189869234-14023.344225890105j), (9787.45494885469+1980.526605239269j), (-41789.42734084199+33109.62355852481j), (77722.47714091674+16230.914008263571j)),
             ((33929.99686388744+69324.24249861641j), (-42944.721176775936+77239.49498876408j), (-24531.218551429585+33511.72101875048j), (38167.50870231599+48640.295148756864j)),
             ((5153.059156082629+82411.43319343394j), (46237.63154724835-24482.044081531683j), (-42350.512586094905+21364.573202395462j), (27884.705448024964+27406.374083228744j)),
             ((26735.822927676723-16229.136581069019j), (-20377.70277018177-48828.24171348816j), (-31499.21640336576-14810.9243228345j), (-7904.957576446148-20229.149437926535j))),
            (((16522.705470639165+17310.831386237332j), (-76148.34984139015+80969.93773509384j), (-24811.00749069672+38047.29320608019j), (20992.172136102876-22605.533994627098j)),
             ((-20970.94805191878-11158.194764479718j), (-24776.365668172788+15335.11508002579j), (31475.965557547286+42101.17185492054j), (-5067.057332858472-32895.81793448307j)),
             ((2682.2679111504112+39461.2760464704j), (8347.197026209295+5913.884037792421j), (-33487.63388762134-63339.09288353257j), (-57157.881248657024+1556.61791639906j)),
             ((-41068.04079080766+48584.07702236999j), (8498.148930898213+14312.48564445498j), (-21316.05107180962+87054.88334056246j), (-21471.236761054563-59830.259831579846j))),
            (((29431.72773241329-33178.12936911729j), (26409.71652118041+55871.26981512072j), (66462.49459072504+84224.15533504078j), (54448.348792259174+24319.32902355393j)),
             ((-71328.38862684023+23367.89957869296j), (-59401.71170696309+49485.585597715195j), (41883.2210796735+19719.46171899562j), (-18795.333157237168-68982.70021560306j)),
             ((59112.62708481629-45457.26410539314j), (-88790.25871362776-25685.393578202435j), (23696.552787236098+41921.596338817966j), (55010.09931039718-93443.28701025742j)),
             ((39099.27462555075-21859.303492325664j), (-13175.71402014888+48223.91127425416j), (-5568.289447947842-299.0850724438933j), (-20467.28086020409+30095.8347308894j)))),
            ((((-16018.868474730582-47768.31986595277j), (51712.37041771119-18503.457760090154j), (-49051.6111606494+23687.829560421436j), (-71984.9830420537+396.65467472762975j)),
              ((10108.4516848598+51857.057322104636j), (-7216.241612916012+72110.85192662141j), (54626.45574269543-59774.72883503144j), (-657.6522009949113+60897.80850184188j)),
              ((-7989.0171800768585-89431.37235259652j), (-24705.702245950946+70622.38482297733j), (-63459.904012077815-42658.85544672633j), (50226.10296539047+34122.54286740055j)),
              ((41218.241998490725+14373.480195603668j), (-54370.32919269195+6808.202490824406j), (-1527.4828551954124+54636.12386175628j), (14472.837501920381-52333.2803360176j))),
            (((-2746.8539660817187-74513.82700384625j), (-61015.83534421826+53680.73473922199j), (-79322.89757472202-30783.74464730559j), (-46747.1238366321-74773.91885253192j)),
             ((38500.41669416314-55405.667973400086j), (1396.4878373664105+24775.37482803031j), (58020.02299122539+48560.59423440633j), (-29555.466456927898-2523.0682893230987j)),
             ((-34617.908388179705-12964.332502930381j), (-45031.80883921236+65602.1489634705j), (-25153.44844444198-12070.474688363538j), (-21543.782061126898-86119.31236007596j)),
             ((71651.55622950022+37553.55829666175j), (73215.11980883317-718.055454765301j), (7065.864625805567-92990.91593148846j), (11751.07053047308-7707.306990859142j))),
            (((-43076.79837010652+24177.23355451846j), (6793.467392677732-20449.573606641366j), (-71013.01101746038-6225.881152518472j), (74055.17870813374+66606.15686348274j)),
             ((11557.138464109157+5388.5729086161155j), (36732.06100793248-37022.585715349574j), (2921.611354480643+55496.19455947708j), (55058.90849766818+18274.262129668554j)),
             ((-58815.802612553605-3016.8034868425602j), (11211.939776371626+85380.70520152281j), (5184.476190878297-46247.039507761j), (66267.98054810712-61671.71978896645j)),
             ((-5378.665820848371+9328.328146237982j), (43895.1264174895-68770.39938502887j), (-17511.997277418195+58343.13845945775j), (-85449.25399769237+5421.878678659821j))),
            (((39759.129208538325-36375.49825029998j), (14116.074449609747-3690.404237582101j), (726.610813691339+20601.83416877652j), (-39703.3656455713+3717.592156691331j)),
             ((59401.5216214189+24249.24110262998j), (32483.925202512517+12478.691995073255j), (-32680.63271733308-27669.460617493503j), (-49228.06429579756+18669.428245068033j)),
             ((44330.3653675318-76.05447174097935j), (-77173.70951748395-14072.79466329544j), (44365.91092729417+30775.448092456412j), (-27126.74998375852-42673.613992010614j)),
             ((69065.27619440478+6707.326801963689j), (30689.082128804977-86800.51053515819j), (49475.52450098609+20399.323850453336j), (-29667.120261194417+8908.992755905914j)))))
        else:
            if rank==0:
                return (1968.4370221151426,)
            if rank==1:
                if no_first_arg_negative:
                    return (24899.54037082102, 25.31549006801)
                else:
                    return (-24899.54037082102, -25.31549006801)
            if rank==2:
                return ((1830.692020877672, 4256.807988523855), (31330.447580824955, 30088.386511638193))
            if rank==3:
                if no_first_arg_negative:
                    return (((22464.14939770715, 9326.07497133658, 58843.28039835674),
                         (39132.51601559524, 28984.021248456513, 87606.99636934511),
                         (82483.31416277925, 58484.471708777906, 28659.10377643413)),
                        ((48055.05693146407, 36954.49346593733, 25658.428739633804),
                         (45374.680239837886, 11967.766802782062, 40707.17918100887),
                         (9272.611657965084, 95545.34560025277, 36552.65523613798)),
                        ((91272.06575583504, 61258.36210240262, 23700.008344466856),
                         (76559.99183386622, 74506.2742834793, 40042.39145376187),
                         (54893.50655552275, 60408.00584363263, 68208.21135366432)))
                else:
                    return (((22464.14939770715, 9326.07497133658, -58843.28039835674),
                         (-39132.51601559524, -28984.021248456513, 87606.99636934511),
                         (82483.31416277925, 58484.471708777906, 28659.10377643413)),
                        ((-48055.05693146407, 36954.49346593733, 25658.428739633804),
                         (-45374.680239837886, -11967.766802782062, 40707.17918100887),
                         (-9272.611657965084, 95545.34560025277, 36552.65523613798)),
                        ((-91272.06575583504, 61258.36210240262, 23700.008344466856),
                         (-76559.99183386622, 74506.2742834793, -40042.39145376187),
                         (54893.50655552275, -60408.00584363263, -68208.21135366432)))
            if rank==4:
                if no_first_arg_negative:
                    return ((((2613.5684369694354, 384.9724701708765, 17359.719572324335, 63393.27571171061), 
                          (41041.42067423142, 14993.16104673734, 27399.273319594344, 60051.2377656026),
                          (18566.32422588553, 73825.83739655794, 95319.0187027347, 27646.03157540709),
                          (88007.8178172775, 60759.47561504975, 31822.15287970184, 38852.10105560179)), 
                         ((89386.29724076603, 79511.70453673883, 1325.344653112319, 14210.631402891566), 
                          (9099.55295712997, 30612.395205911307, 5512.258312859951, 43371.59702594321), 
                          (24961.325742358764, 6843.62962861413, 43588.35638871549, 37139.61080792606), 
                          (59901.00300911518, 59802.81275754453, 26119.701955513825, 70973.81339014533)), 
                         ((10824.815145545523, 37682.56858486531, 14732.374901197414, 14286.141738580205), 
                          (43004.59056324265, 59441.985441533616, 83687.96499450768, 4325.869356595635),
                          (56783.42265829212, 28959.61168144866, 83740.97480554663, 10729.510064491638),
                          (38480.61835393819, 50636.84578373406, 3588.0738023085723, 56239.74544740876)), 
                         ((18228.900850710852, 16632.129655349723, 8015.163471499967, 4521.799739700364), 
                          (51505.380752985875, 1524.7016267364233, 19342.421346891773, 48490.57412835015), 
                          (73047.79149393305, 35239.98425352734, 35458.1734620065, 55659.41536985519), 
                          (13074.184715655007, 6436.319756047014, 28485.17418228225, 11128.300314215841))), 
                        (((321.6371242800669, 35572.05942103493, 9515.23626081155, 4433.077669837556), 
                          (26761.084128710136, 10867.52013173081, 19183.162620251198, 68559.41235027292), 
                          (11484.667574519088, 51828.050682516754, 38005.19565254123, 76175.23771370867), 
                          (5227.790707720851, 71440.84429737684, 50336.09378326664, 37678.70778115371)), 
                         ((45566.74581644684, 5908.975208284799, 22576.958775978783, 50020.73251106653), 
                          (26017.058082686533, 25299.281527364627, 40976.696154446676, 13431.16881318092), 
                          (46649.16989082289, 65794.20117285698, 64119.987425361716, 5497.440022742972), 
                          (2203.2985446707025, 57513.79981817989, 41574.749087273776, 48387.40101623114)), 
                         ((18416.741032524515, 47640.69430896508, 19679.029797917596, 34810.30464977612), 
                          (62831.92844082225, 27725.457434774376, 3134.438446956716, 49315.595365086476), 
                          (73623.94099730175, 20978.640264794405, 5573.772319421056, 36086.56272512534), 
                          (28660.57533284233, 58241.55583429358, 51661.45261125069, 50693.33103010448)), 
                         ((56426.23453986164, 78359.73594286482, 38576.297999959155, 8913.910887926046), 
                          (88189.29548286308, 2602.1208204264694, 15718.494307329529, 39115.51872697051), 
                          (50134.74938380022, 42828.424427019796, 39528.932684095016, 71074.105860494), 
                          (10360.78422517216, 3628.4946464838577, 866.6535588603292, 31553.01858562138))), 
                        (((52390.293064785874, 12243.173190398476, 12973.542140497811, 24679.210914480092), 
                          (39068.25465569209, 11375.555598436607, 11948.963084895935, 23180.6696549638), 
                          (7750.742802840003, 4544.655806011273, 17257.229884485423, 38718.29755209083), 
                          (54962.39017442457, 32692.842165301, 79228.7545707964, 46958.08883925094)), 
                         ((58808.04021951867, 31679.09607989699, 60465.22396617634, 3778.4799478359637), 
                          (17712.329884018807, 17874.74312645469, 14954.738052270157, 44753.01307496137), 
                          (24417.93508883397, 25931.24652138863, 27186.302587945494, 37605.82669369904), 
                          (20116.370032880397, 22395.271847421376, 39918.504300252374, 43365.55907616498)), 
                         ((29680.436323581554, 7156.784950690315, 27141.447127045933, 65196.42918411147), 
                          (10851.323062587486, 12918.348059911383, 61892.308397834146, 49367.20478221224), 
                          (12048.985149625369, 61619.01673500164, 19594.218775417554, 52093.668037218595), 
                          (38465.24637072596, 9188.19710817252, 7828.463839086282, 62406.93767988804)), 
                         ((35787.91825329978, 23117.81737012358, 49967.17854684144, 9030.162340133553), 
                          (31476.492123520133, 42646.47087585365, 80893.4161545868, 38123.68731593444), 
                          (23794.505773755358, 1648.5505688887642, 18076.67754352292, 50552.198666715994), 
                          (31218.595116869503, 36051.76246348336, 23827.05367940539, 6983.787679924746))), 
                        (((22717.871470141312, 41000.10860057993, 57170.77347009425, 45870.224065572766), 
                          (75749.5782286584, 82016.05469083885, 15364.324776823589, 34421.772637849994), 
                          (56091.35037369725, 69020.64428824343, 54982.82912975482, 26461.710610700044), 
                          (6865.449808598962, 1458.33269652327, 19976.42644646119, 71857.51352483299)), 
                         ((19816.540187252496, 8015.999325965924, 73384.47295519276, 29811.76619356322), 
                          (13116.527286368713, 76148.43320182734, 694.6129571919591, 51487.98727764331), 
                          (18216.321286892955, 19089.44846330423, 49851.57024753545, 23061.547216296094), 
                          (4069.367521493099, 4460.02510964268, 70628.47512058845, 5320.222269188191)), 
                         ((55551.601997272795, 15131.970379855833, 9132.149357120768, 1748.749782567771), 
                          (25571.39049857318, 22269.3336119335, 40013.90106400979, 41645.300687720875), 
                          (19942.423484201456, 77721.3403094917, 49049.37211887518, 42125.028417728725), 
                          (20331.591530993654, 18849.729202127564, 11172.361078239031, 42378.721186597744)), 
                         ((28162.219784186906, 27005.38411136302, 28047.626970074954, 48907.05914121462), 
                          (21561.00249540838, 16819.379648924267, 21373.309752767862, 81217.52472263864), 
                          (14530.50392089595, 38878.60235910026, 18507.98920090188, 38187.90762089445),
                          (24234.820876412356, 50390.96325142998, 5976.493423496679, 16545.42618485792))))                    
                else:
                    return ((((-2613.5684369694354, 384.9724701708765, 17359.719572324335, -63393.27571171061), 
                          (41041.42067423142, -14993.16104673734, -27399.273319594344, 60051.2377656026),
                          (18566.32422588553, -73825.83739655794, 95319.0187027347, -27646.03157540709),
                          (-88007.8178172775, 60759.47561504975, 31822.15287970184, -38852.10105560179)), 
                         ((-89386.29724076603, 79511.70453673883, 1325.344653112319, 14210.631402891566), 
                          (9099.55295712997, 30612.395205911307, -5512.258312859951, -43371.59702594321), 
                          (24961.325742358764, 6843.62962861413, -43588.35638871549, 37139.61080792606), 
                          (-59901.00300911518, -59802.81275754453, -26119.701955513825, 70973.81339014533)), 
                         ((10824.815145545523, -37682.56858486531, -14732.374901197414, -14286.141738580205), 
                          (-43004.59056324265, 59441.985441533616, -83687.96499450768, 4325.869356595635),
                          (-56783.42265829212, -28959.61168144866, -83740.97480554663, -10729.510064491638),
                          (-38480.61835393819, -50636.84578373406, -3588.0738023085723, 56239.74544740876)), 
                         ((18228.900850710852, -16632.129655349723, 8015.163471499967, -4521.799739700364), 
                          (-51505.380752985875, -1524.7016267364233, -19342.421346891773, -48490.57412835015), 
                          (-73047.79149393305, -35239.98425352734, -35458.1734620065, 55659.41536985519), 
                          (-13074.184715655007, 6436.319756047014, -28485.17418228225, -11128.300314215841))), 
                        (((321.6371242800669, 35572.05942103493, 9515.23626081155, -4433.077669837556), 
                          (26761.084128710136, 10867.52013173081, -19183.162620251198, -68559.41235027292), 
                          (11484.667574519088, 51828.050682516754, 38005.19565254123, -76175.23771370867), 
                          (-5227.790707720851, 71440.84429737684, -50336.09378326664, 37678.70778115371)), 
                         ((-45566.74581644684, 5908.975208284799, 22576.958775978783, -50020.73251106653), 
                          (-26017.058082686533, -25299.281527364627, 40976.696154446676, -13431.16881318092), 
                          (-46649.16989082289, -65794.20117285698, 64119.987425361716, 5497.440022742972), 
                          (-2203.2985446707025, -57513.79981817989, 41574.749087273776, 48387.40101623114)), 
                         ((-18416.741032524515, -47640.69430896508, 19679.029797917596, -34810.30464977612), 
                          (62831.92844082225, 27725.457434774376, -3134.438446956716, 49315.595365086476), 
                          (-73623.94099730175, 20978.640264794405, -5573.772319421056, 36086.56272512534), 
                          (28660.57533284233, -58241.55583429358, -51661.45261125069, -50693.33103010448)), 
                         ((56426.23453986164, 78359.73594286482, -38576.297999959155, 8913.910887926046), 
                          (-88189.29548286308, 2602.1208204264694, -15718.494307329529, 39115.51872697051), 
                          (50134.74938380022, -42828.424427019796, 39528.932684095016, 71074.105860494), 
                          (10360.78422517216, 3628.4946464838577, 866.6535588603292, 31553.01858562138))), 
                        (((52390.293064785874, -12243.173190398476, 12973.542140497811, 24679.210914480092), 
                          (-39068.25465569209, 11375.555598436607, -11948.963084895935, 23180.6696549638), 
                          (7750.742802840003, -4544.655806011273, 17257.229884485423, -38718.29755209083), 
                          (-54962.39017442457, -32692.842165301, 79228.7545707964, -46958.08883925094)), 
                         ((58808.04021951867, 31679.09607989699, 60465.22396617634, 3778.4799478359637), 
                          (17712.329884018807, 17874.74312645469, 14954.738052270157, -44753.01307496137), 
                          (24417.93508883397, -25931.24652138863, -27186.302587945494, 37605.82669369904), 
                          (-20116.370032880397, 22395.271847421376, 39918.504300252374, 43365.55907616498)), 
                         ((-29680.436323581554, -7156.784950690315, -27141.447127045933, 65196.42918411147), 
                          (-10851.323062587486, -12918.348059911383, -61892.308397834146, -49367.20478221224), 
                          (-12048.985149625369, 61619.01673500164, 19594.218775417554, 52093.668037218595), 
                          (-38465.24637072596, -9188.19710817252, -7828.463839086282, -62406.93767988804)), 
                         ((-35787.91825329978, 23117.81737012358, -49967.17854684144, -9030.162340133553), 
                          (-31476.492123520133, -42646.47087585365, -80893.4161545868, -38123.68731593444), 
                          (23794.505773755358, -1648.5505688887642, -18076.67754352292, -50552.198666715994), 
                          (31218.595116869503, 36051.76246348336, -23827.05367940539, 6983.787679924746))), 
                        (((-22717.871470141312, 41000.10860057993, 57170.77347009425, -45870.224065572766), 
                          (-75749.5782286584, 82016.05469083885, 15364.324776823589, -34421.772637849994), 
                          (56091.35037369725, 69020.64428824343, -54982.82912975482, 26461.710610700044), 
                          (-6865.449808598962, 1458.33269652327, 19976.42644646119, 71857.51352483299)), 
                         ((-19816.540187252496, 8015.999325965924, 73384.47295519276, -29811.76619356322), 
                          (-13116.527286368713, -76148.43320182734, 694.6129571919591, -51487.98727764331), 
                          (-18216.321286892955, 19089.44846330423, -49851.57024753545, -23061.547216296094), 
                          (4069.367521493099, 4460.02510964268, -70628.47512058845, -5320.222269188191)), 
                         ((55551.601997272795, 15131.970379855833, 9132.149357120768, -1748.749782567771), 
                          (25571.39049857318, -22269.3336119335, -40013.90106400979, -41645.300687720875), 
                          (19942.423484201456, -77721.3403094917, -49049.37211887518, -42125.028417728725), 
                          (-20331.591530993654, 18849.729202127564, -11172.361078239031, -42378.721186597744)), 
                         ((-28162.219784186906, 27005.38411136302, 28047.626970074954, -48907.05914121462), 
                          (21561.00249540838, 16819.379648924267, 21373.309752767862, 81217.52472263864), 
                          (14530.50392089595, -38878.60235910026, 18507.98920090188, 38187.90762089445),
                          (24234.820876412356, 50390.96325142998, 5976.493423496679, -16545.42618485792))))
        
    def get_scalar_input1(self, cplx, no_first_arg_negative=False, no_zeros=False):
        if not cplx:
            v1=0.6081355033581688
        else:
            v1=(0.27562178672620385+0.42446484939332796j)
        return (v1,v1)

    def get_scalar_input2(self, cplx, no_zeros=False):   # optional arg doesn't matter here
        if not cplx:
            v2=0.5327268994219874
        else:
            v2=(0.49976275640416334+0.4141087840210876j)
        return (v2,v2)
    
    def get_scalar_input3(self, cplx):
        if not cplx:
            v3=0.4903827487969067
        else:
            v3=(0.4130410618402195+0.7195601418761082j)
        return (v3,v3)
    
    def get_scalar_input4(self, cplx):
        if not cplx:
            v3=-0.4555987934301202
        else:
            v3=(-0.2562657772006671-0.09989064526858904j)
        return (v3,v3)

    def get_python_input1(self, rank, cplx, no_first_arg_negative=False, no_zeros=False):
        if cplx:
            if rank==0:
                return (-0.8493106430566671+0.5498588650604705j,)
            elif rank==1:
                return (0.5213538174366878+0.45097055421243537j, 0.25896216831911634-0.5966874048733529j)
            elif rank==2:
                return (((-0.20984377987310687+0.024810300993445944j), (0.024805532005481523+0.015098314700385806j)),
                        ((0.2292639954026584-0.7100972341857507j), (0.08682454615283863+0.12332238221542291j)))
            elif rank==3:
                return ((((-0.6550500854131598-0.2127087989155373j), (-0.017627543683440106-0.3665841375380333j), (0.12492952051741912+0.05374046981620606j)),
                         ((-0.11197479851035519-0.5608269924316476j), (-0.5419470859968003+0.07356518546227997j), (-0.42635110929148823+0.47738396333288513j)),
                         ((-0.16556333483163588+0.009959138144123747j), (-0.17095759700320223+0.19563542216810015j), (0.38543550056193654-0.7475542165987907j))),
                        (((-0.773039591137402+0.001304882415888442j), (0.005692414753365238+0.7073228171031704j), (0.16149889599978962-0.5779399189029897j)),
                         ((-0.39240472327059983+0.5661946692113213j), (0.15619153376084327-0.2043397442335695j), (0.12989734566943012-0.6797045089189742j)),
                         ((-0.2953545934396431+0.00897241164565632j), (0.2424492369208724-0.3012099081809274j), (-0.5052331641775911-0.7560269073586534j))),
                        (((-0.4330063069118393+0.4662418513842146j), (0.14640515390237896+0.21280275920042335j), (-0.6848374766215319+0.688509731931129j)),
                         ((-0.7009542857194253+0.5090290863675988j), (-0.08777137326709294-0.0642088074588476j), (-0.22469168662839534-0.20497222057853204j)),
                         ((-0.004895060324627054-0.0007149461427655712j), (0.5733987472263198-0.020876347517763794j), (0.14677839835678597-0.33748939337229644j))))
            elif rank==4:
                return (((((0.790344350-0.675690862j), (-0.33969014424346+0.004667354999j), (-0.353898037-0.23103247j), (-0.173453149+0.350472j)),
                          ((0.262318036+0.493920836j), (0.122221063-0.2126j), (-0.2002867+0.62847794j), (0.2585201923791-0.1055537993243j)),
                          ((0.1024287918-0.07259437049j), (0.3338564+0.44847000j), (0.0433384-0.2616078j), (-0.1717597+0.2503709670077j)),
                          ((-0.768778331-0.076403010j), (0.251991719+0.62985777j), (0.4514933+0.47696252j), (-0.0351486+0.023037350246j))),
                         (((-0.4863645-0.24921801802j), (-0.023444472-0.5599386j), (-0.56592+0.4275706j), (0.41222370286-0.34867238925j)),
                          ((-0.5252590+0.04977251205j), (-0.33737953+0.27293775j), (0.66428+0.87048705j), (0.0672260574-0.416242856449j)),
                          ((0.148783460-0.4520517585j), (-0.41537536+0.16140391j), (0.6278127+0.263838j), (-0.07760387-0.1831155602152j)),
                          ((0.0470026875+0.481795665j), (0.0215979296-0.081945318j), (0.7913869-0.0861j), (0.39466-0.36695690929153635j))),
                         (((-0.143371038+0.36692459j), (-0.05936024660+0.57744203j), (0.35210125-0.3062606j), (0.09250628940+0.2787772j)),
                          ((0.5220814-0.12741648j), (-0.054402209613+0.8726141525j), (0.19115503+0.059924j), (0.34340959513-0.13436584j)),
                          ((-0.854710-0.3580477j), (-0.45614757+0.0726810109j), (-0.11809521670+0.15075834042j), (-0.028270996-0.13039829j)),
                          ((-0.4935554755+0.6616606j), (-0.77712599-0.292285j), (-0.29938802+0.958714551j), (0.1306040901396+0.5467211724j))), 
                         (((-0.490711642822499-0.0679784225903799j), (-0.028147241164195536-0.03995809530085381j), (-0.28811330815221536-0.6815443637070895j), (0.07421744086779403+0.7402073156785812j)),
                          ((-0.20556698745139912+0.15621646747170792j), (0.09192542596753261+0.12253579257228409j), (-0.3264483145179087-0.946630282004219j), (0.224192041112648-0.90380956121387j)),
                          ((-0.23224656797256493-0.6768140280012414j), (0.2451768408209073+0.17488492564153224j), (-0.6851697787753819-0.08406614036650228j), (0.20088414272093424+0.1669900873662492j)),
                          ((-0.36057308512412345+0.4226878583624143j), (0.4490488492756204+0.11240390126422484j), (0.03234540063577107+0.5512325765639803j), (-0.5756238638317961-0.46042879607009257j)))), 
                        ((((0.24124399378525585-0.8745635773731907j), (-0.19374525735285986+0.5380787997061398j), (-0.4489085299026162-0.6296073456803736j), (0.056570144039872594-0.135671341894538j)),
                          ((0.05530486113034205-0.45284053426797866j), (-0.04502323738051783+0.0630304737860885j), (0.004887893978963609+0.5713368827301133j), (0.18732743423207876-0.7143408881411377j)),
                          ((-0.3712234322764927+0.24908466763268566j), (0.6364022966205971-0.21808571418727385j), (-0.5670900710572193+0.6348611392807915j), (0.9245683140013828+0.3899090325528184j)),
                          ((-0.6094870272364956-0.2685532809269351j), (0.6287315177811945+0.4264823533706973j), (0.1676715780106749-0.48029733292855326j), (-0.5376630517471747-0.7182359990772386j))), 
                         (((-0.018779747197580465+0.6114308512727638j), (0.3732084851158033-0.4786089418474003j), (-0.5663664323873074+0.39392462282467033j), (0.13538143997466878-0.12985823792228446j)),
                          ((-0.4780490757496574-0.39805216916921526j), (-0.2797090849741719+0.93720217986173j), (0.6203622207514078-0.15823957110011522j), (0.6060182605777774+0.018214531480017393j)),
                          ((-0.3696664461011986+0.07708468475820063j), (0.3066672994071602-0.345863819817513j), (-0.18002308351634988+0.6758164917135244j), (-0.3299795335012433+0.3472073085359665j)),
                          ((-0.018635553181191034-0.2527645838168162j), (0.10050559250760827-0.2682102608362037j), (0.15675656210201427-0.8704771160865176j), (0.2060240187990353+0.1525158024705776j))),
                         (((0.17233488629737193-0.11594210469295985j), (0.06281516668626441-0.2603634509290429j), (-0.05642743012335105-0.655594777161181j), (0.6959785793264961+0.19699171017512973j)),
                          ((-0.17348338509217032-0.6986792371372937j), (-0.8346643961801311-0.3680194342368177j), (0.12864312443903647-0.32156359553027614j), (0.6861567331359726-0.010430013106866087j)),
                          ((-0.6222917331694062+0.112713903157314j), (-0.12621654439609742+0.28520452943867913j), (0.44508284456518865-0.7522188187153122j), (-0.2636591190438452-0.040926078833506185j)),
                          ((-0.19805372118683406+0.5858778783604968j), (0.8721978764710994-0.548302441850661j), (-0.15430327558703272-0.01201209901149869j), (-0.3670274756811238+0.5474051336552919j))),
                         (((0.25034760722181615-0.15703084151927404j), (-0.14637027923589074+0.2928912738619156j), (0.46416445476921286+0.08042093992929411j), (-0.506840204775922-0.1186965040818393j)),
                          ((-0.37679950234230697-0.2701183493158177j), (0.043782673009088335+0.06734842807447794j), (0.2517828293729589-0.46888255931621403j), (0.6737055518213005-0.812365808946925j)),
                          ((-0.2661639821601073+0.13557351739334367j), (0.7392306869051049+0.5132140781139445j), (0.8364137600747349+0.15412542220542547j), (-0.5220472284167044-0.3618612888730691j)),
                          ((0.13970939326026943-0.3036727447660489j), (-0.38759251152342933+0.18775991991684293j), (-0.2487083238049208-0.4778652656270306j), (0.1390512499757498-0.9134653209907141j)))), 
                        ((((0.04833008268979544+0.15219347436054742j), (0.26103655160455086-0.3757461116008425j), (-0.12150273403936784-0.617683120749172j), (0.06086845258368434+0.5631830804862105j)),
                          ((0.5300453373682147+0.4166351114705461j), (-0.6851243348127105+0.6918703698362535j), (-0.7710255671005614+0.10134603710624956j), (-0.26008427165733317+0.39542364464404867j)),
                          ((0.5854498106749348+0.549731062650827j), (0.2519939415821417-0.43527531634423966j), (-0.6267811606157699+0.8585285100484663j), (0.07940474123690222-0.5124720092697812j)),
                          ((0.30776303313314646-0.15530657580599816j), (0.007522269998118669+0.35668611429959896j), (-0.1919814476250188-0.30177638543157503j), (-0.1646298733419591-0.7159546806725978j))), 
                         (((-0.30218788472478486+0.7760123719336528j), (-0.6314508280105797-0.1304090779644247j), (0.47461359141937576+0.4479088939414325j), (-0.06334668969380608+0.1323481666947287j)),
                          ((-0.7546949847971307+0.1756014222789699j), (0.25493728708952623-0.1282474899023487j), (0.390011965026524+0.4028431671750171j), (0.06598509493063165+0.7155111116236064j)),
                          ((0.07881709314826968-0.24405932002231845j), (0.36802109324120846-0.15749901166866498j), (-0.4450424875753707+0.7030178139999255j), (0.27329963141708635-0.7119309406610871j)),
                          ((-0.30933381776423374-0.11542316873853986j), (-0.0254248179097315-0.7313241377455038j), (0.10700540112141-0.4660169963118146j), (-0.3013644469437623+0.17540339691654783j))), 
                         (((0.0008073119623276392-0.05721575120619815j), (-0.08090870209548084-0.5957005266957907j), (0.17886979689341065+0.08798138279240286j), (-0.47326181867795214-0.39061153265097337j)),
                          ((-0.6240780356387126+0.931178282425864j), (0.49774084728858903-0.6554241240698137j), (-0.5514724615982707-0.1357900974308276j), (0.9476110679732384-0.8254384162767912j)),
                          ((0.339181898243627-0.26749719725842025j), (-0.8452885480367868+0.6880217962554664j), (0.6160319781793868-0.21382367328363683j), (-0.10773724739140955+0.12900692439104278j)), 
                          ((-0.4524239681624962-0.05106064389660214j), (0.3502260661172907-0.28486272434416593j), (-0.22611152371676513-0.4473111544777272j), (-0.07889011849494199-0.902189430409641j))), 
                         (((-0.4549255061524088+0.8204137745689758j), (0.6262794891433104-0.07873980006902381j), (-0.2099285569580429-0.5700098959028401j), (0.6485809522680817+0.09567054692828636j)),
                          ((0.20594301199461984+0.4902067661467834j), (0.3329644191869927-0.02188884849814343j), (-0.22921000231707434+0.3393725843732417j), (0.5813376839449342+0.09312340842380451j)),
                          ((-0.44184373889384854-0.22572260324605908j), (-0.2865930601221123+0.2963815667447778j), (0.23401187102772336-0.14690195925951866j), (0.5821368881041176+0.7899842539077798j)),
                          ((0.21512923589171906-0.5017618261731809j), (-0.29710734954707296+0.3058974064324922j), (-0.3522028959822211-0.5211544989239631j), (0.0833888296839771+0.47110470479799904j)))), 
                        ((((-0.12192871299046759-0.7328491805867747j), (0.7012790193629233+0.7216675552235499j), (0.07746694166806012+0.07187177179945092j), (-0.11596083131875812-0.3402549730235054j)),
                          ((0.4419525909212654-0.41264178907481297j), (-0.9540280899180219-0.0931626138838092j), (0.5850822884122122+0.023613319021177137j), (0.6204747822108139+0.4817530149691007j)),
                          ((0.12857240889181254+0.26765277556782874j), (-0.5905959060433874+0.30391448875145577j), (-0.33009606966351646-0.5133049903372606j), (0.3911371567574641-0.5699531083068055j)),
                          ((-0.06951047741648464+0.09185412570082496j), (0.23412720494006845-0.15301303449597714j), (-0.3632868271203742-0.0854806981764501j), (-0.06898965841232252-0.08895516025337646j))),
                         (((-0.30228646550028415+0.06630530646305488j), (0.1386457118575627-0.48419676726390337j), (-0.22729071025997294-0.7643686926101503j), (-0.3622425543437594+0.8590754571587876j)), 
                          ((-0.22721539776726385-0.36036034464849565j), (0.31497309604817325+0.3597772211828737j), (0.5069787174234794-0.8219928876404512j), (0.11194139062149588+0.26293050464964796j)),
                          ((0.4612090001013647+0.27388506472486907j), (-0.30194776224929665-0.13263761985638922j), (-0.6233657091294399+0.44684364446587943j), (-0.3450340301594258+0.11542147033108141j)),
                          ((-0.014881287063154947-0.18928627620951555j), (-0.015185483586684478+0.8972647988323046j), (-0.4351552758945989-0.7299460860379636j), (-0.12326133732628419+0.012329124057677388j))), 
                         (((0.10476345838581014-0.14636495344507483j), (-0.32803930125358216+0.27809153512314866j), (0.22528172636268207+0.3494691827944507j), (-0.5310240602410343+0.04119281216138804j)),
                          ((-0.5533642323509493+0.2906558493993484j), (-0.33436396580317507+0.4888660226122262j), (-0.4290629671820184+0.09348041559022413j), (-0.1635189281331263+0.08219415853032563j)),
                          ((-0.48564277336559514-0.39389189119028456j), (-0.07859439546331881-0.5417512106196164j), (-0.2634745687747664+0.6393823484701674j), (-0.4717227533107514+0.7886365869114986j)),
                          ((0.1574555874287038-0.15735823857163656j), (-0.3407071004516933-0.00848408628817443j), (0.111461150743281-0.4496138761616306j), (-0.08146922125654954+0.5259628325969007j))), 
                         (((0.34656176542518824+0.07880355128877725j), (-0.258503112094004+0.23322070678863926j), (-0.6883175179734812+0.06684313489266447j), (-0.04692875801791596+0.7384728581195434j)),
                          ((-0.3413224116842536-0.07621317699837815j), (0.23082091132507632-0.7397247652890888j), (-0.7325151537847703-0.2399819049396238j), (-0.13092551439092603-0.2541771655857984j)),
                          ((-0.6621690601235584-0.3959496844857878j), (-0.18349053313548802-0.8753348282719295j), (-0.03908532595157577+0.35883701301557147j), (-0.051099789459096545+0.22309220396282636j)), 
                          ((0.255858259117678+0.5962976405338435j), (-0.4167005907396837-0.7228120687439701j), (-0.4251792094268322-0.5598290555324366j), (-0.02547169493326129-0.10035815157656547j)))))          
        else:
            if rank==0:
                return (0.16343477325759478,)
            elif rank==1:
                if no_first_arg_negative:
                    return (0.414447461350212, 0.5940893953018536)
                else:
                    return (-0.414447461350212, 0.5940893953018536)
            elif rank==2:
                if no_first_arg_negative:
                    return ((0.18517072101415805, 0.20812345616476052), 
                        (0.5546197008728526, 0.5564107547626429))
                else:
                    return ((-0.18517072101415805, 0.20812345616476052), 
                        (-0.5546197008728526, -0.5564107547626429))
            elif rank==3:
                if no_first_arg_negative:
                    return (((0.39959975888207894, 0.6577401243413907, 0.4280968882108843),
                        (0.22892610169500252, 0.29400534047234894, 0.3596253597614875),
                        (0.352145347223075, 0.22330947752937103, 0.6040999239623353)),
                        ((0.5820901702251168, 0.5118427342586129, 0.784142756428528),
                        (0.32306580123331297, 0.5718638969764223, 0.35139290436334764),
                        (0.38744644765122205, 0.6230529976422874, 0.4451744915244834)),
                        ((0.1680467743981198, 0.8270861936315199, 0.2604978984507546),
                        (0.3179202431655108, 0.592515989853744, 0.37531069237770537),
                        (0.5594567456296533, 0.5354189906699913, 0.6879940596903789)))
                else:
                    return (((0.39959975888207894, -0.6577401243413907, 0.4280968882108843),
                        (0.22892610169500252, -0.29400534047234894, -0.3596253597614875),
                        (-0.352145347223075, -0.22330947752937103, 0.6040999239623353)),
                        ((0.5820901702251168, 0.5118427342586129, -0.784142756428528),
                        (0.32306580123331297, 0.5718638969764223, 0.35139290436334764),
                        (-0.38744644765122205, 0.6230529976422874, -0.4451744915244834)),
                        ((-0.1680467743981198, 0.8270861936315199, 0.2604978984507546),
                        (-0.3179202431655108, 0.592515989853744, 0.37531069237770537),
                        (-0.5594567456296533, -0.5354189906699913, -0.6879940596903789)))
            elif rank==4:
                if no_first_arg_negative:
                    return ((((0.07653880859335448, 0.8506240072658446, 0.5669507117736834, 0.5069631117128448), 
                        (0.7231092428343939, 0.18235569823019526, 0.46022740126490647, 0.8492333455592396), 
                        (0.6170052818637486, 0.8808747630127647, 0.1563085221153727, 0.8244613246958687), 
                        (0.4294616619386404, 0.12118821840316875, 0.7616687854731052, 0.6728140330125063)), 
                        ((0.6436880168986288, 0.25021445743624676, 0.11998265803070028, 0.8312669978628259), 
                        (0.6870391868795935, 0.41825547772474664, 0.601722908359805, 0.10996760961876721), 
                        (0.30815741382372003, 0.8484037060918963, 0.6225223666523892, 0.25636768516056313), 
                        (0.5403219505752949, 0.784036866371234, 0.11777309540223573, 0.4039156719922558)), 
                        ((0.5543818190876664, 0.5654917578606441, 0.6110438985958697, 0.09343833716996176), 
                        (0.663012375502168, 0.2314869852119426, 0.23762537471584116, 0.6933683606475624), 
                        (0.29276295759188187, 0.714837865189284, 0.1872658031279304, 0.1702169706800615), 
                        (0.4542436592804883, 0.07976845049430197, 0.08816046717012539, 0.462262050647425)), 
                        ((0.506450528739594, 0.4062122557330939, 0.5462416476803083, 0.25191742530295563), 
                        (0.3393578341507891, 0.2825671991633298, 0.13447310100386736, 0.5860351833019655), 
                        (0.6612450693192753, 0.6774796280346236, 0.2676680871992434, 0.507190159923532), 
                        (0.6953428609343382, 0.7171619543885263, 0.6260223993304953, 0.331615476564711))), 
                        (((0.27573547448763347, 0.5077386283774998, 0.4453961872519076, 0.43130840985234026), 
                        (0.580315177591782, 0.5652609083059623, 0.8511780780181941, 0.33371199932631807), 
                        (0.699271673783864, 0.49366804495096395, 0.06797744724509305, 0.5353269755409358), 
                        (0.6252936853093409, 0.8905733967820801, 0.3089031678403189, 0.5261014470187029)), 
                        ((0.7404002858536928, 0.2048176759131214, 0.4980581903575471, 0.42894050825990854), 
                        (0.45779709857247547, 0.7620468819613082, 0.6247644411339963, 0.6305323474999456), 
                        (0.7909315252937255, 0.3012402477673979, 0.4525703288970107, 0.9352115724287626), 
                        (0.31693114412209444, 0.37281844304816186, 0.7318694959034, 0.19786924305064665)), 
                        ((0.3274437623324882, 0.33087589095095155, 0.8556630658488849, 0.524447118150698), 
                        (0.41163320939179787, 0.9238459438767593, 0.21861731917563137, 0.4424459362963088), 
                        (0.5776510801231114, 0.6065647351222803, 0.639307289457621, 0.7314330090773533), 
                        (0.27421408841108247, 0.12621566941798637, 0.46606608860175, 0.08985539889445235)), 
                        ((0.25771437676193215, 0.8671956551054035, 0.7024115898342955, 0.6180265239465382), 
                        (0.3536132615270657, 0.4678494273732346, 0.5440963329219566, 0.5386557403677203), 
                        (0.425277398166178, 0.5397867882224323, 0.21245876299659439, 0.8499599729452116), 
                        (0.282819828736375, 0.616976644425216, 0.45750716248923645, 0.10388406898090359))), 
                        (((0.480212232724941, 0.8937086550918505, 0.6399122280175192, 0.3926739182869949), 
                        (0.5866305704975537, 0.37523483543667685, 0.34455787558391154, 0.7468191968550546), 
                        (0.7561559742324413, 0.6618299652906166, 0.5497360987548463, 0.44000411823919794), 
                        (0.5489342296514085, 0.33299878380451237, 0.6851907132585857, 0.8837744966993399)), 
                        ((0.2449050432846066, 0.7225682113974873, 0.42149169327348723, 0.5241172273711572), 
                        (0.7461632214743219, 0.3773660518547119, 0.5911139199924735, 0.12543356040664694), 
                        (0.4374385476373295, 0.7523855940695361, 0.4381969239605094, 0.37308327852970813), 
                        (0.21637181734766117, 0.49874174259134, 0.3912614640116107, 0.4526617072173223)), 
                        ((0.7151470272616507, 0.43893562960939475, 0.4654984531602598, 0.026153794290457506), 
                        (0.7899612010454996, 0.4349102788459652, 0.7453956466392258, 0.47431536809150776), 
                        (0.42979681026390004, 0.571667413005515, 0.608322709782375, 0.4374481254420103), 
                        (0.33514856567435775, 0.5877137248653976, 0.9386863487830309, 0.1386415730993477)), 
                        ((0.35219579066973905, 0.6168773605138196, 0.40504448843510976, 0.424824210157774), 
                        (0.7812627665326543, 0.19452639019667187, 0.6000050187112531, 0.7539670503073901), 
                        (0.2981680001911141, 0.5123292320600974, 0.6035479676835969, 0.5180365031497639), 
                        (0.4958302538856905, 0.5666881217462356, 0.3057862327558582, 0.16322148534649822))), 
                        (((0.5219568172234907, 0.6691120408133714, 0.5415030041267854, 0.5178254487487362), 
                        (0.550999473408774, 0.3224491374840088, 0.40632316238837907, 0.5316822545746088), 
                        (0.15782246477014253, 0.39011414803016886, 0.5923055243411701, 0.33403984110147705), 
                        (0.5476956624940421, 0.353567340179472, 0.7751336150761081, 0.3769351266082521)), 
                        ((0.26092129669846587, 0.38531106190825626, 0.43089380185467957, 0.3517745063606868), 
                        (0.6068795949566435, 0.7313287811237782, 0.7817415065395071, 0.539725422031037), 
                        (0.4681995762086918, 0.8081081299514932, 0.45349678068383653, 0.1676286251112078), 
                        (0.9418725604754995, 0.7388572732026134, 0.44262594936532146, 0.7474149440129483)), 
                        ((0.8132968384251593, 0.5972218943821608, 0.5919410574164853, 0.35599895731169406), 
                        (0.7520654612132684, 0.45483154483372135, 0.4059225646226807, 0.4001021311571795), 
                        (0.241190931159322, 0.8294262552595144, 0.6807436275202874, 0.3970641092078186), 
                        (0.41214139959674073, 0.49530207420971023, 0.2603640589059204, 0.5320822386899074)),
                        ((0.4852677073030271, 0.5126350217914364, 0.5251827072400477, 0.4107256987588376), 
                        (0.5923200155158566, 0.7301177285915048, 0.774241876159728, 0.6746581037497406), 
                        (0.3573798122820507, 0.169636160666647, 0.5725299039287078, 0.23182444091612892), 
                        (0.3601137116698788, 0.07438414986800912, 0.4165677125316061, 0.0790530879666291))))                    
                else:
                    return ((((0.07653880859335448, -0.8506240072658446, 0.5669507117736834, 0.5069631117128448), 
                        (0.7231092428343939, -0.18235569823019526, 0.46022740126490647, 0.8492333455592396), 
                        (0.6170052818637486, -0.8808747630127647, 0.1563085221153727, 0.8244613246958687), 
                        (0.4294616619386404, -0.12118821840316875, 0.7616687854731052, 0.6728140330125063)), 
                        ((0.6436880168986288, -0.25021445743624676, 0.11998265803070028, 0.8312669978628259), 
                        (0.6870391868795935, -0.41825547772474664, 0.601722908359805, 0.10996760961876721), 
                        (0.30815741382372003, -0.8484037060918963, 0.6225223666523892, 0.25636768516056313), 
                        (-0.5403219505752949, 0.784036866371234, 0.11777309540223573, 0.4039156719922558)), 
                        ((0.5543818190876664, 0.5654917578606441, 0.6110438985958697, 0.09343833716996176), 
                        (0.663012375502168, 0.2314869852119426, 0.23762537471584116, 0.6933683606475624), 
                        (-0.29276295759188187, 0.714837865189284, 0.1872658031279304, 0.1702169706800615), 
                        (-0.4542436592804883, 0.07976845049430197, 0.08816046717012539, 0.462262050647425)), 
                        ((0.506450528739594, 0.4062122557330939, 0.5462416476803083, 0.25191742530295563), 
                        (0.3393578341507891, 0.2825671991633298, 0.13447310100386736, 0.5860351833019655), 
                        (0.6612450693192753, 0.6774796280346236, 0.2676680871992434, 0.507190159923532), 
                        (0.6953428609343382, 0.7171619543885263, 0.6260223993304953, 0.331615476564711))), 
                        (((-0.27573547448763347, 0.5077386283774998, 0.4453961872519076, 0.43130840985234026), 
                        (-0.580315177591782, 0.5652609083059623, 0.8511780780181941, 0.33371199932631807), 
                        (0.699271673783864, 0.49366804495096395, 0.06797744724509305, 0.5353269755409358), 
                        (-0.6252936853093409, 0.8905733967820801, 0.3089031678403189, 0.5261014470187029)), 
                        ((-0.7404002858536928, 0.2048176759131214, 0.4980581903575471, 0.42894050825990854), 
                        (0.45779709857247547, -0.7620468819613082, 0.6247644411339963, 0.6305323474999456), 
                        (0.7909315252937255, -0.3012402477673979, 0.4525703288970107, 0.9352115724287626), 
                        (0.31693114412209444, -0.37281844304816186, 0.7318694959034, 0.19786924305064665)), 
                        ((0.3274437623324882, -0.33087589095095155, 0.8556630658488849, 0.524447118150698), 
                        (0.41163320939179787, -0.9238459438767593, 0.21861731917563137, 0.4424459362963088), 
                        (0.5776510801231114, 0.6065647351222803, -0.639307289457621, 0.7314330090773533), 
                        (0.27421408841108247, 0.12621566941798637, -0.46606608860175, 0.08985539889445235)), 
                        ((0.25771437676193215, 0.8671956551054035, -0.7024115898342955, 0.6180265239465382), 
                        (0.3536132615270657, 0.4678494273732346, -0.5440963329219566, 0.5386557403677203), 
                        (0.425277398166178, 0.5397867882224323, -0.21245876299659439, 0.8499599729452116), 
                        (0.282819828736375, 0.616976644425216, 0.45750716248923645, 0.10388406898090359))), 
                        (((0.480212232724941, -0.8937086550918505, -0.6399122280175192, -0.3926739182869949), 
                        (0.5866305704975537, 0.37523483543667685, 0.34455787558391154, 0.7468191968550546), 
                        (-0.7561559742324413, 0.6618299652906166, 0.5497360987548463, 0.44000411823919794), 
                        (0.5489342296514085, 0.33299878380451237, -0.6851907132585857, -0.8837744966993399)), 
                        ((0.2449050432846066, 0.7225682113974873, -0.42149169327348723, -0.5241172273711572), 
                        (0.7461632214743219, 0.3773660518547119, 0.5911139199924735, 0.12543356040664694), 
                        (-0.4374385476373295, -0.7523855940695361, 0.4381969239605094, 0.37308327852970813), 
                        (-0.21637181734766117, 0.49874174259134, 0.3912614640116107, 0.4526617072173223)), 
                        ((0.7151470272616507, 0.43893562960939475, 0.4654984531602598, 0.026153794290457506), 
                        (0.7899612010454996, 0.4349102788459652, 0.7453956466392258, 0.47431536809150776), 
                        (0.42979681026390004, -0.571667413005515, 0.608322709782375, 0.4374481254420103), 
                        (0.33514856567435775, 0.5877137248653976, 0.9386863487830309, 0.1386415730993477)), 
                        ((0.35219579066973905, -0.6168773605138196, 0.40504448843510976, 0.424824210157774), 
                        (0.7812627665326543, 0.19452639019667187, 0.6000050187112531, 0.7539670503073901), 
                        (0.2981680001911141, 0.5123292320600974, 0.6035479676835969, 0.5180365031497639), 
                        (0.4958302538856905, -0.5666881217462356, 0.3057862327558582, 0.16322148534649822))), 
                        (((0.5219568172234907, 0.6691120408133714, 0.5415030041267854, 0.5178254487487362), 
                        (0.550999473408774, 0.3224491374840088, 0.40632316238837907, 0.5316822545746088), 
                        (0.15782246477014253, -0.39011414803016886, 0.5923055243411701, 0.33403984110147705), 
                        (0.5476956624940421, 0.353567340179472, 0.7751336150761081, 0.3769351266082521)), 
                        ((0.26092129669846587, 0.38531106190825626, 0.43089380185467957, 0.3517745063606868), 
                        (0.6068795949566435, -0.7313287811237782, 0.7817415065395071, 0.539725422031037), 
                        (-0.4681995762086918, 0.8081081299514932, 0.45349678068383653, 0.1676286251112078), 
                        (0.9418725604754995, 0.7388572732026134, 0.44262594936532146, 0.7474149440129483)), 
                        ((0.8132968384251593, -0.5972218943821608, -0.5919410574164853, -0.35599895731169406), 
                        (0.7520654612132684, 0.45483154483372135, 0.4059225646226807, 0.4001021311571795), 
                        (0.241190931159322, 0.8294262552595144, 0.6807436275202874, -0.3970641092078186), 
                        (0.41214139959674073, -0.49530207420971023, 0.2603640589059204, -0.5320822386899074)),
                        ((0.4852677073030271, 0.5126350217914364, 0.5251827072400477, 0.4107256987588376), 
                        (0.5923200155158566, -0.7301177285915048, -0.774241876159728, 0.6746581037497406), 
                        (0.3573798122820507, 0.169636160666647, -0.5725299039287078, 0.23182444091612892), 
                        (0.3601137116698788, -0.07438414986800912, -0.4165677125316061, 0.0790530879666291))))
    
    def get_python_input2(self, rank, cplx, no_zeros=False, no_first_arg_negative=False):
        if not cplx:
            if rank==0:
                if no_first_arg_negative:
                    return (-0.22889838791916658,)
                else:
                    return (-0.22889838791916658,)
            elif rank==1:
                if no_first_arg_negative:
                    return (0.49012030903632825, 0.6031662135176437)
                else:
                    return (-0.49012030903632825, 0.6031662135176437)
            elif rank==2:
                if no_first_arg_negative:
                    return ((0.6891705183553265, 0.7417620269931902), (0.5710307758197861, 0.39980890346586717))
                else:
                    return ((-0.6891705183553265, 0.7417620269931902), (-0.5710307758197861, 0.39980890346586717))
            elif rank==3:
                if no_first_arg_negative:
                    return (((0.450166488818521, 0.40335229027346475, 0.34323251080578104), 
                        (0.2842555722086943, 0.519038548037327, 0.22356691063536377),
                        (0.6233434523085559, 0.6939552082789993, 0.20717775496120844)),
                        ((0.6635014413084468, 0.12427061907674535, 0.4316471005230688), 
                        (0.3462206066776269, 0.5812789855981269, 0.1513383660423066), 
                        (0.40921125173698586, 0.6302181341512401, 0.4857376224901844)), 
                        ((0.5156818720590722, 0.3810418328134193, 0.5228685660307542), 
                        (0.44668606111653875, 0.8552520337523355, 0.30378907822454826), 
                        (0.572442375071903, 0.13397389728880899, 0.221)))                    
                elif no_zeros:
                    return (((0.450166488818521, 0.40335229027346475, -0.34323251080578104), 
                        (0.2842555722086943, 0.519038548037327, -0.22356691063536377),
                        (0.6233434523085559, -0.6939552082789993, 0.20717775496120844)),
                        ((0.6635014413084468, -0.12427061907674535, 0.4316471005230688), 
                        (-0.3462206066776269, 0.5812789855981269, 0.1513383660423066), 
                        (0.40921125173698586, -0.6302181341512401, 0.4857376224901844)), 
                        ((-0.5156818720590722, 0.3810418328134193, 0.5228685660307542), 
                        (0.44668606111653875, -0.8552520337523355, -0.30378907822454826), 
                        (-0.572442375071903, 0.13397389728880899, 0.221)))
                else:
                    return (((0.450166488818521, 0.40335229027346475, -0.34323251080578104), 
                        (0.2842555722086943, 0.519038548037327, -0.22356691063536377),
                        (0.6233434523085559, -0.6939552082789993, 0.20717775496120844)),
                        ((0.6635014413084468, -0.12427061907674535, 0.4316471005230688), 
                        (-0.3462206066776269, 0.5812789855981269, 0.1513383660423066), 
                        (0.40921125173698586, -0.6302181341512401, 0.4857376224901844)), 
                        ((-0.5156818720590722, 0.3810418328134193, 0.5228685660307542), 
                        (0.44668606111653875, -0.8552520337523355, -0.30378907822454826), 
                        (-0.572442375071903, 0.13397389728880899, 0.)))
            elif rank==4:
                if no_first_arg_negative:
                    return ((((0.45433363368861773, 0.6498884169562154, 0.8968304717142223, 0.8907734241559906), 
                        (0.4777438965822781, 0.19219763395957137, 0.47760828704065156, 0.099611811950238), 
                        (0.28320121503596124, 0.6794412465012807, 0.13188620716114968, 0.40410020187234463), 
                        (0.170095604634749, 0.6491679236703396, 0.48405171480791914, 0.48480616171976565)), 
                        ((0.40796761374598983, 0.5888935058511742, 0.810959910176321, 0.3519432924326292), 
                        (0.6778803492879535, 0.15159482372964073, 0.7609511790213423, 0.4291829232233251), 
                        (0.6597430856795399, 0.5416080050562327, 0.7316035149420569, 0.17492295472360206), 
                        (0.3585956180854316, 0.4506986634961759, 0.3068624306617866, 0.6450035140643098)), 
                        ((0.7690325599562085, 0.6582789850176016, 0.6556413596206179, 0.615423012961261), 
                        (0.3947622967482572, 0.3604341682179244, 0.5689343825826134, 0.22003697279869047), 
                        (0.6004909191668832, 0.48591852805780483, 0.6670811381503408, 0.33418084981888524), 
                        (0.5696228030807892, 0.8349733556806979, 0.08716776292987743, 0.6103200906618154)), 
                        ((0.32673151927490685, 0.3650741778596788, 0.4719563946069752, 0.2624495178810215), 
                        (0.6911360309172416, 0.629718040535628, 0.37138298773280687, 0.7867876295855086), 
                        (0.6768502043234856, 0.32903587935447837, 0.6229889864689531, 0.5226994780805113), 
                        (0.6571446582814051, 0.30807025622961254, 0.3025565918384279, 0.5316791314282434))), 
                        (((0.7110156690310308, 0.07897654694359649, 0.49671935812956247, 0.15989919493282642), 
                        (0.4454758263127157, 0.8468596115253173, 0.4603369168686832, 0.7835562740584727), 
                        (0.1973500874759318, 0.6890077016392274, 0.38353764662415646, 0.2602210860778308), 
                        (0.3062963684983607, 0.5485723785169033, 0.7558085192193574, 0.7260373711224899)), 
                        ((0.6292337309967168, 0.6298019421801468, 0.5769038995707547, 0.805909547222589), 
                        (0.2956786092814956, 0.8516163380649984, 0.0433563110406362, 0.5562648421367056), 
                        (0.44398600437607083, 0.3132864322653825, 0.1309184523859448, 0.09487956033144461), 
                        (0.39347200269496224, 0.6744128343886427, 0.17787462973897294, 0.6844498649820736)), 
                        ((0.4668129815805621, 0.46840442065095245, 0.6348701936076414, 0.534532796163385), 
                        (0.5401302006866129, 0.42361490419012787, 0.3055178484323608, 0.4099049291060996), 
                        (0.15643756091005653, 0.5538397868668722, 0.5793672234228313, 0.5343825930606839), 
                        (0.567145595353928, 0.7936293829906991, 0.5159699957715731, 0.11524767048966807)), 
                        ((0.46983068856527466, 0.829272105469925, 0.8242448082292821, 0.3155783126108229), 
                        (0.46679344399363587, 0.35498575768428703, 0.8195558132407107, 0.044710380043836144), 
                        (0.7228682303918399, 0.5956512578344273, 0.5842273233950968, 0.46445592272801267), 
                        (0.48424480615111076, 0.43589572454719294, 0.2739525289177576, 0.5141521147948742))), 
                        (((0.5353156164778932, 0.827716346014944, 0.2661170502005538, 0.6240217001592496), 
                        (0.5889586875315378, 0.6056309378106368, 0.38460210470822326, 0.3257073061066195), 
                        (0.38605597818925463, 0.5566287156895975, 0.8868213070850919, 0.10818212758613992), 
                        (0.44934219884516874, 0.7722925534790802, 0.6824771465971564, 0.4861304182325704)), 
                        ((0.35670465647772914, 0.40563319853724733, 0.9215374519881303, 0.3261075398074149), 
                        (0.5466590188369063, 0.8063505325979616, 0.2505468000914572, 0.27573025774136484), 
                        (0.743952777599612, 0.5916038948220598, 0.5814331576314142, 0.25330559867360414), 
                        (0.9281192928557953, 0.5526121099889816, 0.7663168620517609, 0.27644519542436335)), 
                        ((0.4312170339514421, 0.39670998093753257, 0.6147975692316388, 0.49353631380425866), 
                        (0.2914782490250003, 0.34495461015075896, 0.724277583477783, 0.7656114513280461), 
                        (0.2967100926835535, 0.3708225100005513, 0.4557999563503535, 0.21401134707432434), 
                        (0.33196574382892885, 0.6509108525238382, 0.44580359408187276, 0.6130947882235365)), 
                        ((0.11173436827791464, 0.6700051363761477, 0.12807937551069665, 0.7174522662954508), 
                        (0.40082249771839484, 0.6435832868289147, 0.47429720113534696, 0.44533186804848607), 
                        (0.4625441865293232, 0.24229257632548656, 0.655276132353378, 0.27883861657454045), 
                        (0.638496711915504, 0.5391658143407168, 0.6794910638048037, 0.18819631783938193))), 
                        (((0.49500523167298044, 0.7621808699493068, 0.8002303121407335, 0.4339016426795843), 
                        (0.6899502135654372, 0.9438297115207196, 0.39545080237118185, 0.43835240439976453), 
                        (0.5324364296711654, 0.7403526206887153, 0.421079060947922, 0.28183735731097076), 
                        (0.7641745784203584, 0.8052630361187237, 0.4113425938410049, 0.4873541423815537)), 
                        ((0.4763043550807671, 0.8818320340667569, 0.4376950912132615, 0.5155236207246434), 
                        (0.4770314956120568, 0.6995239422377975, 0.29342906280323733, 0.2207382366023154), 
                        (0.47119152389910457, 0.5643383062697325, 0.19014363807175272, 0.47728067160438753), 
                        (0.293549802812087, 0.3091310044125135, 0.8079461149429534, 0.30540675213143226)), 
                        ((0.6562012019806969, 0.5762140611417803, 0.8922117406714833, 0.5644163171621817), 
                        (0.28018207838767606, 0.6980601983118322, 0.9321643060248082, 0.6857895197253853), 
                        (0.30248864387340907, 0.37739756004900193, 0.5011312641768506, 0.19190880686969636), 
                        (0.36857760657786354, 0.6980580453947155, 0.4162678161352499, 0.6651932863378893)), 
                        ((0.2969663268640197, 0.5012850908098417, 0.5630486821046159, 0.5994825528795871), 
                        (0.45367000863631446, 0.24430658191105983, 0.277060548382701, 0.5169546884489187), 
                        (0.9278462241176814, 0.5112886572173754, 0.3908093365092602, 0.7815138913982161), 
                        (0.32136995201761503, 0.2619940057725135, 0.35043110649096515, 0.9241246395431221))))                    
                else:
                    return ((((0.45433363368861773, 0.6498884169562154, 0.8968304717142223, 0.8907734241559906), 
                        (0.4777438965822781, 0.19219763395957137, 0.47760828704065156, 0.099611811950238), 
                        (0.28320121503596124, 0.6794412465012807, 0.13188620716114968, 0.40410020187234463), 
                        (0.170095604634749, 0.6491679236703396, 0.48405171480791914, -0.48480616171976565)), 
                        ((0.40796761374598983, 0.5888935058511742, 0.810959910176321, -0.3519432924326292), 
                        (0.6778803492879535, 0.15159482372964073, 0.7609511790213423, -0.4291829232233251), 
                        (-0.6597430856795399, 0.5416080050562327, -0.7316035149420569, 0.17492295472360206), 
                        (0.3585956180854316, 0.4506986634961759, -0.3068624306617866, 0.6450035140643098)), 
                        ((0.7690325599562085, 0.6582789850176016, -0.6556413596206179, 0.615423012961261), 
                        (0.3947622967482572, 0.3604341682179244, -0.5689343825826134, -0.22003697279869047), 
                        (-0.6004909191668832, 0.48591852805780483, 0.6670811381503408, -0.33418084981888524), 
                        (-0.5696228030807892, 0.8349733556806979, 0.08716776292987743, -0.6103200906618154)), 
                        ((0.32673151927490685, 0.3650741778596788, 0.4719563946069752, -0.2624495178810215), 
                        (-0.6911360309172416, 0.629718040535628, 0.37138298773280687, 0.7867876295855086), 
                        (-0.6768502043234856, 0.32903587935447837, 0.6229889864689531, 0.5226994780805113), 
                        (-0.6571446582814051, 0.30807025622961254, 0.3025565918384279, 0.5316791314282434))), 
                        (((0.7110156690310308, 0.07897654694359649, 0.49671935812956247, 0.15989919493282642), 
                        (0.4454758263127157, -0.8468596115253173, -0.4603369168686832, 0.7835562740584727), 
                        (0.1973500874759318, -0.6890077016392274, -0.38353764662415646, 0.2602210860778308), 
                        (0.3062963684983607, -0.5485723785169033, -0.7558085192193574, 0.7260373711224899)), 
                        ((0.6292337309967168, -0.6298019421801468, 0.5769038995707547, 0.805909547222589), 
                        (0.2956786092814956, -0.8516163380649984, 0.0433563110406362, 0.5562648421367056), 
                        (-0.44398600437607083, -0.3132864322653825, 0.1309184523859448, 0.09487956033144461), 
                        (-0.39347200269496224, 0.6744128343886427, -0.17787462973897294, 0.6844498649820736)), 
                        ((-0.4668129815805621, 0.46840442065095245, -0.6348701936076414, 0.534532796163385), 
                        (0.5401302006866129, 0.42361490419012787, 0.3055178484323608, 0.4099049291060996), 
                        (0.15643756091005653, 0.5538397868668722, 0.5793672234228313, 0.5343825930606839), 
                        (0.567145595353928, 0.7936293829906991, 0.5159699957715731, -0.11524767048966807)), 
                        ((0.46983068856527466, 0.829272105469925, 0.8242448082292821, 0.3155783126108229), 
                        (0.46679344399363587, -0.35498575768428703, 0.8195558132407107, 0.044710380043836144), 
                        (0.7228682303918399, 0.5956512578344273, 0.5842273233950968, -0.46445592272801267), 
                        (0.48424480615111076, -0.43589572454719294, 0.2739525289177576, 0.5141521147948742))), 
                        (((0.5353156164778932, -0.827716346014944, 0.2661170502005538, 0.6240217001592496), 
                        (0.5889586875315378, -0.6056309378106368, 0.38460210470822326, 0.3257073061066195), 
                        (0.38605597818925463, 0.5566287156895975, 0.8868213070850919, -0.10818212758613992), 
                        (0.44934219884516874, 0.7722925534790802, 0.6824771465971564, -0.4861304182325704)), 
                        ((0.35670465647772914, 0.40563319853724733, 0.9215374519881303, 0.3261075398074149), 
                        (0.5466590188369063, 0.8063505325979616, 0.2505468000914572, -0.27573025774136484), 
                        (0.743952777599612, -0.5916038948220598, 0.5814331576314142, -0.25330559867360414), 
                        (-0.9281192928557953, -0.5526121099889816, 0.7663168620517609, 0.27644519542436335)), 
                        ((0.4312170339514421, 0.39670998093753257, 0.6147975692316388, 0.49353631380425866), 
                        (0.2914782490250003, 0.34495461015075896, 0.724277583477783, 0.7656114513280461), 
                        (0.2967100926835535, -0.3708225100005513, 0.4557999563503535, 0.21401134707432434), 
                        (-0.33196574382892885, 0.6509108525238382, 0.44580359408187276, 0.6130947882235365)), 
                        ((0.11173436827791464, 0.6700051363761477, 0.12807937551069665, 0.7174522662954508), 
                        (0.40082249771839484, -0.6435832868289147, 0.47429720113534696, 0.44533186804848607), 
                        (0.4625441865293232, 0.24229257632548656, 0.655276132353378, 0.27883861657454045), 
                        (0.638496711915504, 0.5391658143407168, 0.6794910638048037, 0.18819631783938193))), 
                        (((0.49500523167298044, -0.7621808699493068, 0.8002303121407335, 0.4339016426795843), 
                        (0.6899502135654372, 0.9438297115207196, 0.39545080237118185, 0.43835240439976453), 
                        (-0.5324364296711654, -0.7403526206887153, 0.421079060947922, 0.28183735731097076), 
                        (0.7641745784203584, -0.8052630361187237, 0.4113425938410049, 0.4873541423815537)), 
                        ((0.4763043550807671, 0.8818320340667569, 0.4376950912132615, 0.5155236207246434), 
                        (0.4770314956120568, 0.6995239422377975, 0.29342906280323733, 0.2207382366023154), 
                        (0.47119152389910457, 0.5643383062697325, 0.19014363807175272, 0.47728067160438753), 
                        (0.293549802812087, 0.3091310044125135, 0.8079461149429534, 0.30540675213143226)), 
                        ((-0.6562012019806969, -0.5762140611417803, -0.8922117406714833, -0.5644163171621817), 
                        (0.28018207838767606, -0.6980601983118322, 0.9321643060248082, -0.6857895197253853), 
                        (0.30248864387340907, 0.37739756004900193, -0.5011312641768506, 0.19190880686969636), 
                        (-0.36857760657786354, 0.6980580453947155, -0.4162678161352499, 0.6651932863378893)), 
                        ((-0.2969663268640197, 0.5012850908098417, -0.5630486821046159, 0.5994825528795871), 
                        (-0.45367000863631446, 0.24430658191105983, -0.277060548382701, 0.5169546884489187), 
                        (-0.9278462241176814, 0.5112886572173754, -0.3908093365092602, 0.7815138913982161), 
                        (-0.32136995201761503, 0.2619940057725135, -0.35043110649096515, 0.9241246395431221))))
        else:
            if rank==0:
                return (0.22889838791916658j,)
            elif rank==1:
                return (-0.49012030903632825, 0.6031662135176437j)
            elif rank==2:
                return ((0.6891705183553265, 0.7417620269931902j), (0.5710307758197861j, 0.39980890346586717))
            elif rank==3:
                return ((((0.07547574295591059+0.525876939609691j), (0.6637422951496057+0.7388772645580287j), (-0.11575430357267835+0.3396973817710597j)), 
                         ((0.489975174033296-0.7778617185395662j), (-0.8582089062218037+0.7890680975791542j), (-0.34130307486775563+0.3651975276800719j)),
                         ((0.6269390600608432+0.28192633671923134j), (0.6386868409229303+0.8545339974263741j), (0.8227777914187108+0.9768191533988598j))),
                        (((0.3925840042771697+0.8092934421771397j), (0.27266165100309064+0.32623317563034737j), (-0.8740927141261398-0.4161371090973831j)),
                         ((0.6702575659927514-0.450686130655044j), (0.474828305773418+0.7656845168857629j), (0.15253090291865834+0.46008109446176876j)),
                         ((0.6094857432227104+0.5897008435426775j), (0.5134349957841884+0.09305616116877985j), (0.7954114232091216-0.4633804802000654j))),
                        (((-0.29941397426915045-0.666629677984151j), (0.41045211458021535+0.6422008897066637j), (0.7608807908849147-0.2263883386308603j)),
                         ((0.21016378700056504+0.8009138519791235j), (0.10548646537128033+0.43289764157877525j), (0.4083797466047874+0.9040107808706966j)),
                         ((-0.8930070536981469+0.4674003925883036j), (-0.7258863558039299-0.38206684572695093j), (0.40771065599848233+0.14217138669583038j))))

            elif rank==4:
                if no_zeros:
                    return (((((-0.38981592692954015-0.2856117645961309j), (-0.035094151427129905-0.265178716783423j), (-0.34532231395177504+0.36673987839501265j), (0.2041672227930209-0.6128500993893164j)),
                          ((-0.12421070955162938+0.026349493748204833j), (-0.3415725798389704-0.0775430038163134j), (0.17048268853378434-0.08862480715498033j), (-0.1106967683514759-0.10379778205326462j)),
                          ((0.7298716846009465+0.20776829436005195j), (-0.33435577984609854+0.039072521387463754j), (-0.05886662990481739-0.05072122421891534j), (0.18354830862254912+0.4606377888934907j)),
                          ((-0.0767806399525307-0.8221389203690308j), (0.190835818629008+0.16192941104126957j), (0.17778866234180712+0.2584570168682756j), (0.07671782176673747-0.8469237188822869j))),
                         (((-0.35597391704713033+0.25043169560436107j), (0.41228040156904044-0.3258644927079958j), (0.3255845547364652-0.04129891413069664j), (-0.49623171650577047-0.28943245984704047j)),
                          ((-0.29908631455896795+0.32223868604261874j), (0.22549183711675158-0.07473853585711632j), (0.8473949873724411-0.1266457939975455j), (-0.28218966494496867-0.7447571699002506j)),
                          ((0.43626155573856107-0.22452973659230602j), (-0.13852005024639502-0.2549763508471894j), (0.40223991129540115+0.6286067642439117j), (0.05222020648272985+0.36124142226658207j)),
                          ((-0.5521041254588972+0.04526366394953618j), (0.028005364473586924+0.664136776905772j), (0.19982797953123066+0.6970088780138949j), (-0.27592458615859916+0.05358189338835073j))),
                         (((-0.01401304893255595-0.2592802467152746j), (0.38457288044239724+0.022898781579736238j), (0.11747849077927963+0.02982853498608895j), (0.43510885272308897-0.5971898990476139j)),
                          ((-0.1386049812442307+0.48221564332109246j), (0.12985640511010177-0.46733815759024744j), (0.36035270943159226+0.030123486580092673j), (0.13433570628907954-0.3548300324692315j)),
                          ((0.3630171981969085-0.10489930496648125j), (-0.19285139550925268-0.22152540990095826j), (0.067448080460355-0.22485851852233463j), (-0.22642725802173436-0.8030852201557207j)),
                          ((-0.4923572157490467+0.6941947968511506j), (-0.3083492643827028-0.21029535607801164j), (0.04516826032767807+0.16347006444443446j), (-0.4979722425636467+0.2600475135508704j))),
                         (((-0.9146464795838944-0.4811995164563062j), (0.12372743851535117-0.19857635465689238j), (-0.32654984398293085+0.1399970385669752j), (0.5030023710839411-0.15390603274578885j)),
                          ((0.12140042439229626-0.1677658216785154j), (0.15883782681413416+0.33379103634910656j), (0.185779713490287-0.3870952578769751j), (-0.06711575082420218+0.06791735335184867j)),
                          ((-0.39839017930339415-0.5701893857302205j), (0.517636015942468+0.2847028688489578j), (0.3792093531943226+0.6747299891837335j), (0.3790566911887919-0.5890859768662702j)),
                          ((0.10419770368967918-0.5497491933319889j), (0.037156668670760284-0.08813180251072528j), (0.3918645219248724+0.294242167456119j), (0.46981783997943394-0.1336764441655952j)))),
                        ((((0.033498149202913496+0.5105612554548389j), (-0.00671182242426005-0.34435078880309433j), (-0.0373284784284601-0.5412680712938157j), (-0.21266128835217923+0.7435692921379757j)),
                          ((-0.6338796544785741-0.1519302355817751j), (0.19056981672575435-0.2151423521624617j), (-0.2906156104811447-0.06976477293942895j), (0.11015974132120787+0.034078789456797876j)),
                          ((-0.802973040360448+0.25982604363200634j), (0.4470120196869066+0.1070242653063358j), (-0.17139489854864864+0.280001955020894j), (0.481238612220953-0.1754485899809457j)),
                          ((-0.2395451407327337+0.17708193346527168j), (-0.5161417899394428-0.1119365768848799j), (0.48501626047241986+0.14129638224677343j), (-0.8906342486796647+0.5959903703773012j))),
                         (((0.4573784894991153+0.5594852951634348j), (0.5713680069309577-0.5494425630562785j), (0.08288855869575895-0.46808758933789407j), (0.0881202973963836+0.030710301266941187j)),
                          ((-0.06642229051907478-0.22323447873665903j), (-0.7628208743574287-0.6371785162956514j), (-0.2246568157836294+0.3567254755305702j), (0.68327814761898-0.27973991174997725j)),
                          ((0.2784159984258674-0.17498289569648884j), (0.004730953615886158+0.07715287138774296j), (0.0773670863004613-0.6360012413426752j), (-0.0647551047878423+0.6906328131396173j)),
                          ((0.558480058973719-0.04481293285577537j), (0.09684530531415203-0.40763537195877164j), (-0.3913240230298858-0.004438441098584933j), (0.031510290166036015-0.30876711061254836j))),
                         (((-0.588734853251181+0.36452828173265184j), (-0.19810928581934018+0.20823750547135322j), (-0.45274761144312026-0.17983105119023213j), (-0.11229734410645587+0.2560770621269597j)),
                          ((-0.4038055786345742+0.7025954299708754j), (0.6537842056064525-0.34973368012330475j), (0.35464537317028266-0.21682856487173952j), (-0.01815814374546676+0.4718839114346507j)),
                          ((-0.36336965745600625+0.3759504695343794j), (0.2248053726148399+0.22029644112859348j), (-0.09064428722611384-0.08254509314972736j), (0.12418920854260185-0.5471431985662905j)),
                          ((-0.5200354642412095+0.5425236434272582j), (0.4144201383368241-0.03221161184797661j), (0.10973679214385457+0.2887677216157485j), (0.8119344460935665-0.3782880124387231j))),
                         (((0.5132990185785542+0.1269895435922086j), (-0.8779926345719881-0.5240101078639017j), (-0.1083296281261924-0.8147529275948924j), (0.09647199912137994+0.6723852222785147j)),
                          ((0.2928885382594636-0.17353985395496885j), (-0.21364153320667634-0.03557466867344006j), (0.09421518259275086+0.6153556681629657j), (0.42459824192798623-0.17443990666881448j)),
                          ((-0.5143367641601848-0.474941950841561j), (-0.2759320986482162-0.8250929199965524j), (0.21922685784092444-0.5600860240181191j), (-0.8569179028983865-0.16482185283148332j)),
                          ((0.5268959167118139+0.55401274233894j), (0.08678495563788746-0.11915652486681427j), (-0.35416282616502737-0.4847172453257278j), (0.17111904837721448-0.8480796973331977j)))),
                        ((((0.5469697524334143+0.02787502521930041j), (0.2605342686105838-0.0738711527960102j), (-0.39955538445037275-0.37613379994879603j), (-0.013180544897028024-0.5858094274207383j)),
                          ((0.2506641890300212-0.1324021828426094j), (0.2972858012659867+0.21562443620437732j), (-0.6161716236733459-0.16509522040718683j), (-0.18102929570881354-0.6067237673008904j)), 
                          ((-0.13192637373724614-0.6510388019746922j), (-0.5546497186031663+0.15957406458683243j), (0.3605761118213213+0.25686768414528705j), (0.401508970393731-0.5108236410388888j)), 
                          ((0.39953367426110775+0.2252371577830028j), (-0.3347746193719203+0.2263747029817541j), (0.43746847580536263+0.5388066331831217j), (-0.07344964656116226+0.5473468419762271j))), 
                        (((-0.6996969040609468+0.7213293709800207j), (-0.37557411707556654+0.1506412600826108j), (0.46608440397755446+0.013185375483020345j), (0.29865953531762246+0.19232124238469206j)), 
                         ((-0.06222220564236847+0.5694069457780013j), (-0.5368962596793275+0.36703072903452505j), (-0.05182804478010006+0.3840540087750902j), (0.16375632553791997-0.4661212952339301j)), 
                         ((-0.4041070739038938-0.04037225736899819j), (0.24215413306521716-0.06631187616668377j), (-0.2663745437846935+0.611183098942953j), (0.337942638568164-0.5670011153091816j)), 
                         ((0.3876032723274968-0.08306424304240201j), (0.2173881180889634+0.4512865759999062j), (0.8795902664110721+0.8614312675002035j), (0.2870339354260544-0.11303384551216167j))), 
                        (((-0.1319661103321904-0.1378309763195904j), (-0.03263197183994404+0.03651637220786119j), (-0.33530432503994767-0.007463082947573474j), (-0.3110039705203388+0.17231544573803292j)), 
                         ((-0.07377573914613977+0.3678250464292866j), (0.3280327212275782-0.08984395132040435j), (-0.4492548090143319-0.3165726498892033j), (0.4186631970202508+0.14692250754002467j)), 
                         ((0.12300006190073354+0.8465896476419659j), (-0.3266430699658883+0.9258477806121458j), (0.1270817126276501+0.7278063130820139j), (0.04390487289582379-0.07072636756748818j)), 
                         ((0.8411822363338441+0.3271788959517308j), (0.14233232632431592-0.8034008469283107j), (-0.012057809815140041-0.07447549854466373j), (-0.5078787065469428-0.3322994351304549j))), 
                        (((0.2823845584710901+0.3169641678915194j), (-0.14926005618963878-0.06141784434383113j), (-0.3336711008317823+0.608644554614276j), (0.042858723261329-0.25353753207960283j)), 
                         ((-0.19596385250682435+0.29611929225848677j), (0.9379729841249024-0.5720585057483617j), (0.20515378855823352-0.5330821799545553j), (-0.45689594348515183-0.5221745972924456j)), 
                         ((0.005913178294852495-0.04464774171993757j), (-0.3515104653551274-0.8716563770921714j), (-0.5644652537988155+0.4413367915251234j), (-0.32684204426160013-0.6404138884316928j)), 
                         ((-0.20862357692064337-0.008586491450952471j), (0.7110044208850719-0.0260275339943824j), (0.33195949443404427-0.060726068233408204j), (-0.24054180926541135-0.22750131116864736j)))), 
                       ((((-0.19779207637655705-0.06170675636552225j), (0.24801124896429994+0.52301594517833j), (0.3359762170142643+0.006887001901350498j), (-0.5044830473838094-0.023633199563395157j)), 
                         ((0.35936388285588383-0.12044000986333014j), (0.06567318357927132+0.23081033720558908j), (-0.006063340342035706-0.43355358359223595j), (0.560747142959132-0.019735739750161296j)), 
                         ((0.16778778206380363+0.4393001381475511j), (-0.19115676223578515+0.023171844709636735j), (-0.10056678281864495+0.11175155826135519j), (-0.24878623491611496+0.8172452900981119j)), 
                         ((-0.4955653304024539-0.35864956392371916j), (-0.002728871185003978+0.3416061368260671j), (0.12387109110953831-0.2654493686440327j), (0.10889405520664741+0.1218720709206873j))), 
                        (((0.07498012788575437+0.251986782960711j), (0.3511927502553568+0.27907917288389616j), (-0.6319086362942079-0.11628342415274562j), (-0.02365411536311457+0.7887319952841338j)), 
                         ((0.38406832355686993+0.03416295519609358j), (0.35717793709754475+0.7751423080742306j), (0.29135030536626805+0.15668909755462013j), (0.14503883313378096-0.030524299930524523j)), 
                         ((0.3331132451297607-0.03848385697279688j), (0.0750244753960323+0.10077551583585709j), (-0.15490975501427928-0.1798249796504282j), (0.9066796997830194-0.6602275207883976j)), 
                         ((0.1471293723064081+0.10463425812772786j), (-0.5961028228889094+0.5480663905625878j), (0.0026089287349913004-0.5747699075005761j), (0.2899525342736272+0.1268195775198908j))), 
                        (((-0.13364076045805806-0.17688339222579297j), (0.6013848685157083+0.3493674941578545j), (0.22523028276047052-0.21939576800830496j), (0.09111222639027305+0.7894981102504295j)), 
                         ((0.6016850426953799+0.043886903791527376j), (-0.5060405108831854-0.5381151561460222j), (-0.4760569419146623+0.40666832623996363j), (-0.5318516495722876+0.16837803612551516j)), 
                         ((0.024034551466941778-0.2166311876229967j), (0.26494031063742085-0.002375780112287895j), (-0.4730046544996984+0.6644551176991773j), (-0.39157195926624855+0.6199256868216445j)), 
                         ((-0.04324867778353203-0.08087976286625231j), (0.1578599942534593-0.5156287674297576j), (-0.15005517462767193-0.09424179596026594j), (-0.13944975064956378-0.17543248499412656j))), 
                        (((-0.5648616238096438+0.46394478768979586j), (-0.41003938415951235-0.33614785565882677j), (-0.3893130130095368+0.05645367151196268j), (0.3821121473593535+0.17991382827274327j)), 
                         ((-0.5139083991747512+0.789121431720947j), (-0.6171117805874629+0.3660741433822048j), (0.05303400234375011+0.5094418060177022j), (-0.3545806168076585+0.4629656377266379j)), 
                         ((0.2658032276104465+0.4237937984173149j), (0.18183157447480658+0.06623748664233398j), (0.12546831548404336-0.10575754034359075j), (-0.6833387940577664+0.23947306785979094j)), 
                         ((-0.20503554348285624+0.21628529049971312j), (0.23649982943904524-0.6159980174662008j), (-0.20651357702330975+0.08458167450189069j), (0.5j)))))                    
                else:
                    return (((((-0.38981592692954015-0.2856117645961309j), (-0.035094151427129905-0.265178716783423j), (-0.34532231395177504+0.36673987839501265j), (0.2041672227930209-0.6128500993893164j)),
                          ((-0.12421070955162938+0.026349493748204833j), (-0.3415725798389704-0.0775430038163134j), (0.17048268853378434-0.08862480715498033j), (-0.1106967683514759-0.10379778205326462j)),
                          ((0.7298716846009465+0.20776829436005195j), (-0.33435577984609854+0.039072521387463754j), (-0.05886662990481739-0.05072122421891534j), (0.18354830862254912+0.4606377888934907j)),
                          ((-0.0767806399525307-0.8221389203690308j), (0.190835818629008+0.16192941104126957j), (0.17778866234180712+0.2584570168682756j), (0.07671782176673747-0.8469237188822869j))),
                         (((-0.35597391704713033+0.25043169560436107j), (0.41228040156904044-0.3258644927079958j), (0.3255845547364652-0.04129891413069664j), (-0.49623171650577047-0.28943245984704047j)),
                          ((-0.29908631455896795+0.32223868604261874j), (0.22549183711675158-0.07473853585711632j), (0.8473949873724411-0.1266457939975455j), (-0.28218966494496867-0.7447571699002506j)),
                          ((0.43626155573856107-0.22452973659230602j), (-0.13852005024639502-0.2549763508471894j), (0.40223991129540115+0.6286067642439117j), (0.05222020648272985+0.36124142226658207j)),
                          ((-0.5521041254588972+0.04526366394953618j), (0.028005364473586924+0.664136776905772j), (0.19982797953123066+0.6970088780138949j), (-0.27592458615859916+0.05358189338835073j))),
                         (((-0.01401304893255595-0.2592802467152746j), (0.38457288044239724+0.022898781579736238j), (0.11747849077927963+0.02982853498608895j), (0.43510885272308897-0.5971898990476139j)),
                          ((-0.1386049812442307+0.48221564332109246j), (0.12985640511010177-0.46733815759024744j), (0.36035270943159226+0.030123486580092673j), (0.13433570628907954-0.3548300324692315j)),
                          ((0.3630171981969085-0.10489930496648125j), (-0.19285139550925268-0.22152540990095826j), (0.067448080460355-0.22485851852233463j), (-0.22642725802173436-0.8030852201557207j)),
                          ((-0.4923572157490467+0.6941947968511506j), (-0.3083492643827028-0.21029535607801164j), (0.04516826032767807+0.16347006444443446j), (-0.4979722425636467+0.2600475135508704j))),
                         (((-0.9146464795838944-0.4811995164563062j), (0.12372743851535117-0.19857635465689238j), (-0.32654984398293085+0.1399970385669752j), (0.5030023710839411-0.15390603274578885j)),
                          ((0.12140042439229626-0.1677658216785154j), (0.15883782681413416+0.33379103634910656j), (0.185779713490287-0.3870952578769751j), (-0.06711575082420218+0.06791735335184867j)),
                          ((-0.39839017930339415-0.5701893857302205j), (0.517636015942468+0.2847028688489578j), (0.3792093531943226+0.6747299891837335j), (0.3790566911887919-0.5890859768662702j)),
                          ((0.10419770368967918-0.5497491933319889j), (0.037156668670760284-0.08813180251072528j), (0.3918645219248724+0.294242167456119j), (0.46981783997943394-0.1336764441655952j)))),
                        ((((0.033498149202913496+0.5105612554548389j), (-0.00671182242426005-0.34435078880309433j), (-0.0373284784284601-0.5412680712938157j), (-0.21266128835217923+0.7435692921379757j)),
                          ((-0.6338796544785741-0.1519302355817751j), (0.19056981672575435-0.2151423521624617j), (-0.2906156104811447-0.06976477293942895j), (0.11015974132120787+0.034078789456797876j)),
                          ((-0.802973040360448+0.25982604363200634j), (0.4470120196869066+0.1070242653063358j), (-0.17139489854864864+0.280001955020894j), (0.481238612220953-0.1754485899809457j)),
                          ((-0.2395451407327337+0.17708193346527168j), (-0.5161417899394428-0.1119365768848799j), (0.48501626047241986+0.14129638224677343j), (-0.8906342486796647+0.5959903703773012j))),
                         (((0.4573784894991153+0.5594852951634348j), (0.5713680069309577-0.5494425630562785j), (0.08288855869575895-0.46808758933789407j), (0.0881202973963836+0.030710301266941187j)),
                          ((-0.06642229051907478-0.22323447873665903j), (-0.7628208743574287-0.6371785162956514j), (-0.2246568157836294+0.3567254755305702j), (0.68327814761898-0.27973991174997725j)),
                          ((0.2784159984258674-0.17498289569648884j), (0.004730953615886158+0.07715287138774296j), (0.0773670863004613-0.6360012413426752j), (-0.0647551047878423+0.6906328131396173j)),
                          ((0.558480058973719-0.04481293285577537j), (0.09684530531415203-0.40763537195877164j), (-0.3913240230298858-0.004438441098584933j), (0.031510290166036015-0.30876711061254836j))),
                         (((-0.588734853251181+0.36452828173265184j), (-0.19810928581934018+0.20823750547135322j), (-0.45274761144312026-0.17983105119023213j), (-0.11229734410645587+0.2560770621269597j)),
                          ((-0.4038055786345742+0.7025954299708754j), (0.6537842056064525-0.34973368012330475j), (0.35464537317028266-0.21682856487173952j), (-0.01815814374546676+0.4718839114346507j)),
                          ((-0.36336965745600625+0.3759504695343794j), (0.2248053726148399+0.22029644112859348j), (-0.09064428722611384-0.08254509314972736j), (0.12418920854260185-0.5471431985662905j)),
                          ((-0.5200354642412095+0.5425236434272582j), (0.4144201383368241-0.03221161184797661j), (0.10973679214385457+0.2887677216157485j), (0.8119344460935665-0.3782880124387231j))),
                         (((0.5132990185785542+0.1269895435922086j), (-0.8779926345719881-0.5240101078639017j), (-0.1083296281261924-0.8147529275948924j), (0.09647199912137994+0.6723852222785147j)),
                          ((0.2928885382594636-0.17353985395496885j), (-0.21364153320667634-0.03557466867344006j), (0.09421518259275086+0.6153556681629657j), (0.42459824192798623-0.17443990666881448j)),
                          ((-0.5143367641601848-0.474941950841561j), (-0.2759320986482162-0.8250929199965524j), (0.21922685784092444-0.5600860240181191j), (-0.8569179028983865-0.16482185283148332j)),
                          ((0.5268959167118139+0.55401274233894j), (0.08678495563788746-0.11915652486681427j), (-0.35416282616502737-0.4847172453257278j), (0.17111904837721448-0.8480796973331977j)))),
                        ((((0.5469697524334143+0.02787502521930041j), (0.2605342686105838-0.0738711527960102j), (-0.39955538445037275-0.37613379994879603j), (-0.013180544897028024-0.5858094274207383j)),
                          ((0.2506641890300212-0.1324021828426094j), (0.2972858012659867+0.21562443620437732j), (-0.6161716236733459-0.16509522040718683j), (-0.18102929570881354-0.6067237673008904j)), 
                          ((-0.13192637373724614-0.6510388019746922j), (-0.5546497186031663+0.15957406458683243j), (0.3605761118213213+0.25686768414528705j), (0.401508970393731-0.5108236410388888j)), 
                          ((0.39953367426110775+0.2252371577830028j), (-0.3347746193719203+0.2263747029817541j), (0.43746847580536263+0.5388066331831217j), (-0.07344964656116226+0.5473468419762271j))), 
                        (((-0.6996969040609468+0.7213293709800207j), (-0.37557411707556654+0.1506412600826108j), (0.46608440397755446+0.013185375483020345j), (0.29865953531762246+0.19232124238469206j)), 
                         ((-0.06222220564236847+0.5694069457780013j), (-0.5368962596793275+0.36703072903452505j), (-0.05182804478010006+0.3840540087750902j), (0.16375632553791997-0.4661212952339301j)), 
                         ((-0.4041070739038938-0.04037225736899819j), (0.24215413306521716-0.06631187616668377j), (-0.2663745437846935+0.611183098942953j), (0.337942638568164-0.5670011153091816j)), 
                         ((0.3876032723274968-0.08306424304240201j), (0.2173881180889634+0.4512865759999062j), (0.8795902664110721+0.8614312675002035j), (0.2870339354260544-0.11303384551216167j))), 
                        (((-0.1319661103321904-0.1378309763195904j), (-0.03263197183994404+0.03651637220786119j), (-0.33530432503994767-0.007463082947573474j), (-0.3110039705203388+0.17231544573803292j)), 
                         ((-0.07377573914613977+0.3678250464292866j), (0.3280327212275782-0.08984395132040435j), (-0.4492548090143319-0.3165726498892033j), (0.4186631970202508+0.14692250754002467j)), 
                         ((0.12300006190073354+0.8465896476419659j), (-0.3266430699658883+0.9258477806121458j), (0.1270817126276501+0.7278063130820139j), (0.04390487289582379-0.07072636756748818j)), 
                         ((0.8411822363338441+0.3271788959517308j), (0.14233232632431592-0.8034008469283107j), (-0.012057809815140041-0.07447549854466373j), (-0.5078787065469428-0.3322994351304549j))), 
                        (((0.2823845584710901+0.3169641678915194j), (-0.14926005618963878-0.06141784434383113j), (-0.3336711008317823+0.608644554614276j), (0.042858723261329-0.25353753207960283j)), 
                         ((-0.19596385250682435+0.29611929225848677j), (0.9379729841249024-0.5720585057483617j), (0.20515378855823352-0.5330821799545553j), (-0.45689594348515183-0.5221745972924456j)), 
                         ((0.005913178294852495-0.04464774171993757j), (-0.3515104653551274-0.8716563770921714j), (-0.5644652537988155+0.4413367915251234j), (-0.32684204426160013-0.6404138884316928j)), 
                         ((-0.20862357692064337-0.008586491450952471j), (0.7110044208850719-0.0260275339943824j), (0.33195949443404427-0.060726068233408204j), (-0.24054180926541135-0.22750131116864736j)))), 
                       ((((-0.19779207637655705-0.06170675636552225j), (0.24801124896429994+0.52301594517833j), (0.3359762170142643+0.006887001901350498j), (-0.5044830473838094-0.023633199563395157j)), 
                         ((0.35936388285588383-0.12044000986333014j), (0.06567318357927132+0.23081033720558908j), (-0.006063340342035706-0.43355358359223595j), (0.560747142959132-0.019735739750161296j)), 
                         ((0.16778778206380363+0.4393001381475511j), (-0.19115676223578515+0.023171844709636735j), (-0.10056678281864495+0.11175155826135519j), (-0.24878623491611496+0.8172452900981119j)), 
                         ((-0.4955653304024539-0.35864956392371916j), (-0.002728871185003978+0.3416061368260671j), (0.12387109110953831-0.2654493686440327j), (0.10889405520664741+0.1218720709206873j))), 
                        (((0.07498012788575437+0.251986782960711j), (0.3511927502553568+0.27907917288389616j), (-0.6319086362942079-0.11628342415274562j), (-0.02365411536311457+0.7887319952841338j)), 
                         ((0.38406832355686993+0.03416295519609358j), (0.35717793709754475+0.7751423080742306j), (0.29135030536626805+0.15668909755462013j), (0.14503883313378096-0.030524299930524523j)), 
                         ((0.3331132451297607-0.03848385697279688j), (0.0750244753960323+0.10077551583585709j), (-0.15490975501427928-0.1798249796504282j), (0.9066796997830194-0.6602275207883976j)), 
                         ((0.1471293723064081+0.10463425812772786j), (-0.5961028228889094+0.5480663905625878j), (0.0026089287349913004-0.5747699075005761j), (0.2899525342736272+0.1268195775198908j))), 
                        (((-0.13364076045805806-0.17688339222579297j), (0.6013848685157083+0.3493674941578545j), (0.22523028276047052-0.21939576800830496j), (0.09111222639027305+0.7894981102504295j)), 
                         ((0.6016850426953799+0.043886903791527376j), (-0.5060405108831854-0.5381151561460222j), (-0.4760569419146623+0.40666832623996363j), (-0.5318516495722876+0.16837803612551516j)), 
                         ((0.024034551466941778-0.2166311876229967j), (0.26494031063742085-0.002375780112287895j), (-0.4730046544996984+0.6644551176991773j), (-0.39157195926624855+0.6199256868216445j)), 
                         ((-0.04324867778353203-0.08087976286625231j), (0.1578599942534593-0.5156287674297576j), (-0.15005517462767193-0.09424179596026594j), (-0.13944975064956378-0.17543248499412656j))), 
                        (((-0.5648616238096438+0.46394478768979586j), (-0.41003938415951235-0.33614785565882677j), (-0.3893130130095368+0.05645367151196268j), (0.3821121473593535+0.17991382827274327j)), 
                         ((-0.5139083991747512+0.789121431720947j), (-0.6171117805874629+0.3660741433822048j), (0.05303400234375011+0.5094418060177022j), (-0.3545806168076585+0.4629656377266379j)), 
                         ((0.2658032276104465+0.4237937984173149j), (0.18183157447480658+0.06623748664233398j), (0.12546831548404336-0.10575754034359075j), (-0.6833387940577664+0.23947306785979094j)), 
                         ((-0.20503554348285624+0.21628529049971312j), (0.23649982943904524-0.6159980174662008j), (-0.20651357702330975+0.08458167450189069j), (0.j)))))

    def get_python_input3(self, rank, cplx, no_first_arg_negative=False, no_zeros=False):
        if not cplx:
            if rank==0:
                return (0.37126864363799594,)
            elif rank==1:
                return (0.6234908821535466, 0.6138798448505429)
            elif rank==2:
                return ((0.46458697075264, 0.47521353725342), (0.2769991834092879, 0.7655903776259643))
            elif rank==3:
                if no_first_arg_negative:
                    return (((0.7087306338233754, 0.5091993546185136, 0.7823132790082358), 
                        (0.7809687239325918, 0.7618301393180559, 0.13846012311583666), 
                        (0.24584408982123662, 0.30175410391822816, 0.6508820239463049)),
                        ((0.0799357328507963, 0.4268164046934606, 0.3340669557843154), 
                        (0.5352665915599666, 0.4796035531201281, 0.31521200922735726),
                        (0.6181927127971525, 0.5813500010909463, 0.6485259530707308)),
                        ((0.607591622470962, 0.5134390179382666, 0.7761053727330762), 
                        (0.30315911281007185, 0.7772739725828741, 0.21981237010097954), 
                        (0.5720780258224234, 0.7199923733581466, 0.8038912730779739)))
                else:
                    return (((-0.7087306338233754, 0.5091993546185136, 0.7823132790082358), 
                        (0.7809687239325918, -0.7618301393180559, 0.13846012311583666), 
                        (0.24584408982123662, -0.30175410391822816, -0.6508820239463049)),
                        ((0.0799357328507963, 0.4268164046934606, -0.3340669557843154), 
                        (-0.5352665915599666, -0.4796035531201281, -0.31521200922735726),
                        (0.6181927127971525, 0.5813500010909463, 0.6485259530707308)),
                        ((-0.607591622470962, 0.5134390179382666, -0.7761053727330762), 
                        (0.30315911281007185, -0.7772739725828741, 0.21981237010097954), 
                        (0.5720780258224234, -0.7199923733581466, -0.8038912730779739)))
            elif rank==4:
                if no_first_arg_negative:
                    return ((((0.45368972429803434, 0.43658224238126836, 0.327332664787118, 0.1659399779578305), 
                        (0.5908203393014015, 0.6287596060266886, 0.26557881193403204, 0.7640693855123928), 
                        (0.6550472925659366, 0.4051939492486867, 0.39695234748477226, 0.7984350907853707), 
                        (0.45152587205177463, 0.6915126698433964, 0.1408991785923076, 0.5555466803377823)), 
                        ((0.7111217974645967, 0.22442551434498154, 0.33601178864915354, 0.21277504641255832),
                        (0.10220061513651632, 0.4523774178793611, 0.4079018667540423, 0.909554257065248), 
                        (0.6139494095517375, 0.3083569476979329, 0.6376629339543685, 0.28647430781873456), 
                        (0.36486444793373635, 0.3134770531471723, 0.19207850462685053, 0.2742209367740796)), 
                        ((0.5206089841634276, 0.24952538894537638, 0.0933758645493489, 0.24304014487798284), 
                        (0.2039225828898289, 0.423378042043955, 0.21847415387885974, 0.4823494123271024), 
                        (0.49781756385165377, 0.42765522168555026, 0.4737414300227722, 0.5398374124738391), 
                        (0.2640617374368053, 0.34880253282719476, 0.32357380359477705, 0.7111559522372015)), 
                        ((0.23761918672652063, 0.1593578505403827, 0.6530063992033186, 0.9079894778799892),
                        (0.8250545394400823, 0.4861804550261372, 0.3406365725750848, 0.42480466462889144),
                        (0.47203132507512685, 0.20818156589620632, 0.20161698569028383, 0.42806013456132413),
                        (0.7792675299565863, 0.3736959703075082, 0.19761992623295196, 0.4235372420916027))), 
                        (((0.6459817009885405, 0.2610419659851719, 0.39888588458573665, 0.6358114313716404),
                        (0.6789108524310596, 0.4760144991501489, 0.5132407231875976, 0.7445121079875354), 
                        (0.3165090047475903, 0.1882290433512482, 0.30270840337500193, 0.4773827304127968), 
                        (0.27248041023839426, 0.5356669773887297, 0.5759298824759524, 0.27026295067501754)), 
                        ((0.2225815420544881, 0.6296933698530776, 0.6420994781708316, 0.5292910401325328),
                        (0.4838814244753861, 0.3015763358329324, 0.5966105067811018, 0.437555952383517),
                        (0.2508886824465943, 0.3688703869280175, 0.44848941714873286, 0.6968286221703255), 
                        (0.7458127375857659, 0.5565367320025696, 0.5221639609134333, 0.5043229193744858)), 
                        ((0.2704586652152375, 0.23622061144754075, 0.5668874045788246, 0.7383427762361463),
                        (0.3326338826927772, 0.5180692401299339, 0.6853188934444896, 0.5105611442225273),
                        (0.32729714889258044, 0.7634733351662765, 0.2667323956085834, 0.3813499428619923),
                        (0.707758750914661, 0.23317899979615478, 0.6728060312412778, 0.5045011288076484)),
                        ((0.2758755420358796, 0.6849112850328907, 0.7208465949939953, 0.5465246847493163),
                        (0.5382081700827642, 0.4626841039902129, 0.38537277159039474, 0.5918003106653552),
                        (0.7410792569612943, 0.848695963275266, 0.35215371840141596, 0.7859807920638692),
                        (0.439754658119384, 0.09885040884202281, 0.20324265625602897, 0.6393707779383961))), 
                        (((0.45932643067605416, 0.4867064483724215, 0.2492845750953945, 0.4071869028109893),
                        (0.518352199566693, 0.6265341328104177, 0.5719047743470251, 0.8284284690707643),
                        (0.1255527780884896, 0.133522460155736, 0.5939516968011829, 0.6970674969748756),
                        (0.821986273583169, 0.3881563859122813, 0.16941736964779366, 0.288738726812057)),
                        ((0.48102730273877786, 0.1522385344365771, 0.5752150637264031, 0.39522973755221524), 
                        (0.732520712427958, 0.49190588963815157, 0.6125760080831428, 0.2716702847832281), 
                        (0.8159103581398756, 0.34507912842082983, 0.7075657623779936, 0.29530894172611916), 
                        (0.5278442353318813, 0.6539972004065213, 0.48897529954240776, 0.7932403888940815)),
                        ((0.831358889394934, 0.48050345824229945, 0.6410388202892018, 0.8982570057418652), 
                        (0.30388123939085465, 0.727912073121657, 0.543834760379716, 0.6710456332208439), 
                        (0.4417259323277584, 0.5717441377950038, 0.3664067245442036, 0.3280925201821242), 
                        (0.5400907637568426, 0.43207844188064887, 0.732946675152238, 0.5339092194505695)),
                        ((0.22369526225631464, 0.5665529520090573, 0.2653373517286637, 0.6914637769007175),
                        (0.22138531453529892, 0.17362665833183274, 0.11873401018627547, 0.48731998406078203),
                        (0.5508721228465998, 0.3492961556670177, 0.5692351404720578, 0.5107492266892075), 
                        (0.17584863513461835, 0.35755945941346623, 0.7650117785966029, 0.48316674832514844))), 
                        (((0.2982202413301988, 0.2348830652298031, 0.3114442699257422, 0.20218169729319738), 
                        (0.6846710473822368, 0.8119003553529898, 0.1790804504032915, 0.4781252356617314), 
                        (0.3766941501672641, 0.5437027703705374, 0.24423050827666176, 0.3323763024031741), 
                        (0.8189980567608663, 0.5888920746029481, 0.17138276940268463, 0.5475256795584624)), 
                        ((0.4569907243510988, 0.5255497422421722, 0.1680676291815795, 0.2879601796868399), 
                        (0.5419471765149858, 0.8765158329632898, 0.21847500231364173, 0.6219663103942495),
                        (0.8993300494227338, 0.5256551743041176, 0.4312579142909404, 0.28503463714017396),
                        (0.6601304123050094, 0.9681225089321629, 0.6110831795473397, 0.630733896684101)),
                        ((0.7252549854189686, 0.48905018110325815, 0.3949054499635516, 0.6883221515509466), 
                        (0.3510020242244332, 0.5964593467774927, 0.665927090247577, 0.19851750424804993),
                        (0.49702295298466764, 0.32952301722148614, 0.6578112215358343, 0.629875436683738), 
                        (0.5228582859488041, 0.48834135369831744, 0.8107618558028835, 0.012956020032133924)), 
                        ((0.3966015859408982, 0.19901168720580728, 0.6930890007374754, 0.7839579709117346), 
                        (0.8623235889093344, 0.5661301336086426, 0.6547115325958021, 0.6689861941430038),
                        (0.5105113993916237, 0.7066840336625683, 0.8004786600943815, 0.5266845579213677),
                        (0.689553296868146, 0.773301352126767, 0.7364812118860565, 0.5377502245784462))))                    
                else:
                    return ((((0.45368972429803434, 0.43658224238126836, 0.327332664787118, 0.1659399779578305), 
                        (0.5908203393014015, 0.6287596060266886, 0.26557881193403204, 0.7640693855123928), 
                        (0.6550472925659366, 0.4051939492486867, 0.39695234748477226, 0.7984350907853707), 
                        (0.45152587205177463, -0.6915126698433964, -0.1408991785923076, 0.5555466803377823)), 
                        ((-0.7111217974645967, 0.22442551434498154, 0.33601178864915354, 0.21277504641255832),
                        (0.10220061513651632, 0.4523774178793611, 0.4079018667540423, -0.909554257065248), 
                        (0.6139494095517375, 0.3083569476979329, 0.6376629339543685, -0.28647430781873456), 
                        (-0.36486444793373635, 0.3134770531471723, 0.19207850462685053, 0.2742209367740796)), 
                        ((0.5206089841634276, 0.24952538894537638, 0.0933758645493489, 0.24304014487798284), 
                        (0.2039225828898289, 0.423378042043955, 0.21847415387885974, 0.4823494123271024), 
                        (0.49781756385165377, 0.42765522168555026, -0.4737414300227722, -0.5398374124738391), 
                        (0.2640617374368053, 0.34880253282719476, 0.32357380359477705, -0.7111559522372015)), 
                        ((0.23761918672652063, 0.1593578505403827, 0.6530063992033186, 0.9079894778799892),
                        (-0.8250545394400823, 0.4861804550261372, -0.3406365725750848, 0.42480466462889144),
                        (0.47203132507512685, 0.20818156589620632, 0.20161698569028383, 0.42806013456132413),
                        (0.7792675299565863, 0.3736959703075082, 0.19761992623295196, 0.4235372420916027))), 
                        (((0.6459817009885405, 0.2610419659851719, 0.39888588458573665, 0.6358114313716404),
                        (0.6789108524310596, 0.4760144991501489, 0.5132407231875976, -0.7445121079875354), 
                        (-0.3165090047475903, 0.1882290433512482, -0.30270840337500193, 0.4773827304127968), 
                        (0.27248041023839426, 0.5356669773887297, -0.5759298824759524, 0.27026295067501754)), 
                        ((0.2225815420544881, 0.6296933698530776, 0.6420994781708316, 0.5292910401325328),
                        (0.4838814244753861, 0.3015763358329324, 0.5966105067811018, 0.437555952383517),
                        (0.2508886824465943, -0.3688703869280175, 0.44848941714873286, 0.6968286221703255), 
                        (0.7458127375857659, 0.5565367320025696, -0.5221639609134333, -0.5043229193744858)), 
                        ((0.2704586652152375, 0.23622061144754075, 0.5668874045788246, -0.7383427762361463),
                        (0.3326338826927772, -0.5180692401299339, 0.6853188934444896, -0.5105611442225273),
                        (0.32729714889258044, 0.7634733351662765, 0.2667323956085834, 0.3813499428619923),
                        (-0.707758750914661, 0.23317899979615478, -0.6728060312412778, -0.5045011288076484)),
                        ((-0.2758755420358796, -0.6849112850328907, 0.7208465949939953, 0.5465246847493163),
                        (0.5382081700827642, 0.4626841039902129, 0.38537277159039474, -0.5918003106653552),
                        (0.7410792569612943, -0.848695963275266, -0.35215371840141596, 0.7859807920638692),
                        (0.439754658119384, -0.09885040884202281, -0.20324265625602897, 0.6393707779383961))), 
                        (((0.45932643067605416, 0.4867064483724215, -0.2492845750953945, 0.4071869028109893),
                        (0.518352199566693, 0.6265341328104177, 0.5719047743470251, -0.8284284690707643),
                        (0.1255527780884896, -0.133522460155736, 0.5939516968011829, 0.6970674969748756),
                        (0.821986273583169, 0.3881563859122813, 0.16941736964779366, -0.288738726812057)),
                        ((0.48102730273877786, 0.1522385344365771, 0.5752150637264031, 0.39522973755221524), 
                        (-0.732520712427958, -0.49190588963815157, 0.6125760080831428, 0.2716702847832281), 
                        (-0.8159103581398756, 0.34507912842082983, 0.7075657623779936, 0.29530894172611916), 
                        (0.5278442353318813, -0.6539972004065213, -0.48897529954240776, 0.7932403888940815)),
                        ((0.831358889394934, 0.48050345824229945, -0.6410388202892018, 0.8982570057418652), 
                        (0.30388123939085465, 0.727912073121657, -0.543834760379716, -0.6710456332208439), 
                        (0.4417259323277584, -0.5717441377950038, 0.3664067245442036, 0.3280925201821242), 
                        (-0.5400907637568426, 0.43207844188064887, -0.732946675152238, -0.5339092194505695)),
                        ((-0.22369526225631464, 0.5665529520090573, 0.2653373517286637, 0.6914637769007175),
                        (0.22138531453529892, 0.17362665833183274, 0.11873401018627547, 0.48731998406078203),
                        (0.5508721228465998, -0.3492961556670177, -0.5692351404720578, 0.5107492266892075), 
                        (0.17584863513461835, -0.35755945941346623, 0.7650117785966029, 0.48316674832514844))), 
                        (((0.2982202413301988, 0.2348830652298031, 0.3114442699257422, -0.20218169729319738), 
                        (0.6846710473822368, -0.8119003553529898, 0.1790804504032915, 0.4781252356617314), 
                        (-0.3766941501672641, 0.5437027703705374, -0.24423050827666176, 0.3323763024031741), 
                        (0.8189980567608663, -0.5888920746029481, 0.17138276940268463, 0.5475256795584624)), 
                        ((0.4569907243510988, 0.5255497422421722, -0.1680676291815795, -0.2879601796868399), 
                        (0.5419471765149858, 0.8765158329632898, -0.21847500231364173, 0.6219663103942495),
                        (0.8993300494227338, 0.5256551743041176, 0.4312579142909404, -0.28503463714017396),
                        (-0.6601304123050094, 0.9681225089321629, 0.6110831795473397, 0.630733896684101)),
                        ((0.7252549854189686, -0.48905018110325815, 0.3949054499635516, 0.6883221515509466), 
                        (0.3510020242244332, 0.5964593467774927, -0.665927090247577, -0.19851750424804993),
                        (0.49702295298466764, -0.32952301722148614, 0.6578112215358343, 0.629875436683738), 
                        (0.5228582859488041, 0.48834135369831744, -0.8107618558028835, 0.012956020032133924)), 
                        ((-0.3966015859408982, 0.19901168720580728, 0.6930890007374754, 0.7839579709117346), 
                        (0.8623235889093344, 0.5661301336086426, -0.6547115325958021, -0.6689861941430038),
                        (-0.5105113993916237, 0.7066840336625683, 0.8004786600943815, 0.5266845579213677),
                        (0.689553296868146, -0.773301352126767, -0.7364812118860565, 0.5377502245784462))))
        else:
            if rank==0:
                return (0.37126864363799594j,)
            elif rank==1:
                return (-0.6234908821535466, -0.6138798448505429j)
            elif rank==2:
                return ((0.46458697075264j, 0.47521353725342), (-0.2769991834092879j, 0.7655903776259643))
            elif rank==3:
                return ((((0.1454277130999836-0.7320274919853249j), (0.6111728482069096+0.3264747748994178j), (-0.5638648912195857-0.17450334170489668j)),
                         ((0.36745011107568193-0.36335598004419434j), (0.17683177117302507+0.17980667188917787j), (0.1975971229124449+0.4658291218212517j)),
                         ((-0.2014605099933947+0.2948257187520392j), (-0.744292187546457-0.06353546214852701j), (0.5315307768500955+0.20979695600906234j))),
                        (((0.6658801836076347-0.5765177344716341j), (0.49015610705744417-0.5133880244308395j), (-0.1502312930208003+0.1561733784845012j)),
                         ((-0.661348002857287-0.029644750992120383j), (0.006584637699976192-0.4781943559540154j), (0.12266084950784173+0.46542904661542706j)),
                         ((-0.41697557600280255-0.3537461357888442j), (0.26272750067625794-0.43630595153076535j), (-0.21367481541695688-0.2618183771588106j))),
                        (((-0.22338538892626414-0.10818049912878436j), (-0.040383727106552136+0.48764341091162444j), (0.014766392396017403+0.8223651913211714j)),
                         ((0.003228290936111433-0.37610619147220625j), (-0.01103811298925661-0.39851348234329986j), (0.20163130848248234-0.49523755760391885j)),
                         ((0.37812723691663375+0.354147330172629j), (0.08116710042198705+0.1643297763093502j), (0.015047687899757523-0.05023160033399521j))))
            elif rank==4:
                return ((((0.45368972429803434, 0.43658224238126836j, 0.327332664787118, 0.1659399779578305), 
                        (0.5908203393014015, 0.6287596060266886, 0.26557881193403204, 0.7640693855123928), 
                        (0.6550472925659366, 0.4051939492486867, 0.39695234748477226, 0.7984350907853707), 
                        (0.45152587205177463, 0.6915126698433964, 0.1408991785923076, 0.5555466803377823)), 
                        ((0.7111217974645967, -0.224425514+0.12j, -0.336011788-0.35j, 0.21277504641255832),
                        (0.10220061513651632, 0.4523774178793611, 0.4079018667540423, 0.909554257065248), 
                        (0.6139494095517375, 0.3083569476979329, 0.6376629339543685, 0.28647430781873456), 
                        (0.36486444793373635, -0.31347705-0.8765j, 0.19207850462685053, 0.2742209367740796)), 
                        ((0.5206089841634276, 0.24952538894537638, 0.0933758645493489, 0.24304014487798284), 
                        (-0.2039225828898289, 0.423378042043955, 0.21847415387885974, 0.4823494123271024), 
                        (0.49781756385165377, 0.42765522168555026j, 0.4737414300227722, 0.5398374124738391), 
                        (0.2640617374368053, 0.34880253282719476, 0.32357380359477705, 0.7111559522372015)), 
                        ((0.23761918672652063, -0.1593578505403827, 0.6530063992033186, 0.9079894778799892),
                        (0.8250545394400823, 0.4861804550261372, 0.3406365725750848, 0.42480466462889144),
                        (0.47203132507512685, 0.20818156589620632, 0.20161698569028383, 0.42806013456132413),
                        (0.7792675299565863, -0.3736959703075082, -0.19761992623295196, 0.4235372420916027))), 
                        (((0.6459817009885405, 0.2610419659851719, -0.39888588458573665, 0.6358114313716404),
                        (0.6789108524310596, 0.4760144991501489, 0.5132407231875976, 0.7445121079875354), 
                        (0.3165090047475903, 0.1882290433512482, 0.30270840337500193, 0.4773827304127968), 
                        (0.27248041023839426, 0.5356669773887297, 0.5759298824759524, 0.27026295067501754)), 
                        ((-0.2225815420544881, 0.6296933698530776, 0.6420994781708316, 0.5292910401325328),
                        (0.4838814244753861, 0.3015763358329324, 0.5966105067811018, 0.437555952383517),
                        (0.2508886824465943, 0.3688703869280175j, 0.44848941714873286, 0.6968286221703255), 
                        (0.7458127375857659, 0.5565367320025696, 0.5221639609134333, 0.5043229193744858)), 
                        ((0.2704586652152375, 0.23622061144754075, 0.5668874045788246, 0.7383427762361463),
                        (0.3326338826927772, 0.5180692401299339, -0.6853188934444896, 0.5105611442225273),
                        (0.32729714889258044, 0.7634733351662765, 0.2667323956085834, 0.3813499428619923),
                        (0.707758750914661, 0.23317899979615478, 0.6728060312412778, 0.5045011288076484)),
                        ((0.2758755420358796, 0.6849112850328907, 0.7208465949939953, 0.5465246847493163),
                        (0.5382081700827642, 0.4626841039902129, 0.38537277159039474, 0.5918003106653552),
                        (0.7410792569612943, 0.848695963275266, 0.35215371840141596, 0.7859807920638692),
                        (0.439754658119384, 0.09885040884202281, 0.20324265625602897, 0.6393707779383961))), 
                        (((0.45932643067605416, 0.4867064483724215, 0.2492845750953945, 0.4071869028109893),
                        (0.518352199566693, 0.6265341328104177, 0.5719047743470251, 0.8284284690707643),
                        (0.1255527780884896, 0.133522460155736, 0.5939516968011829, 0.6970674969748756),
                        (0.821986273583169, 0.3881563859122813, 0.16941736964779366, 0.288738726812057)),
                        ((0.48102730273877786, 0.1522385344365771, 0.5752150637264031, 0.39522973755221524), 
                        (0.732520712427958, 0.49190588963815157, 0.6125760080831428, 0.2716702847832281), 
                        (0.8159103581398756, 0.34507912842082983, 0.7075657623779936, 0.29530894172611916), 
                        (0.5278442353318813, 0.6539972004065213, 0.48897529954240776, 0.7932403888940815)),
                        ((0.831358889394934, 0.48050345824229945, 0.6410388202892018, 0.8982570057418652), 
                        (0.30388123939085465, 0.727912073121657, 0.543834760379716, 0.6710456332208439), 
                        (0.4417259323277584, 0.5717441377950038, 0.3664067245442036, 0.3280925201821242), 
                        (0.5400907637568426, 0.43207844188064887, 0.732946675152238, 0.5339092194505695)),
                        ((0.22369526225631464, 0.5665529520090573, 0.2653373517286637, 0.6914637769007175),
                        (0.22138531453529892, 0.17362665833183274, 0.11873401018627547, 0.48731998406078203),
                        (0.5508721228465998, 0.3492961556670177, 0.5692351404720578, 0.5107492266892075), 
                        (0.17584863513461835, 0.35755945941346623, 0.7650117785966029, 0.48316674832514844))), 
                        (((0.2982202413301988, 0.2348830652298031, 0.3114442699257422, 0.20218169729319738), 
                        (0.6846710473822368, 0.8119003553529898, 0.1790804504032915, 0.4781252356617314), 
                        (0.3766941501672641, 0.5437027703705374, 0.24423050827666176, 0.3323763024031741), 
                        (0.8189980567608663, 0.5888920746029481, 0.17138276940268463, 0.5475256795584624)), 
                        ((0.4569907243510988, 0.5255497422421722, 0.1680676291815795, 0.2879601796868399), 
                        (0.5419471765149858, 0.8765158329632898, 0.21847500231364173, 0.6219663103942495),
                        (0.8993300494227338, 0.5256551743041176, 0.4312579142909404, 0.28503463714017396),
                        (0.6601304123050094, 0.9681225089321629, 0.6110831795473397, 0.630733896684101)),
                        ((0.7252549854189686, 0.48905018110325815, 0.3949054499635516, 0.6883221515509466), 
                        (0.3510020242244332, 0.5964593467774927, 0.665927090247577, 0.19851750424804993),
                        (0.49702295298466764, 0.32952301722148614, 0.6578112215358343, 0.629875436683738), 
                        (0.5228582859488041, 0.48834135369831744, 0.8107618558028835, 0.012956020032133924)), 
                        ((0.3966015859408982, 0.19901168720580728, 0.6930890007374754, 0.7839579709117346), 
                        (0.8623235889093344, 0.5661301336086426, 0.6547115325958021, 0.6689861941430038),
                        (0.5105113993916237, 0.7066840336625683, 0.8004786600943815, 0.5266845579213677),
                        (0.689553296868146j, 0.773301352126767j, 0.7364812118860565j, 0.5377502245784462))))            


    def get_python_input4(self, rank, cplx):
        if not cplx:
            if rank==0:
                return (0.9153151043823012,)
            elif rank==1:
                return (-0.5162647246320071, -0.06342374260616546)
            elif rank==2:
                return ((0.7011839736500342, 0.625639950892529), (-0.17902320165992758, -0.049498389409131205))
            elif rank==3:
                return (((0.008309531806355519, 0.22817074073669497, 0.7216964601946423), 
                         (0.33084432155377397, -0.48472374994936485, 0.4116921963539759), 
                         (0.0257088350105531, -0.2370277105296692, 0.3798801329452163)), 
                        ((0.01734050560720546, 0.16286702001136155, -0.6501064087150703), 
                         (0.02818311904325488, -0.4169901835040347, 0.4086136135582773), 
                         (0.5627838019596704, -0.16279418267275847, 0.581645077262897)), 
                        ((0.6948207132066607, 0.23541367609085884, -0.4228977230091847), 
                         (0.3216823894305869, 0.37129093662137924, -0.7134833270353387), 
                         (0.710538786928389, -0.08441890458925416, -0.458946120570603)))
            elif rank==4:((((0.37235501675945715, 0.11714100415866135, -0.034600745840691394, -0.15554835015413082), 
                          (0.47232144877941096, 0.07592804193006697, 0.3405294503824112, 0.21948666128556615), 
                          (-0.5332861513294131, -0.08886798656259431, -0.6083217502772487, 0.2316612945607922), 
                          (0.5258922230225587, 0.08708823651373188, 0.4234306811212821, -0.04179959817125434)), 
                         ((-0.3035688394279197, 0.49483554648710537, 0.05745994060999282, -0.12747381808333336), 
                          (0.39332214767836104, 0.46534190633912254, 0.04365690803996003, 0.7650379849598341), 
                          (0.1999985251338784, 0.0842829190391805, -0.16877153378631293, 0.31704188690973756), 
                          (-0.329887201245436, -0.6511923884088605, -0.6220851080992487, -0.8494734189039839)), 
                         ((-0.3231756242543622, -0.554464792835363, 0.4522418958171174, 0.7395807541527681), 
                          (0.5498904829451162, -0.05590161752080769, 0.4965321688385663, 0.5728276202033591), 
                          (0.5272647821134722, 0.5774470894044734, -0.12581800958739708, 0.9700828208339976), 
                          (0.2893640797543078, -0.5484780137131635, 0.9149540829219327, -0.4070409332163506)), 
                         ((0.8068861149817737, -0.08739646094657716, 0.15604945884485388, 0.2159877611783666), 
                          (-0.08757552786227041, -0.5362091498098354, 0.1379626847994988, -0.02263864637679258), 
                          (-0.05403837215914031, 0.6570928483791509, -0.014539560837078191, 0.1353955668770478), 
                          (-0.033528144292161, -0.471223531770967, -0.012087259265455508, -0.4443704288092061))), 
                        (((0.12815286370127177, 0.08288774005977806, -0.6591909957174392, -0.3227689715877514), 
                          (-0.1560371012493621, 0.04563397602135244, 0.18608455069144747, -0.23618249688069137), 
                          (-0.1857379168928618, -0.6112258715631804, 0.257168970542067, 0.46862986167858467), 
                          (0.8134511475542548, -0.2930500911112336, 0.31803040987546893, -0.1662974948326531)), 
                         ((0.09994879483867802, 0.11656104212951102, 0.24567744809079017, 0.2466674448155779), 
                          (0.10560598300537649, 0.5594223586687221, -0.269996887150316, -0.006724786299250773), 
                          (0.2494899758063036, -0.25302027094976054, 0.5346069245538014, 0.9394209073288707), 
                          (0.4895739925732161, -0.6973902811154304, -0.04576531668146877, -0.05978124614419089)), 
                         ((-0.4225874457751645, 0.06851122980870317, -0.12558552929455635, -0.059065175364745315), 
                          (-0.5869469247109007, -0.4166268018632071, -0.17595117034871932, 0.34489468842970006), 
                          (0.19428688924434412, -0.4410298941561428, -0.07610953060207504, 0.2917329578363864), 
                          (-0.3087834675004315, -0.5017232119230198, 0.6463307399810825, -0.0997023137687646)), 
                         ((-0.3721359342866154, 0.012800610965129966, -0.026147073121323694, 0.249394864255974), 
                          (-0.19732520391810227, 0.6877652212967196, 0.09328828976651427, -0.6510049138630607), 
                          (-0.4848547986972145, -0.35735976033028827, 0.09165186856904783, -0.40982007152443234), 
                          (-0.574914716311232, -0.6488293490765211, 0.4883180795650498, -0.15158507647269426))), 
                        (((0.5708514963046316, -0.21247793529126768, 0.11030963705086905, -0.4353799184220226), 
                          (0.37323818315312085, -0.10788816216228836, -0.16605325912656543, 0.6104874356554703), 
                          (-0.30847384861896465, -0.16679636365155348, 0.5922642783564169, 0.07864692847157684), 
                          (0.2791875133546827, 0.09496780294295215, -0.45316050025403876, -0.1610271603413973)), 
                         ((-0.42706418878716035, 0.2928066574998388, 0.571174679676169, -0.14560760969133357), 
                          (0.9643430182032757, -0.008481138733699822, 0.39800339772985893, -0.7013700291692165), 
                          (0.8355714997095827, -0.624788527855067, 0.05466250391400629, -0.9509878285060569), 
                          (-0.0973807062547889, -0.5407506278578577, 0.09210951353229235, 0.154379276954291)), 
                         ((0.020599999927309876, -0.27945772586433815, -0.4172996686676522, -0.01634325308254314), 
                          (0.20212191256900158, 0.21234576281878903, -0.639142180879022, 0.5580272378144776), 
                          (0.010545335242685194, 0.29112493809527284, 0.018635172247521647, -0.06613029570794826), 
                          (0.155833422371511, -0.09759470769513245, -0.03063166472236889, -0.05428014538371495)), 
                         ((0.23577186821185425, 0.1500579314912276, -0.3132878838809129, -0.1097762446447772), 
                          (0.6269933317054381, 0.6145430525526924, -0.10380848715617585, -0.06442945507207776), 
                          (0.2787614779979213, -0.7358725127577144, 0.20796477034679672, -0.23780046049837644), 
                          (0.9267282798589614, 0.16997001112351162, -0.5282983087375479, 0.36538217505303483))), 
                        (((-0.8579510472159081, -0.7246352503258299, -0.15510078775815894, 0.12619420876242404), 
                          (0.24743683866122967, 0.5615110519360075, -0.3322404459371945, -0.7048709861468231), 
                          (0.7028698531861565, -0.22853480716765995, 0.7720042932598191, -0.13166327267604216), 
                          (-0.2435352215957982, 0.22931953113912795, 0.3655062306761121, -0.12152100998681392)), 
                         ((-0.5677571978624529, 0.6521623982090801, 0.45610015782255964, 0.04035238030125632), 
                          (-0.23851414230639378, -0.30637231258391073, 0.046201693717512704, -0.27136411183670195), 
                          (0.6857476357642973, 0.3410717717617201, 0.01289524408298115, -0.013360855488527279), 
                          (-0.032304469159492366, -0.0835492970599051, -0.0028734534647487653, -0.4654703960804025)), 
                         ((-0.2290818436503651, -0.5296681103638246, 0.6162246876592932, 0.17128234890632854), 
                          (-0.5811995554061062, 0.6417647758172681, 0.7628730887993724, -0.37912907041142807), 
                          (-0.09316417943340105, -0.27095731560349645, 0.6194574328449272, -0.24948060355551438), 
                          (0.5728337951942462, -0.0924305626769284, 0.3298848378599575, 0.415637420812042)), 
                         ((-0.061049694343089045, -0.8003464214443662, 0.29655140456799867, -0.7735799111191699), 
                          (0.29917720082276444, 0.3643378373698344, -0.7665309138650984, 0.35430115119764427), 
                          (0.39429765393905847, -0.215211613577439, 0.22195510594293588, -0.20240345662915427), 
                          (-0.06694319875792287, -0.6036398699231508, -0.6775777632295854, -0.13199173966023536))))
            if rank==0:
                return (0.1916945198248936-0.13520600821648232j,)
            elif rank==1:
                return (-0.1083940865407611-0.43931913768613384j,)
            elif rank==2:
                return (((-0.0008665125778035687+0.43157678379712805j), (0.6508260343349135-0.13342532131846585j)), 
                        ((0.25346912744174943-0.6465790214072134j), (-0.15962337965743467-0.01177491815738152j)))
            elif rank==3:
                return ((((-0.11057636539275228-0.3175750394405701j), (0.7947500539520073-0.3434988438428338j), (-0.17008528220766383+0.22755084276175752j)),
                         ((-0.21753248913138046+0.3073046922694095j), (-0.5987149773078468+0.6404407581234498j), (-0.2733628131901674-0.7312216337378827j)),
                         ((0.46021781519502736-0.32179246699810604j), (-0.4650578935554821+0.5210104598763112j), (-0.36535537317523425+0.4289080712460789j))),
                        (((-0.19578475796433725-0.1604786258607791j), (-0.070669831429362-0.34066414803932554j), (0.09414012129996285-0.14312760505028332j)),
                         ((0.02962390819527594-0.08129940555056081j), (-0.29691277198852295+0.5445730330279386j), (-0.8314872838325128-0.10478982105017087j)),
                         ((-0.4875034489526303+0.11245261823496877j), (0.307917858878609+0.4746451405124599j), (0.41480950330452204+0.4450092645412934j))),
                        (((0.7567520084099626-0.5568082250793975j), (0.5933616991307173-0.03406028934656724j), (-0.5349903124170489-0.396773950179919j)),
                         ((0.46555626239679515-0.16518290948365888j), (0.8398844896104096+0.28827801011043974j), (-0.05286735815237498-0.06537526369272817j)),
                         ((-0.7873594868218297+0.23455905590585036j), (0.43718502738612375+0.17935920771077773j), (-0.16708210956083058-0.29630670019555294j))))
            elif rank==4:
                return (((((-0.37669743703656033+0.3367989420952414j), (0.3280202440259613-0.42443833914985585j), (0.5751207710641884-0.6671925997156294j), (0.033365010564720965+0.18416168128369115j)),
                          ((-0.1305885333955732+0.258183803742714j), (0.5291111892948406+0.12768860044004038j), (0.3044538909411727+0.7018739119608393j), (-0.09384324555072743+0.44762291055240755j)),
                          ((-0.5172985234092151-0.4246631134190011j), (-0.05697702151796369-0.38499672629412374j), (-0.16330371172505076+0.6329885254352636j), (0.005206064694582513-0.6979570589219142j)), 
                          ((0.10042081122085011+0.2730078961548078j), (0.37702343940278005+0.3312339822361312j), (-0.38712718853663275+0.043310706850169245j), (-0.43842301200718425-0.15186366915048966j))), 
                        (((0.5818010551152796-0.36497243994846373j), (0.19198280565084558-0.5011496107835725j), (0.24413432536571378-0.6379767379075698j), (0.34455193126269634-0.08664637040123147j)), 
                         ((0.8168889487559259-0.9032656902589163j), (-0.5603456946720172-0.1195734571550765j), (-0.1605088593685251-0.13483121449943314j), (0.6174544850032535-0.44047675305704703j)), 
                         ((-0.42307951312623815-0.30327702147916713j), (-0.2618637556175357-0.1986043745329351j), (0.04718850736664815-0.10738185754632579j), (0.23587778497574974-0.6504332945195135j)), 
                         ((-0.05076119295424941+0.4047602633806563j), (0.11093393085917946+0.12427836254382474j), (-0.8132105391139205+0.37548311525522216j), (-0.26953236282243265-0.31159778790771064j))), 
                        (((-0.39250877863236644-0.4881044190028637j), (-0.8360834411136835+0.02188471873683928j), (0.02679042509032048+0.6879063358020456j), (-0.10048291543082566-0.2233219762257277j)), 
                         ((0.5390402819188123+0.1204054017602989j), (-0.2931730415212582+0.11535281644759132j), (-0.30548980198917564+0.026411218830157512j), (-0.19472436433201012+0.081056583169555j)),
                         ((0.0005464858113309878+0.3225143801544309j), (-0.07709944497321652-0.7481530123633068j), (0.16337505191696478+0.5103151975086361j), (-0.12750402538858618+0.7545748937897574j)), 
                         ((-0.9554831189812294+0.13987108056505082j), (-0.27483290404380223+0.8226848528344127j), (0.521831527873442+0.4970661477293601j), (0.9061234743087792-0.1504241528269118j))), 
                        (((-0.1882846543421225+0.5304062832141554j), (0.10579078542556297+0.22780442714023064j), (-0.011294630508038805+0.5779466549586485j), (0.08025599710182507+0.12004966544994533j)), 
                         ((-0.6586788023520793+0.29488829592817734j), (-0.5025090781370782-0.495410044054439j), (-0.10853891499487645+0.024651020003384483j), (-0.0009726817521150677-0.0764710138257787j)), 
                         ((0.32613207343707196+0.7335754039319815j), (0.3508358997571207+0.3240452830066213j), (-0.3835841075004305-0.06342158227281525j), (0.32061104148045194+0.14435049185949667j)), 
                         ((-0.4702553987233302+0.2965575488822829j), (0.2726823746215451+0.21252896664298115j), (-0.40480220218419416+0.5684528722603068j), (-0.8662951962865073-0.02018655350327081j)))), 
                       ((((0.31691366872411453-0.16565320239992465j), (0.6119279222420073+0.4061058795460125j), (-0.2724431323567723+0.47301403973137435j), (-0.07564711093225085+0.3318822040674295j)), 
                         ((0.35444199580157854-0.17921840306504822j), (0.029427102181375453-0.03357316416483169j), (-0.21032412371046783+0.01667316554624787j), (0.0663834457499225+0.2586952407467318j)), 
                         ((0.20873626161577874+0.5387073631313459j), (-0.7679630879573601-0.6188379588287274j), (-0.3195249665071722-0.597193477248285j), (0.43272245800774456-0.23801421308823245j)), 
                         ((-0.037613711494389857+0.10818062943154994j), (0.057829377352543676-0.04999419264444982j), (0.3022465975999099-0.5349078896215682j), (-0.5231242476811815+0.42998339300583244j))), 
                        (((0.0856361897298803-0.6998154250460039j), (-0.11463868513147624+0.16152088708434365j), (-0.5127925687279636+0.11002739851016519j), (0.2599079869203247+0.02491071853128979j)), 
                         ((0.7015604708055061-0.591598702814476j), (0.7022599678145434+0.41303884890699527j), (0.6770753952633466+0.003915616175995784j), (0.5221501522889332+0.015184149238084643j)), 
                         ((-0.18955797508665606+0.5485819797109972j), (-0.2971676553030109-0.17836414101585318j), (-0.5711343992923511+0.06842526899025392j), (0.059265927050955236-0.6735602608611315j)), 
                         ((-0.1520857966428928+0.29169887374589554j), (-0.3146943503222607+0.5076388713801581j), (0.9710102040544507-0.04876498142745134j), (-0.19853216294306686-0.6550688027359666j))), 
                        (((0.18951981302782428-0.2412793295224065j), (-0.7777428226995063-0.365999090785438j), (0.6228318351380273+0.06995476281885704j), (-0.6740149120004719+0.10711001728991842j)),
                         ((0.1129279157975297-0.4283924106357786j), (0.40438767481744564-0.2635679354391648j), (-0.29420852109617845+0.4847733937880756j), (-0.3510828870978512-0.11064283076221504j)), 
                         ((-0.060808250212467274-0.3509555798569397j), (0.2310570876125041+0.27346891132233075j), (-0.07736476852424223-0.24833618620126674j), (-0.78074150969927-0.7230863125461435j)), 
                         ((-0.4757051297031988-0.3783458735437233j), (-0.5401521152562118+0.035723203675678605j), (-0.17580640953890292+0.6827665743960223j), (0.39326629202893215-0.21993263281519704j))), 
                        (((0.6386214443611579-0.764943290056036j), (0.18713633494964688+0.06363868519230131j), (0.29526336078276216+0.30178995300494826j), (0.05001443364181468-0.42881696117817625j)), 
                         ((-0.5275034189476259+0.6550538425247074j), (0.6943261638500615+0.006776882726276523j), (0.5153311571315663-0.05413330835192531j), (0.2988153589979258+0.3049744060817169j)), 
                         ((0.14785883887146767+0.0763051493978234j), (0.16312754082743042-0.03795584967477661j), (0.040229546593022913+0.16530303115245693j), (-0.1831772476291057+0.06328054147911466j)), 
                         ((0.1476643669767913+0.325680674159183j), (0.14707868586431028+0.9047584008820329j), (0.9011552589577467+0.11447442281060494j), (0.005973469367093642+0.22263862886336694j)))), 
                       ((((0.19283269000734593-0.28709497228064285j), (-0.47096481014533476-0.13173508510496024j), (0.562477770007406+0.26491027512352816j), (0.03783964186943933-0.03978396106597093j)), 
                         ((-0.5333576204038937+0.20057170743675312j), (-0.5554481805321547-0.7626212986623081j), (-0.47686434977953773+0.5802686248218423j), (0.5920361023050167-0.14491946597655325j)), 
                         ((-0.4126930229374297+0.39966003185091536j), (0.2850544980649773+0.7484131241244993j), (-0.13160190253004878-0.10190869345823672j), (0.34602765479100117-0.18205626983880596j)), 
                         ((0.4219188545707674+0.5086508238768007j), (-0.0677006672392233-0.11766198036052333j), (0.45020863060305627+0.20389430038718892j), (-0.5125265629738844-0.4771765521430704j))), 
                        (((0.37701640058097874+0.05204261987362635j), (0.17260326851591068+0.4285264843160088j), (0.10359131962737611-0.8137076321465101j), (-0.02286762418683419+0.34829474498525603j)), 
                         ((-0.010410922319074811+0.09105220878486131j), (0.43519051427706756-0.20447320160562032j), (-0.14399460120821073-0.46777690039065734j), (-0.04872781035419649-0.4298525400055381j)), 
                         ((0.18896093939688385-0.16705599789601278j), (-0.5364935215041872+0.3749309246942789j), (-0.18268036947723632-0.25385754338743693j), (-0.5091199532153567-0.32766264406763357j)), 
                         ((-0.35235219578282473-0.661850620842785j), (-0.6072679594510126+0.09656978047989173j), (-0.5126540710595279-0.1672195227969181j), (-0.2830351750830662+0.12517368885386926j))), 
                        (((0.04686974540685185+0.08438300173836044j), (-0.17974800480460484-0.5026848778236594j), (0.4496570044640005+0.5182953038532189j), (0.673892629006263-0.39593441606543756j)), 
                         ((0.2166952267447365+0.191459813763583j), (0.6130127820906195+0.3500318802920196j), (-0.5419820712240513-0.13457437148792628j), (0.3343112127482648+0.07888610819775466j)), 
                         ((-0.39276675341804346+0.7570076529568248j), (-0.3988406372126929-0.8608946218773403j), (0.576648779753566-0.6725693071527158j), (-0.4183085693722922-0.21421782809798817j)), 
                         ((-0.3010231423615105-0.020535079475881957j), (0.006107583077576639-0.0812822534033596j), (-0.029724210300932086+0.35635178233793563j), (-0.20891136846604574-0.34703924148843457j))), 
                        (((-0.07194174413486831-0.07558412271718051j), (0.04378297698600753+0.17934356988151767j), (-0.47189150794751766+0.5780327062551049j), (-0.5125220316089242+0.24892817111211096j)), 
                         ((-0.36059488463245326-0.44683441504676813j), (-0.26735527077170973-0.3699650359773635j), (-0.3057966963958182-0.07264390084352956j), (-0.7281560219128018-0.3560659246206268j)), 
                         ((0.1329046092679943-0.03153689763456646j), (-0.39271490859265534+0.2062873488401652j), (0.3802945078185428+0.013499908367005542j), (-0.1756191306230559+0.40464153144244697j)), 
                         ((0.4802390728634319+0.5190956065611698j), (0.14465603430627616+0.005475878763732567j), (0.3018703057475889-0.0352888758412262j), (0.6943368615391298+0.019202869121973j)))), 
                       ((((-0.07500673578875616+0.44031114122474946j), (-0.7205258224784314+0.16865003592058259j), (0.07737090079316822-0.1946277248102064j), (-0.34196425749144166+0.03817853010131156j)), 
                         ((0.17755139682612575+0.8238455606059262j), (-0.6657721503572177+0.19793806270117809j), (0.29019650633065386-0.0952118701646486j), (0.14507433328106534-0.515594049976717j)), 
                         ((-0.2046653289959417-0.3621700659367947j), (0.6793982849049701+0.3212370850216385j), (0.752468338707956-0.2485926849867901j), (0.5515361195592884-0.4034740977009965j)), 
                         ((0.06814007049845194-0.598437803123891j), (-0.5946418447770585-0.6903237885619982j), (-0.36295182276739524-0.7047248113077316j), (0.4250255531847421-0.18162334572617966j))), 
                        (((-0.330842360993192+0.6332178345844492j), (-0.1626525936800115+0.12616954195069585j), (-0.1140983493592932+0.16115787963839956j), (-0.5146861509822459-0.47712128557213807j)), 
                         ((-0.5846954521386037+0.05824004235002933j), (-0.6485911018934266+0.16723477358583239j), (0.08398284507027964-0.43010971419153865j), (0.473289092320511-0.20201419097169482j)), 
                         ((0.4304851671543479+0.3403150943545751j), (-0.5833160536688786+0.4728250155621482j), (0.07457807134020211-0.7490249718724199j), (-0.7171842661512965-0.4045396117992399j)), 
                         ((0.38115752645606027-0.581098921622462j), (-0.5631172341731325-0.3169246082570121j), (-0.3187906517321565-0.04147798917128964j), (0.19509565877461577-0.2078658202899497j))), 
                        (((0.8183607993729423+0.6099794355828236j), (-0.04871961343893161-0.07033624155755913j), (-0.5113369106671184+0.057386755559324376j), (-0.8572333770628078-0.11952395630018997j)), 
                         ((-0.14669430492038193+0.22930953454219938j), (0.8593021941295018-0.6567284188814271j), (-0.3498005693861541+0.4978632770594925j), (0.09110494626854304+0.14596905700539575j)), 
                         ((0.5821565237474922+0.0943377064392501j), (-0.12067415509324098-0.2752660016054119j), (-0.06217450649156364-0.024782412571600743j), (0.18973873165376154-0.4643249608246277j)), 
                         ((0.221440302929653-0.013268267564226499j), (0.23727492055667987+0.24335036846179636j), (0.20590162356036257-0.7773726672859762j), (0.6650340170009239+0.23251699933937398j))), 
                        (((-0.10862117873771637-0.37601603473231926j), (0.14702795412639624+0.2708674170765101j), (-0.6649779112242566-0.18174044911513143j), (-0.8835405205485383+0.67800004101369j)), 
                         ((0.5595753470648851+0.06619472947501459j), (-0.5165025553457263+0.06614105368782663j), (0.6989509353083828-0.04232214137167867j), (0.27326536666764734+0.055971454399568965j)), 
                         ((-0.4225563301191848-0.46504433769204656j), (0.021693692712496326+0.03736604787989928j), (-0.5231593756263367-0.2743293424312542j), (0.030072300170931987-0.5713925695108115j)), 
                         ((0.2665173135965969-0.541283942963249j), (0.10755640859650628-0.30605105910067054j), (-0.5717247916937898+0.2531629402939769j), (0.6106906755601762-0.8354582126221755j)))))

    def get_array_inputL(self, rank, cplx, no_first_arg_negative=False, no_zeros=False, mult=1000):
        z=self.get_python_inputL(rank, cplx, no_first_arg_negative=no_first_arg_negative, no_zeros=no_zeros)
        r=numpy.array(z)
        if mult!=1000:
            r=r/1000*mult
        return (r,r.copy())
    
    # does not have the same range as get_array_inputL but hopefully this will be enough
    # at the moment it is only used in tagged testing
    def get_array_inputL2(self, rank, cplx):
        z=self.get_python_input2(rank, cplx)
        return (10*numpy.array(z), 10*numpy.array(z))
        
    def get_array_input1(self, rank, cplx, no_first_arg_negative=False, no_zeros=False):
        z=self.get_python_input1(rank, cplx, no_first_arg_negative, no_zeros=no_zeros)
        return (numpy.array(z),numpy.array(z))

    def get_array_input2(self, rank, cplx, no_zeros=False, no_first_arg_negative=False):
        z=self.get_python_input2(rank, cplx, no_zeros=no_zeros, no_first_arg_negative=no_first_arg_negative)
        return (numpy.array(z),numpy.array(z))

    def get_array_input3(self, rank, cplx, no_first_arg_negative=False, no_zeros=False):
        z=self.get_python_input3(rank, cplx, no_first_arg_negative=no_first_arg_negative, no_zeros=no_zeros)
        return (numpy.array(z),numpy.array(z))
    
    def get_array_input4(self, rank, cplx):
        z=self.get_python_input4(rank, cplx)
        return (numpy.array(z),numpy.array(z))    

    def get_const_inputL(self, rank, fs, cplx):
        z=self.get_python_inputL(rank, cplx)
        d=Data(z, fs)
        return (d, numpy.array(z))
    
    def get_const_input1(self, rank, fs, cplx, no_first_arg_negative=False, no_zeros=False):
        z=self.get_python_input1(rank, cplx, no_first_arg_negative, no_zeros=no_zeros)
        d=Data(z, fs)
        return (d, numpy.array(z))
    
    def get_const_input2(self, rank, fs, cplx, no_zeros=False):
        z=self.get_python_input2(rank, cplx, no_zeros=no_zeros)
        d=Data(z, fs)
        return (d, numpy.array(z))
 
    def get_expanded_inputL(self, rank, fs, cplx, no_first_arg_negative=False, no_zeros=False, mult=1000):
        base=self.get_python_inputL(rank, cplx, no_first_arg_negative=no_first_arg_negative, no_zeros=no_zeros)
        v='x'
        if mult!=1000:
            base=numpy.array(base)
            v=Lsup(base)
            base=base/1000*mult
            base=base.tolist()
        x=fs.getX()[0]
        mask=whereNegative(x-0.5)   # A trick from Lutz Gross' version of tests
        r=base*mask+(1-mask)*base
        return (r, base)
 
    def get_expanded_inputL2(self, rank, fs, cplx, no_zeros=False, mult=1000):
        base=numpy.array(self.get_python_input1(rank, cplx))*mult
        base=base.tolist()
        x=fs.getX()[0]
        mask=whereNegative(x-0.5)   # A trick from Lutz Gross' version of tests
        r=base*mask+(1-mask)*base
        return (r, base)
 
    def get_expanded_input1(self, rank, fs, cplx):
        base=self.get_python_input1(rank, cplx)
        x=fs.getX()[0]
        mask=whereNegative(x-0.5)   # A trick from Lutz Gross' version of tests
        r=base*mask+(1-mask)*base
        return (r, base)

    def get_expanded_input2(self, rank, fs, cplx):
        base=self.get_python_input2(rank, cplx)
        x=fs.getX()[0]
        mask=whereNegative(x-0.5)   # A trick from Lutz Gross' version of tests
        r=base*mask+(1-mask)*base
        return (r, base)
    
    def get_tagged_inputL(self, rank, fs, cplx):
        z=self.get_python_inputL(rank, cplx)
        ref=self.get_array_inputL(rank, cplx)[0]
        d=Data(z, fs)
        d.tag()
        return (d, ref)   
    
    def get_tagged_input2(self, rank, fs, cplx):
        z=self.get_python_input2(rank, cplx)
        ref=self.get_array_input2(rank, cplx)[0]
        d=Data(z, fs)
        d.tag()
        return (d, ref)      

    def get_tagged_with_tagL1(self, rank, fs, cplx, set_tags=False, no_first_arg_negative=False, no_zeros=False, mult=1000):
        ref=self.get_array_inputL(rank, cplx, no_first_arg_negative=no_first_arg_negative, no_zeros=no_zeros, mult=mult)[0]
        d=Data(ref,fs)
        d.tag()
        if set_tags:
            ref1=self.get_array_input2(rank, cplx, no_first_arg_negative=no_first_arg_negative, no_zeros=no_zeros)[0]*mult
            for t in fs.getListOfTags():
                d.setTaggedValue(t, ref1)*mult
                ref=ref1
        return (d,ref)
    
    def get_tagged_with_tagL2(self, rank, fs, cplx, set_tags=False, no_zeros=False, mult=1000):
        ref=self.get_array_input2(rank, cplx, no_zeros=no_zeros)[0]*mult
        d=Data(ref,fs)
        if set_tags:
            ref1=self.get_array_input3(rank, cplx, no_zeros=no_zeros)[0]*mult
            for t in fs.getListOfTags():
                d.setTaggedValue(t, ref1)
                ref=ref1
        return (d,ref)
    
    def get_tagged_input1(self, rank, fs, cplx):
        z=self.get_python_input1(rank, cplx)
        ref=self.get_array_input1(rank, cplx)[0]
        d=Data(z, fs)
        d.tag()
        return (d, ref)
    
    def execute_ce_params(self, pars):
        for v in pars:
            a=v[0]
            op=v[1]
            misccheck=v[2]
            ref=v[3]
            oraclecheck=v[4]
            description=v[5]
            res=eval(op)
            if misccheck is not None:
                miscres=eval(misccheck)
                if not miscres:
                    print("Failed check:"+misccheck)
                    print(type(a))
                    print(" vs ")
                    print(type(res))
                    print(" values:")
                    print(a)
                    print(res)
                self.assertTrue(miscres,"Failed check for "+description)
            oraclevalue=eval(oraclecheck)
            oracleres=Lsup(res-oraclevalue)<=self.RES_TOL*Lsup(oraclevalue)
            if not oracleres:
                print("Wrong result:"+oraclecheck)
                print(type(res))
                print(" vs ")
                print(type(oraclevalue))
                print(" values:")
                print(res)
                print(" vs ")
                print(oraclevalue)                
            self.assertTrue(oracleres,"wrong result for "+description)
            
    def execute_ce_throws(self, pars):
        for v in pars:
            a=v[0]
    def get_expanded_input1(self, rank, fs, cplx):
        base=self.get_python_input1(rank, cplx)
        x=fs.getX()[0]
        mask=whereNegative(x-0.5)   # A trick from Lutz Gross' version of tests
        r=base*mask+(1-mask)*base
        return (r, base)
    
    def get_tagged_inputL(self, rank, fs, cplx):
        z=self.get_python_inputL(rank, cplx)
        ref=self.get_array_inputL(rank, cplx)[0]
        d=Data(z, fs)
        d.tag()
        return (d, ref)   

    def get_tagged_input1(self, rank, fs, cplx):
        z=self.get_python_input1(rank, cplx)
        ref=self.get_array_input1(rank, cplx)[0]
        d=Data(z, fs)
        d.tag()
        return (d, ref)
    
    def get_array_by_shape(self, s, cplx):
        dt=numpy.float64 if not cplx else numpy.complex128
        n=numpy.prod(s)
        a=numpy.arange(n, dtype=dt).reshape(s)
        if cplx:
            a-=((1+numpy.arange(n, dtype=dt))*1j).reshape(s)
        return a
    
    def make_constant_from_array(self, a, fs):
        d=Data(a, fs)
        return d
        
    def make_tagged_from_array(self, a, fs):
        d=Data(1.5, getShape(a), fs)
        if a.dtype.kind=='c':
            d.promote()
        for n in fs.getListOfTags():
            d.setTaggedValue(n, a)
        return d
        
    def make_expanded_from_array(self, a, fs):
        d=Data(a, fs)
        d.expand()
        return d
    
    def execute_ce_params(self, pars):
        for v in pars:
            a=v[0]
            op=v[1]
            misccheck=v[2]
            ref=v[3]
            oraclecheck=v[4]
            description=v[5]
            res=eval(op)
            if misccheck is not None:
                miscres=eval(misccheck)
                if not miscres:
                    print("Failed check:"+misccheck)
                    print(type(a))
                    print(" vs ")
                    print(type(res))
                    print(" values:")
                    print(a)
                    print(res)
                self.assertTrue(miscres,"Failed check for "+description)
            if oraclecheck is not None:
                oraclevalue=eval(oraclecheck)
                if isinstance(res, Data):
                    res.resolve()
                if isinstance(oraclevalue, Data):
                    oraclevalue.resolve()
                oracleres=Lsup(res-oraclevalue)<=self.RES_TOL*Lsup(oraclevalue)
                if not oracleres:
                    print("Wrong result:"+oraclecheck)
                    print(type(res))
                    print(" vs ")
                    print(type(oraclevalue))
                    print(" values:")
                    print(res)
                    print(" vs ")
                    print(oraclevalue)                
                self.assertTrue(oracleres,"wrong result for "+description)
            
    def execute_ce_throws(self, pars):
        for v in pars:
            a=v[0]
            op=v[1]
            misccheck=v[2]
            ref=v[3]
            oraclecheck=v[4]
            description=v[5]
            expected_exceptions=v[6]
            if expected_exceptions is None:
                expected_exceptions=(TypeError, RuntimeError)   # These are used for unsupported complex
            try:
                res=eval(op)                
            except Exception as ex:
                # unfortunately, we don't return a single exception type in this case
                self.assertTrue(type(ex) in expected_exceptions, "Exception was raised but it was of unexpected type ("+str(type(ex))+")")
            else:
                print("Processing expression "+str(op)+" with value "+str(a))
                raise AssertionError("Exception not raised")
    
    def execute_t_params(self, pars):
        for v in pars:
            description=v[0]
            a=v[1]
            tagcount=1
            for step in v[3:]:
                a.setTaggedValue(tagcount, step[0])
                op=step[1]
                misccheck=step[2]
                ref=step[3]
                oraclecheck=step[4]
                res=eval(op)
                if misccheck is not None:
                    self.assertTrue(eval(misccheck), "Check failed for "+description+" checking "+misccheck)
                if oraclecheck is not None:
                    oraclevalue=eval(oraclecheck)
                else:
                    oraclevalue=ref
                if oraclevalue is not None:
                    oracleres=Lsup(res-oraclevalue)<=self.RES_TOL*Lsup(oraclevalue)
                    if not oracleres:
                        print(v)
                        print(" This step ")
                        print(step)
                        print("Failed comparison:")
                        print(res)
                        print(" vs ")
                        print(oraclevalue)
                        print(" a= ")
                        print(a)
                        print(" ref== ")
                        print(ref)
                        print(" oraclecheck= ")
                        print(oraclecheck)
                    self.assertTrue(oracleres,"wrong result for "+description+" for tag "+str(tagcount))
                    tagcount+=1

        
    def execute_t_throws(self, pars):
        for v in pars:
            description=v[0]
            a=v[1]
            tagcount=1
            expected_exceptions=v[2]
            if expected_exceptions is None:
                expected_exceptions=(TypeError, RuntimeError)   # These are used for unsupported complex
            for step in v[3:]:
                a.setTaggedValue(tagcount, step[0])
                op=step[1]
                misccheck=step[2]
                ref=step[3]
                oraclecheck=step[4]
                with self.assertRaises(Exception) as err:  
                    res=eval(op)
                self.assertTrue(type(err.exception) in expected_exceptions, "Exception was raised but it was of unexpected type ("+str(type(err.exception))+")")                
                tagcount+=1

    def generate_operation_test_batch(self, supportcplx, opstring, misccheck, oraclecheck, opname, update1, input_trans=None, no_scalars=False, minrank=0, maxrank=4,
                                      expect_raise_on_ranks=None, expected_exceptions=None):
        """
        supportcplx is a boolean indicating whether complex operations should be checked for values (True)
             or tested to see if they raise (False)
        opstring is a string of the operation to be performed (in terms of argument a) eg "Lsup(a)"
        misccheck is a string giving a check to be run after the operation eg "isinstance(res,float)"
        opname is a string used to describe the operation being tested eg "inf"
        update1 and update2 are strings giving code used to update a variable rmerge to  
            account for tag additions for tagged data.
            eg:             update1="r2.min()"
                            update2="min(rmerge, r3.min())"
            would result in     rmerge=eval(update1) running after the first tag is calculatedand 
                                rmerge=eval(update2) running after the second
        """
        self.generate_operation_test_batch_driver(supportcplx, opstring, misccheck, oraclecheck, opname, update1, input_trans, no_scalars, minrank, maxrank,
                                      expect_raise_on_ranks, expected_exceptions, inp_scalar1=self.get_scalar_input1, inp_array1=self.get_array_input1, inp_array2=self.get_array_input2,  inp_const1=self.get_const_input1,  inp_expanded1=self.get_expanded_input1, inp_tagged1=self.get_tagged_input1)

    def generate_operation_test_batch_driver(self, supportcplx, opstring, misccheck, oraclecheck, opname, update1, input_trans=None, no_scalars=False, minrank=0, maxrank=4,
                                      expect_raise_on_ranks=None, expected_exceptions=None, inp_scalar1=None, inp_array1=None, inp_array2=None,  inp_const1=None,  inp_expanded1=None, inp_tagged1=None):
        """
        supportcplx is a boolean indicating whether complex operations should be checked for values (True)
             or tested to see if they raise (False)
        opstring is a string of the operation to be performed (in terms of argument a) eg "Lsup(a)"
        misccheck is a string giving a check to be run after the operation eg "isinstance(res,float)"
        opname is a string used to describe the operation being tested eg "inf"
        update1 and update2 are strings giving code used to update a variable rmerge to  
            account for tag additions for tagged data.
            eg:             update1="r2.min()"
                            update2="min(rmerge, r3.min())"
            would result in     rmerge=eval(update1) running after the first tag is calculatedand 
                                rmerge=eval(update2) running after the second
        """
        if input_trans is None:
            input_trans=lambda x: x
        if expect_raise_on_ranks is None:
            expect_raise_on_ranks=()
        pars=[]
        epars=[]    # operations which should throw
        if not no_scalars:
            (f1,f2)=inp_scalar1(False)
            f1=input_trans(f1)
            f2=input_trans(f2)
            pars.append((f1, opstring, misccheck, numpy.array(f2), oraclecheck, opname+" - scalar", expected_exceptions))
            if supportcplx:
                (f1,f2)=inp_scalar1(True)
                f1=input_trans(f1)
                f2=input_trans(f2)
                pars.append((f1, opstring, misccheck, numpy.array(f2), oraclecheck, opname+" - complex scalar",expected_exceptions))
            else:
                (f1,f2)=inp_scalar1(True)
                f1=input_trans(f1)
                f2=input_trans(f2)                
                epars.append((f1, opstring, misccheck, numpy.array(f2), oraclecheck, opname+" - complex scalar", expected_exceptions))
        for c in (False, True):
            if c:
                cs="complex "
            else:
                cs=""
            for rank in range(minrank, maxrank+1):
                dest=pars
                if rank in expect_raise_on_ranks or (c and not supportcplx):
                    dest=epars                
                if not no_scalars:
                    (a, r)=inp_array1(rank, c)
                    a=input_trans(a)
                    r=input_trans(r)
                    p=(a, opstring, misccheck, numpy.array(r), oraclecheck, opname+" - "+cs+"array rank "+str(rank), expected_exceptions)
                    dest.append(p)
                (a, r)=inp_const1(rank, self.functionspace, c)
                a=input_trans(a)
                r=input_trans(numpy.array(r))                
                p=(a, opstring, misccheck, r, oraclecheck, opname+" - "+cs+"Constant Data rank "+str(rank), expected_exceptions)
                dest.append(p)
                (a, r)=inp_expanded1(rank, self.functionspace, c)
                a=input_trans(a)
                r=input_trans(numpy.array(r))                
                p=(a, opstring, misccheck, r, oraclecheck, opname+" - "+cs+"Expanded Data rank "+str(rank), expected_exceptions)
                dest.append(p)
        self.execute_ce_params(pars)
        self.execute_ce_throws(epars)
        del pars
        del epars
        tpars=[]    # tagged versions
        epars=[]
        for c in (False, True):
            if c:
                cs="complex "
            else:
                cs=""
            for rank in range(minrank, maxrank+1):
                dest=tpars
                if rank in expect_raise_on_ranks or (c and not supportcplx):
                    dest=epars                
                test=[opname+" - "+cs+"tagged rank "+str(rank),]
                (a, r)=inp_tagged1(rank, self.functionspace, c)
                a=input_trans(a)
                r=input_trans(numpy.array(r))                                
                test.append(a)
                test.append(expected_exceptions)
                # arguments are new tagged value, operation, extra check, reference_value, reference_check
                (t2, r2)=inp_array2(rank, c)
                t2=input_trans(t2)
                r2=input_trans(numpy.array(r2))
                if update1 is not None:
                    rmerge=eval(update1)
                else:
                    rmerge=None
                test.append((t2, opstring, misccheck, rmerge, None,))
                dest.append(test)
        self.execute_t_params(tpars)
        self.execute_t_throws(epars)   


    def generate_operation_test_batch(self, supportcplx, opstring, misccheck, oraclecheck, opname, update1, input_trans=None, no_scalars=False, minrank=0, maxrank=4,
                                      expect_raise_on_ranks=None, expected_exceptions=None):
        """
        supportcplx is a boolean indicating whether complex operations should be checked for values (True)
             or tested to see if they raise (False)
        opstring is a string of the operation to be performed (in terms of argument a) eg "Lsup(a)"
        misccheck is a string giving a check to be run after the operation eg "isinstance(res,float)"
        opname is a string used to describe the operation being tested eg "inf"
        update1 and update2 are strings giving code used to update a variable rmerge to  
            account for tag additions for tagged data.
            eg:             update1="r2.min()"
                            update2="min(rmerge, r3.min())"
            would result in     rmerge=eval(update1) running after the first tag is calculatedand 
                                rmerge=eval(update2) running after the second
        """
        self.generate_operation_test_batch_driver(supportcplx, opstring, misccheck, oraclecheck, opname, update1, input_trans=input_trans, no_scalars=no_scalars, minrank=minrank, maxrank=maxrank,
                                      expect_raise_on_ranks=expect_raise_on_ranks, expected_exceptions=expected_exceptions, inp_scalar1=self.get_scalar_input1, inp_array1=self.get_array_input1, inp_array2=self.get_array_input2,  inp_const1=self.get_const_input1,  inp_expanded1=self.get_expanded_input1, inp_tagged1=self.get_tagged_input1)


    def generate_operation_test_batch_large(self, supportcplx, opstring, misccheck, oraclecheck, opname, update1, input_trans=None, no_scalars=False, minrank=0, maxrank=4, expect_raise_on_ranks=None, expected_exceptions=None):
        """
        Uses the same logic as generate_operation_test_batch but uses larger values.
        supportcplx is a boolean indicating whether complex operations should be checked for values (True)
             or tested to see if they raise (False)
        opstring is a string of the operation to be performed (in terms of argument a) eg "Lsup(a)"
        misccheck is a string giving a check to be run after the operation eg "isinstance(res,float)"
        opname is a string used to describe the operation being tested eg "inf"
        update1 and update2 are strings giving code used to update a variable rmerge to  
            account for tag additions for tagged data.
            eg:             update1="r2.min()"
                            update2="min(rmerge, r3.min())"
            would result in     rmerge=eval(update1) running after the first tag is calculatedand 
                                rmerge=eval(update2) running after the second
        """
        self.generate_operation_test_batch_driver(supportcplx, opstring, misccheck, oraclecheck, opname, update1, input_trans, no_scalars, minrank, maxrank,
                                      expect_raise_on_ranks, expected_exceptions, inp_scalar1=self.get_scalar_inputL, inp_array1=self.get_array_inputL, inp_array2=self.get_array_inputL2,  inp_const1=self.get_const_inputL,  inp_expanded1=self.get_expanded_inputL, inp_tagged1=self.get_tagged_inputL)

    def execute_binary_params(self, pars):
        for v in pars:
            a=v[0]
            b=v[1]
            op=v[2]
            misccheck=v[3]
            refa=v[4]
            refb=v[5]
            oraclecheck=v[6]
            description=v[7]
            res=eval(op)
            if misccheck is not None:
                miscres=eval(misccheck)
                if not miscres:
                    print("Failed check:"+misccheck)
                    print(type(a))
                    print(" , ")
                    print(type(b))
                    print(" vs ")
                    print(type(res))
                    print(" values:")
                    print(a)
                    print(" , ")
                    print(b)
                    print(res)
                self.assertTrue(miscres,"Failed check for "+description)
            oraclevalue=eval(oraclecheck)
            oracleres=Lsup(res-oraclevalue)<=self.RES_TOL*Lsup(oraclevalue)
            if not oracleres:
                print("Wrong result:"+oraclecheck)
                print(" refa=")
                print(str(refa))
                print(" refb=")
                print(str(refb))
                print(" res=") 
                print(type(res))
                print(" vs oraclevalue=")
                print(type(oraclevalue))
                print(" values:")
                print(res)
                print(" vs ")
                print(oraclevalue)
            self.assertTrue(oracleres,"wrong result for "+description)

    def generate_binary_matrixlike_operation_test_batch_large(self, opstring, misccheck, oraclecheck, opname, input_trans=None, minrank=0, maxrank=4, aranks=(2,)):
        """
        Generates a set of tests for binary operations.
        It is similar to the unary versions but with some unneeded options removed.
        For example, all operations in this type should accept complex arguments.
        opstring is a string of the operation to be performed (in terms of arguments a and b) eg "inner(a,b)"
        misccheck is a string giving a check to be run after the operation eg "isinstance(res,float)"
        opname is a string used to describe the operation being tested eg "inner"
        update1 is a string giving code used to update a variable rmerge to  
            account for tag additions for tagged data.
            eg:             update1="r2.min()"
            would result in     rmerge=eval(update1) running after the first tag is calculated 
        """
        if input_trans is None:
            input_trans=lambda x: x
        pars=[]
        for ac in (False, True):    # complex or real arguments
            for bc in (False, True):
                astr="real" if ac else "complex"
                bstr="real" if bc else "complex"
                aargset=[]
                bargset=[]
                arange=aranks
                for atype in "ACTE":   # Array/Constant/Tagged/Expanded
                    if atype=='A':
                        for r in arange:
                            aargset.append((self.get_array_input1(r,ac),astr+' array rank '+str(r), r))
                    elif atype=='C':
                        for r in arange:
                            aargset.append((self.get_const_input1(r, self.functionspace, ac), astr+' Constant rank '+str(r), r))
                    elif atype=='T':
                        for r in arange:
                            aargset.append((self.get_tagged_with_tagL1(r, self.functionspace, ac, set_tags=False),astr+' Tagged rank '+str(r), r))
                    elif atype=='E':
                        for r in arange:
                            aargset.append((self.get_expanded_inputL(r,self.functionspace, ac),astr+' Expanded rank '+str(r), r)) 
                # Now we have a set of a args, match them with possible b's
                for v in aargset:
                    arg=v[0][0]
                    argref=v[0][1]
                    adescr=v[1]
                    rank=v[2]
                    if rank==2:
                        brange=(1,2)
                    elif rank==4:
                        brange=(2,3,4)
                    for br in brange:
                        tshape=(rank,)*br
                        bargref=self.get_array_by_shape(tshape, bc)
                        # now convert it to each possbile input type
                        barg=self.make_constant_from_array(bargref, self.functionspace)
                        bdescr=bstr+' Constant rank '+str(br)
                        p=(arg, barg, opstring, misccheck, 
                           numpy.array(argref), numpy.array(bargref), 
                           oraclecheck, opname+' '+adescr+'/'+bdescr)
                        pars.append(p)      
                        barg=self.make_tagged_from_array(bargref, self.functionspace)
                        bdescr=bstr+' Tagged rank '+str(br)
                        p=(arg, barg, opstring, misccheck, 
                           numpy.array(argref), numpy.array(bargref), 
                           oraclecheck, opname+' '+adescr+'/'+bdescr)
                        pars.append(p)           
                        barg=self.make_expanded_from_array(bargref, self.functionspace)
                        bdescr=bstr+' Expanded rank '+str(br)
                        p=(arg, barg, opstring, misccheck, 
                           numpy.array(argref), numpy.array(bargref), 
                           oraclecheck, opname+' '+adescr+'/'+bdescr)
                        pars.append(p)           
        self.execute_binary_params(pars)

    def generate_binary_operation_test_batch(self, opstring, misccheck, oraclecheck, opname, minrank=0, maxrank=4, no_shape_mismatch=False, permit_scalar_mismatch=True, cap_combined_rank=False, fix_rank_a=None, fix_rank_b=None, support_cplx=True, permit_array_op_data=True, 
                                                   no_second_arg_zeros=False, no_first_arg_zeros=False, no_first_arg_negative=False, second_large_args=True, first_large_args=True):
        """
        Generates a set of tests for binary operations.
        It is similar to the unary versions but with some unneeded options removed.
        opstring is a string of the operation to be performed (in terms of arguments a and b) eg "inner(a,b)"
        misccheck is a string giving a check to be run after the operation eg "isinstance(res,float)"
        opname is a string used to describe the operation being tested eg "inner"
        update1 is a string giving code used to update a variable rmerge to  
            account for tag additions for tagged data.
            eg:             update1="r2.min()"
            would result in     rmerge=eval(update1) running after the first tag is calculated 
        Note: permit_array_op_data indicates whether cases with a numpy.ndarray as the first
        argument and Data as the second will be generated. (This is to avoid numpy's interpretation of +)
        """
        pars=[]
        if support_cplx:
            complexity=(False, True)
        else:
            complexity=(False,)
        for ac in complexity:    # complex or real arguments
            for bc in complexity:
                astr="real" if not ac else "complex"
                bstr="real" if not bc else "complex"
                aargset=[]
                bargset=[]
                if fix_rank_a is not None:
                    arange=fix_rank_a
                else:
                    arange=range(minrank, maxrank+1)
                if fix_rank_b is not None:
                    brange=fix_rank_b
                else:
                    brange=range(minrank, maxrank+1)
                amult=1000 if first_large_args else 10
                bmult=1000 if second_large_args else 10
                for atype in "SACTE":   # Scalar/Array/Constant/Tagged/Expanded
                    if atype=='S':
                        aargset.append((self.get_scalar_input1(ac, no_first_arg_negative=no_first_arg_negative, no_zeros=no_first_arg_zeros),astr+' scalar'))
                    elif atype=='A':
                        for r in arange:
                            aargset.append((self.get_array_input1(r,ac, no_first_arg_negative=no_first_arg_negative, no_zeros=no_first_arg_zeros),astr+' array rank '+str(r)))
                    elif atype=='C':
                        for r in arange:
                            aargset.append((self.get_const_input1(r, self.functionspace, ac, no_first_arg_negative=no_first_arg_negative, no_zeros=no_first_arg_zeros), astr+' Constant rank '+str(r)))
                    elif atype=='T':
                        for r in arange:
                            aargset.append((self.get_tagged_with_tagL1(r, self.functionspace, ac, set_tags=False, no_first_arg_negative=no_first_arg_negative, no_zeros=no_first_arg_zeros, mult=amult),astr+' Tagged rRank '+str(r)))
                    elif atype=='E':
                        for r in arange:
                            aargset.append((self.get_expanded_inputL(r,self.functionspace, ac, no_first_arg_negative=no_first_arg_negative, no_zeros=no_first_arg_zeros, mult=amult),astr+' Expanded rank '+str(r)))    
                for atype in "SACTE":   # Scalar/Array/Constant/Tagged/Expanded
                    if atype=='S':
                        bargset.append((self.get_scalar_input2(bc, no_zeros=no_second_arg_zeros),bstr+' scalar'))
                    elif atype=='A':
                        for r in brange:
                            bargset.append((self.get_array_input2(r, bc, no_zeros=no_second_arg_zeros),bstr+' array rank '+str(r)))
                    elif atype=='C':
                        for r in brange:
                            bargset.append((self.get_const_input2(r, self.functionspace, bc, no_zeros=no_second_arg_zeros),bstr+' Constant rank '+str(r)))
                    elif atype=='T':
                        for r in brange:
                            bargset.append((self.get_tagged_with_tagL2(r, self.functionspace, bc, set_tags=True, no_zeros=no_second_arg_zeros, mult=bmult), bstr+' Tagged rank '+str(r)))                                
                    elif atype=='E':
                        for r in brange:
                            bargset.append((self.get_expanded_inputL2(r, self.functionspace, bc, no_zeros=no_second_arg_zeros, mult=bmult),bstr+' Expanded rank '+str(r)))
                # now we have a complete set of possible args    
                for aarg in aargset:
                    for barg in bargset:
                        if cap_combined_rank and getRank(aarg[0][0])+getRank(barg[0][0])>4:
                            continue  #resulting object too big
                        if no_shape_mismatch:
                            sa=getShape(aarg[0][0])
                            sb=getShape(barg[0][0])
                            if sa!=sb:
                                if not permit_scalar_mismatch or (sa!=() and sb!=()):
                                   continue
                        if isinstance(aarg[0][0], numpy.ndarray) and isinstance(barg[0][0], Data) and not permit_array_op_data:
                            continue
                        p=(aarg[0][0], barg[0][0], opstring, misccheck, 
                           numpy.array(aarg[0][1]), numpy.array(barg[0][1]), 
                           oraclecheck, opname+' '+aarg[1]+'/'+barg[1])
                        pars.append(p)           
        self.execute_binary_params(pars)

class Test_util_base(Test_util_values):
   """
   basic tests on util.py
   """
#=========================================================
#  File writer
#=========================================================
   def setup(self):
        self.workdir=ESCRIPT_WORKDIR

   def __checkContent(self,fn,ref_cont):
        cont=open(fn,'r').readlines()
        self.assertTrue(len(cont)==len(ref_cont),"wrong number of records")
        for i in range(len(cont)):
           self.assertTrue(cont[i].strip()==ref_cont[i],"wrong records %s"%i)
   def test_FileWriter_W(self):
        fn=os.path.join(self.workdir, "filewriter_w.txt")
        self.assertRaises(IOError,FileWriter,fn="",append=False)
        f=FileWriter(fn,append=False)
        self.assertTrue(f.name==fn, "wrong file name.")
        self.assertTrue(f.mode=='w', "wrong mode")
        self.assertTrue(f.newlines==os.linesep, "wrong line seps")
        self.assertTrue(not f.closed,"file shuold not be closed.")
        f.write("line1"+f.newlines)
        f.flush()
        self.__checkContent(fn,["line1"])
        f.writelines(["line2"+f.newlines, "line3"+f.newlines])
        f.close()
        self.assertTrue(f.closed,"file shuold be closed.")
        self.__checkContent(fn,["line1", "line2", "line3"])
        MPIBarrierWorld()
        if getMPIRankWorld()==0: os.unlink(fn)

   def test_FileWriter_A(self):
        fn=os.path.join(self.workdir, "filewriter_a.txt")
        if getMPIRankWorld()==0: open(fn,'w').write("line1"+os.linesep)
        self.assertRaises(IOError,FileWriter,fn="",append=True)
        f=FileWriter(fn,append=True)
        self.assertTrue(f.name==fn, "wrong file name.")
        self.assertTrue(f.mode=='a', "wrong mode")
        self.assertTrue(f.newlines==os.linesep, "wrong line seps")
        self.assertTrue(not f.closed,"file shuold not be closed.")
        f.write("line2"+f.newlines)
        f.flush()
        self.__checkContent(fn,["line1", "line2"])
        f.writelines(["line3"+f.newlines, "line4"+f.newlines])
        f.close()
        self.assertTrue(f.closed,"file shuold be closed.")
        self.__checkContent(fn,["line1", "line2", "line3", "line4"])
        MPIBarrierWorld()
        if getMPIRankWorld()==0: os.unlink(fn)

   def test_FileWriter_A_loc(self):
        fn=os.path.join(self.workdir, "filewriter_a_loc.txt")
        if getMPIRankWorld()>0:
            fn2=fn+".%s"%getMPIRankWorld()
        else:
            fn2=fn
        open(fn2,'w').write("line1"+os.linesep)
        self.assertRaises(IOError,FileWriter,fn="",append=True, createLocalFiles=True)
        f=FileWriter(fn,append=True,createLocalFiles=True)
        self.assertTrue(f.name==fn, "wrong file name.")
        self.assertTrue(f.mode=='a', "wrong mode")
        self.assertTrue(f.newlines==os.linesep, "wrong line seps")
        self.assertTrue(not f.closed,"file shuold not be closed.")
        f.write("line2"+f.newlines)
        f.flush()
        self.__checkContent(fn2,["line1", "line2"])
        f.writelines(["line3"+f.newlines, "line4"+f.newlines])
        f.close()
        self.assertTrue(f.closed,"file shuold be closed.")
        self.__checkContent(fn2,["line1", "line2", "line3", "line4"])
        os.unlink(fn2)

   def test_FileWriter_W_loc(self):
        fn=os.path.join(self.workdir, "filewriter_w_loc.txt")
        if getMPIRankWorld()>0:
            fn2=fn+".%s"%getMPIRankWorld()
        else:
            fn2=fn
        self.assertRaises(IOError,FileWriter,fn="",append=True, createLocalFiles=True)
        f=FileWriter(fn,append=False,createLocalFiles=True)
        self.assertTrue(f.name==fn, "wrong file name.")
        self.assertTrue(f.mode=='w', "wrong mode")
        self.assertTrue(f.newlines==os.linesep, "wrong line seps")
        self.assertTrue(not f.closed,"file shuold not be closed.")
        f.write("line1"+f.newlines)
        f.flush()
        self.__checkContent(fn2,["line1"])
        f.writelines(["line2"+f.newlines, "line3"+f.newlines])
        f.close()
        self.assertTrue(f.closed,"file shuold be closed.")
        self.__checkContent(fn2,["line1", "line2", "line3"])
        os.unlink(fn2)

#=========================================================
#  constants
#=========================================================
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_kronecker_1(self):
      val=kronecker(d=1)
      self.assertTrue(isinstance(val,numpy.ndarray),"wrong type of result.")
      self.assertEqual(val[0,0],1.0,"wrong value for (0,0)")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_kronecker_2(self):
      val=kronecker(d=2)
      self.assertTrue(isinstance(val,numpy.ndarray),"wrong type of result.")
      self.assertEqual(val[0,0],1.0,"wrong value for (0,0)")
      self.assertEqual(val[0,1],0.0,"wrong value for (0,1)")
      self.assertEqual(val[1,0],0.0,"wrong value for (1,0)")
      self.assertEqual(val[1,1],1.0,"wrong value for (1,1)")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_kronecker_3(self):
      val=kronecker(d=3)
      self.assertTrue(isinstance(val,numpy.ndarray),"wrong type of result.")
      self.assertEqual(val[0,0],1.0,"wrong value for (0,0)")
      self.assertEqual(val[0,1],0.0,"wrong value for (0,1)")
      self.assertEqual(val[0,2],0.0,"wrong value for (0,2)")
      self.assertEqual(val[1,0],0.0,"wrong value for (1,0)")
      self.assertEqual(val[1,1],1.0,"wrong value for (1,1)")
      self.assertEqual(val[1,2],0.0,"wrong value for (1,2)")
      self.assertEqual(val[2,0],0.0,"wrong value for (2,0)")
      self.assertEqual(val[2,1],0.0,"wrong value for (2,1)")
      self.assertEqual(val[2,2],1.0,"wrong value for (2,2)")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_kronecker_domain(self):
      val=kronecker(d=self.domain)
      self.assertTrue(isinstance(val,numpy.ndarray),"wrong type of result.")
      self.assertEqual(val.shape,(self.domain.getDim(),self.domain.getDim()),"wrong shape.")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_kronecker_functionspace(self):
      val=kronecker(d=self.functionspace)
      self.assertTrue(isinstance(val,escript.Data),"wrong type of result.")
      self.assertEqual(val.getShape(),(self.functionspace.getDim(),self.functionspace.getDim()),"wrong shape.")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_identityTensor_1(self):
      val=identityTensor(d=1)
      self.assertTrue(isinstance(val,numpy.ndarray),"wrong type of result.")
      self.assertEqual(val[0,0],1.0,"wrong value for (0,0)")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_identityTensor_2(self):
      val=identityTensor(d=2)
      self.assertTrue(isinstance(val,numpy.ndarray),"wrong type of result.")
      self.assertEqual(val[0,0],1.0,"wrong value for (0,0)")
      self.assertEqual(val[0,1],0.0,"wrong value for (0,1)")
      self.assertEqual(val[1,0],0.0,"wrong value for (1,0)")
      self.assertEqual(val[1,1],1.0,"wrong value for (1,1)")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_identityTensor_3(self):
      val=identityTensor(d=3)
      self.assertTrue(isinstance(val,numpy.ndarray),"wrong type of result.")
      self.assertEqual(val[0,0],1.0,"wrong value for (0,0)")
      self.assertEqual(val[0,1],0.0,"wrong value for (0,1)")
      self.assertEqual(val[0,2],0.0,"wrong value for (0,2)")
      self.assertEqual(val[1,0],0.0,"wrong value for (1,0)")
      self.assertEqual(val[1,1],1.0,"wrong value for (1,1)")
      self.assertEqual(val[1,2],0.0,"wrong value for (1,2)")
      self.assertEqual(val[2,0],0.0,"wrong value for (2,0)")
      self.assertEqual(val[2,1],0.0,"wrong value for (2,1)")
      self.assertEqual(val[2,2],1.0,"wrong value for (2,2)")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_identityTensor_domain(self):
      val=identityTensor(d=self.domain)
      self.assertTrue(isinstance(val,numpy.ndarray),"wrong type of result.")
      self.assertEqual(val.shape,(self.domain.getDim(),self.domain.getDim()),"wrong shape.")
      if self.domain.getDim()==2:
         self.assertEqual(val[0,0],1.0,"wrong value for (0,0)")
         self.assertEqual(val[0,1],0.0,"wrong value for (0,1)")
         self.assertEqual(val[1,0],0.0,"wrong value for (1,0)")
         self.assertEqual(val[1,1],1.0,"wrong value for (1,1)")
      if self.domain.getDim()==3:
         self.assertEqual(val[0,0],1.0,"wrong value for (0,0)")
         self.assertEqual(val[0,1],0.0,"wrong value for (0,1)")
         self.assertEqual(val[0,2],0.0,"wrong value for (0,2)")
         self.assertEqual(val[1,0],0.0,"wrong value for (1,0)")
         self.assertEqual(val[1,1],1.0,"wrong value for (1,1)")
         self.assertEqual(val[1,2],0.0,"wrong value for (1,2)")
         self.assertEqual(val[2,0],0.0,"wrong value for (2,0)")
         self.assertEqual(val[2,1],0.0,"wrong value for (2,1)")
         self.assertEqual(val[2,2],1.0,"wrong value for (2,2)")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_identityTensor_functionspace(self):
      val=identityTensor(d=self.functionspace)
      self.assertTrue(isinstance(val,escript.Data),"wrong type of result.")
      self.assertEqual(val.getShape(),(self.functionspace.getDim(),self.functionspace.getDim()),"wrong shape.")
      if self.domain.getDim()==2:
         self.assertTrue(Lsup(val[0,0]-1.0)<=self.RES_TOL,"wrong value for (0,0)")
         self.assertTrue(Lsup(val[0,1]-0.0)<=self.RES_TOL,"wrong value for (0,1)")
         self.assertTrue(Lsup(val[1,0]-0.0)<=self.RES_TOL,"wrong value for (1,0)")
         self.assertTrue(Lsup(val[1,1]-1.0)<=self.RES_TOL,"wrong value for (1,1)")
      if self.domain.getDim()==3:
         self.assertTrue(Lsup(val[0,0]-1.0)<=self.RES_TOL,"wrong value for (0,0)")
         self.assertTrue(Lsup(val[0,1]-0.0)<=self.RES_TOL,"wrong value for (0,1)")
         self.assertTrue(Lsup(val[0,2]-0.0)<=self.RES_TOL,"wrong value for (0,2)")
         self.assertTrue(Lsup(val[1,0]-0.0)<=self.RES_TOL,"wrong value for (1,0)")
         self.assertTrue(Lsup(val[1,1]-1.0)<=self.RES_TOL,"wrong value for (1,1)")
         self.assertTrue(Lsup(val[1,2]-0.0)<=self.RES_TOL,"wrong value for (1,2)")
         self.assertTrue(Lsup(val[2,0]-0.0)<=self.RES_TOL,"wrong value for (2,0)")
         self.assertTrue(Lsup(val[2,1]-0.0)<=self.RES_TOL,"wrong value for (2,1)")
         self.assertTrue(Lsup(val[2,2]-1.0)<=self.RES_TOL,"wrong value for (2,2)")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_identityTensor4_1(self):
      val=identityTensor4(d=1)
      self.assertTrue(isinstance(val,numpy.ndarray),"wrong type of result.")
      self.assertEqual(val[0,0,0,0],1.0,"wrong value for (0,0,0,0)")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_identityTensor4_2(self):
      val=identityTensor4(d=2)
      self.assertTrue(isinstance(val,numpy.ndarray),"wrong type of result.")
      self.assertEqual(val[0,0,0,0],1.0,"wrong value for (0,0,0,0)")
      self.assertEqual(val[0,0,0,1],0.0,"wrong value for (0,0,0,1)")
      self.assertEqual(val[0,0,1,0],0.0,"wrong value for (0,0,1,0)")
      self.assertEqual(val[0,0,1,1],0.0,"wrong value for (0,0,1,1)")
      self.assertEqual(val[0,1,0,0],0.0,"wrong value for (0,1,0,0)")
      self.assertEqual(val[0,1,0,1],1.0,"wrong value for (0,1,0,1)")
      self.assertEqual(val[0,1,1,0],0.0,"wrong value for (0,1,1,0)")
      self.assertEqual(val[0,1,1,1],0.0,"wrong value for (0,1,1,1)")
      self.assertEqual(val[1,0,0,0],0.0,"wrong value for (1,0,0,0)")
      self.assertEqual(val[1,0,0,1],0.0,"wrong value for (1,0,0,1)")
      self.assertEqual(val[1,0,1,0],1.0,"wrong value for (1,0,1,0)")
      self.assertEqual(val[1,0,1,1],0.0,"wrong value for (1,0,1,1)")
      self.assertEqual(val[1,1,0,0],0.0,"wrong value for (1,1,0,0)")
      self.assertEqual(val[1,1,0,1],0.0,"wrong value for (1,1,0,1)")
      self.assertEqual(val[1,1,1,0],0.0,"wrong value for (1,1,1,0)")
      self.assertEqual(val[1,1,1,1],1.0,"wrong value for (1,1,1,1)")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_identityTensor4_3(self):
      val=identityTensor4(d=3)
      self.assertTrue(isinstance(val,numpy.ndarray),"wrong type of result.")
      self.assertEqual(val[0,0,0,0],1.0,"wrong value for (0,0,0,0)")
      self.assertEqual(val[0,0,0,1],0.0,"wrong value for (0,0,0,1)")
      self.assertEqual(val[0,0,0,2],0.0,"wrong value for (0,0,0,2)")
      self.assertEqual(val[0,0,1,0],0.0,"wrong value for (0,0,1,0)")
      self.assertEqual(val[0,0,1,1],0.0,"wrong value for (0,0,1,1)")
      self.assertEqual(val[0,0,1,2],0.0,"wrong value for (0,0,1,2)")
      self.assertEqual(val[0,0,2,0],0.0,"wrong value for (0,0,2,0)")
      self.assertEqual(val[0,0,2,1],0.0,"wrong value for (0,0,2,1)")
      self.assertEqual(val[0,0,2,2],0.0,"wrong value for (0,0,2,2)")
      self.assertEqual(val[0,1,0,0],0.0,"wrong value for (0,1,0,0)")
      self.assertEqual(val[0,1,0,1],1.0,"wrong value for (0,1,0,1)")
      self.assertEqual(val[0,1,0,2],0.0,"wrong value for (0,1,0,2)")
      self.assertEqual(val[0,1,1,0],0.0,"wrong value for (0,1,1,0)")
      self.assertEqual(val[0,1,1,1],0.0,"wrong value for (0,1,1,1)")
      self.assertEqual(val[0,1,1,2],0.0,"wrong value for (0,1,1,2)")
      self.assertEqual(val[0,1,2,0],0.0,"wrong value for (0,1,2,0)")
      self.assertEqual(val[0,1,2,1],0.0,"wrong value for (0,1,2,1)")
      self.assertEqual(val[0,1,2,2],0.0,"wrong value for (0,1,2,2)")
      self.assertEqual(val[0,2,0,0],0.0,"wrong value for (0,2,0,0)")
      self.assertEqual(val[0,2,0,1],0.0,"wrong value for (0,2,0,1)")
      self.assertEqual(val[0,2,0,2],1.0,"wrong value for (0,2,0,2)")
      self.assertEqual(val[0,2,1,0],0.0,"wrong value for (0,2,1,0)")
      self.assertEqual(val[0,2,1,1],0.0,"wrong value for (0,2,1,1)")
      self.assertEqual(val[0,2,1,2],0.0,"wrong value for (0,2,1,2)")
      self.assertEqual(val[0,2,2,0],0.0,"wrong value for (0,2,2,0)")
      self.assertEqual(val[0,2,2,1],0.0,"wrong value for (0,2,2,1)")
      self.assertEqual(val[0,2,2,2],0.0,"wrong value for (0,2,2,2)")
      self.assertEqual(val[1,0,0,0],0.0,"wrong value for (1,0,0,0)")
      self.assertEqual(val[1,0,0,1],0.0,"wrong value for (1,0,0,1)")
      self.assertEqual(val[1,0,0,2],0.0,"wrong value for (1,0,0,2)")
      self.assertEqual(val[1,0,1,0],1.0,"wrong value for (1,0,1,0)")
      self.assertEqual(val[1,0,1,1],0.0,"wrong value for (1,0,1,1)")
      self.assertEqual(val[1,0,1,2],0.0,"wrong value for (1,0,1,2)")
      self.assertEqual(val[1,0,2,0],0.0,"wrong value for (1,0,2,0)")
      self.assertEqual(val[1,0,2,1],0.0,"wrong value for (1,0,2,1)")
      self.assertEqual(val[1,0,2,2],0.0,"wrong value for (1,0,2,2)")
      self.assertEqual(val[1,1,0,0],0.0,"wrong value for (1,1,0,0)")
      self.assertEqual(val[1,1,0,1],0.0,"wrong value for (1,1,0,1)")
      self.assertEqual(val[1,1,0,2],0.0,"wrong value for (1,1,0,2)")
      self.assertEqual(val[1,1,1,0],0.0,"wrong value for (1,1,1,0)")
      self.assertEqual(val[1,1,1,1],1.0,"wrong value for (1,1,1,1)")
      self.assertEqual(val[1,1,1,2],0.0,"wrong value for (1,1,1,2)")
      self.assertEqual(val[1,1,2,0],0.0,"wrong value for (1,1,2,0)")
      self.assertEqual(val[1,1,2,1],0.0,"wrong value for (1,1,2,1)")
      self.assertEqual(val[1,1,2,2],0.0,"wrong value for (1,1,2,2)")
      self.assertEqual(val[1,2,0,0],0.0,"wrong value for (1,2,0,0)")
      self.assertEqual(val[1,2,0,1],0.0,"wrong value for (1,2,0,1)")
      self.assertEqual(val[1,2,0,2],0.0,"wrong value for (1,2,0,2)")
      self.assertEqual(val[1,2,1,0],0.0,"wrong value for (1,2,1,0)")
      self.assertEqual(val[1,2,1,1],0.0,"wrong value for (1,2,1,1)")
      self.assertEqual(val[1,2,1,2],1.0,"wrong value for (1,2,1,2)")
      self.assertEqual(val[1,2,2,0],0.0,"wrong value for (1,2,2,0)")
      self.assertEqual(val[1,2,2,1],0.0,"wrong value for (1,2,2,1)")
      self.assertEqual(val[1,2,2,2],0.0,"wrong value for (1,2,2,2)")
      self.assertEqual(val[2,0,0,0],0.0,"wrong value for (2,0,0,0)")
      self.assertEqual(val[2,0,0,1],0.0,"wrong value for (2,0,0,1)")
      self.assertEqual(val[2,0,0,2],0.0,"wrong value for (2,0,0,2)")
      self.assertEqual(val[2,0,1,0],0.0,"wrong value for (2,0,1,0)")
      self.assertEqual(val[2,0,1,1],0.0,"wrong value for (2,0,1,1)")
      self.assertEqual(val[2,0,1,2],0.0,"wrong value for (2,0,1,2)")
      self.assertEqual(val[2,0,2,0],1.0,"wrong value for (2,0,2,0)")
      self.assertEqual(val[2,0,2,1],0.0,"wrong value for (2,0,2,1)")
      self.assertEqual(val[2,0,2,2],0.0,"wrong value for (2,0,2,2)")
      self.assertEqual(val[2,1,0,0],0.0,"wrong value for (2,1,0,0)")
      self.assertEqual(val[2,1,0,1],0.0,"wrong value for (2,1,0,1)")
      self.assertEqual(val[2,1,0,2],0.0,"wrong value for (2,1,0,2)")
      self.assertEqual(val[2,1,1,0],0.0,"wrong value for (2,1,1,0)")
      self.assertEqual(val[2,1,1,1],0.0,"wrong value for (2,1,1,1)")
      self.assertEqual(val[2,1,1,2],0.0,"wrong value for (2,1,1,2)")
      self.assertEqual(val[2,1,2,0],0.0,"wrong value for (2,1,2,0)")
      self.assertEqual(val[2,1,2,1],1.0,"wrong value for (2,1,2,1)")
      self.assertEqual(val[2,1,2,2],0.0,"wrong value for (2,1,2,2)")
      self.assertEqual(val[2,2,0,0],0.0,"wrong value for (2,2,0,0)")
      self.assertEqual(val[2,2,0,1],0.0,"wrong value for (2,2,0,1)")
      self.assertEqual(val[2,2,0,2],0.0,"wrong value for (2,2,0,2)")
      self.assertEqual(val[2,2,1,0],0.0,"wrong value for (2,2,1,0)")
      self.assertEqual(val[2,2,1,1],0.0,"wrong value for (2,2,1,1)")
      self.assertEqual(val[2,2,1,2],0.0,"wrong value for (2,2,1,2)")
      self.assertEqual(val[2,2,2,0],0.0,"wrong value for (2,2,2,0)")
      self.assertEqual(val[2,2,2,1],0.0,"wrong value for (2,2,2,1)")
      self.assertEqual(val[2,2,2,2],1.0,"wrong value for (2,2,2,2)")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_identityTensor4_domain(self):
      val=identityTensor4(d=self.domain)
      self.assertTrue(isinstance(val,numpy.ndarray),"wrong type of result.")
      self.assertEqual(val.shape,(self.domain.getDim(),self.domain.getDim(),self.domain.getDim(),self.domain.getDim()),"wrong shape.")
      if self.domain.getDim()==2:
         self.assertEqual(val[0,0,0,0],1.0,"wrong value for (0,0,0,0)")
         self.assertEqual(val[0,0,0,1],0.0,"wrong value for (0,0,0,1)")
         self.assertEqual(val[0,0,1,0],0.0,"wrong value for (0,0,1,0)")
         self.assertEqual(val[0,0,1,1],0.0,"wrong value for (0,0,1,1)")
         self.assertEqual(val[0,1,0,0],0.0,"wrong value for (0,1,0,0)")
         self.assertEqual(val[0,1,0,1],1.0,"wrong value for (0,1,0,1)")
         self.assertEqual(val[0,1,1,0],0.0,"wrong value for (0,1,1,0)")
         self.assertEqual(val[0,1,1,1],0.0,"wrong value for (0,1,1,1)")
         self.assertEqual(val[1,0,0,0],0.0,"wrong value for (1,0,0,0)")
         self.assertEqual(val[1,0,0,1],0.0,"wrong value for (1,0,0,1)")
         self.assertEqual(val[1,0,1,0],1.0,"wrong value for (1,0,1,0)")
         self.assertEqual(val[1,0,1,1],0.0,"wrong value for (1,0,1,1)")
         self.assertEqual(val[1,1,0,0],0.0,"wrong value for (1,1,0,0)")
         self.assertEqual(val[1,1,0,1],0.0,"wrong value for (1,1,0,1)")
         self.assertEqual(val[1,1,1,0],0.0,"wrong value for (1,1,1,0)")
         self.assertEqual(val[1,1,1,1],1.0,"wrong value for (1,1,1,1)")
      if self.domain.getDim()==3:
         self.assertEqual(val[0,0,0,0],1.0,"wrong value for (0,0,0,0)")
         self.assertEqual(val[0,0,0,1],0.0,"wrong value for (0,0,0,1)")
         self.assertEqual(val[0,0,0,2],0.0,"wrong value for (0,0,0,2)")
         self.assertEqual(val[0,0,1,0],0.0,"wrong value for (0,0,1,0)")
         self.assertEqual(val[0,0,1,1],0.0,"wrong value for (0,0,1,1)")
         self.assertEqual(val[0,0,1,2],0.0,"wrong value for (0,0,1,2)")
         self.assertEqual(val[0,0,2,0],0.0,"wrong value for (0,0,2,0)")
         self.assertEqual(val[0,0,2,1],0.0,"wrong value for (0,0,2,1)")
         self.assertEqual(val[0,0,2,2],0.0,"wrong value for (0,0,2,2)")
         self.assertEqual(val[0,1,0,0],0.0,"wrong value for (0,1,0,0)")
         self.assertEqual(val[0,1,0,1],1.0,"wrong value for (0,1,0,1)")
         self.assertEqual(val[0,1,0,2],0.0,"wrong value for (0,1,0,2)")
         self.assertEqual(val[0,1,1,0],0.0,"wrong value for (0,1,1,0)")
         self.assertEqual(val[0,1,1,1],0.0,"wrong value for (0,1,1,1)")
         self.assertEqual(val[0,1,1,2],0.0,"wrong value for (0,1,1,2)")
         self.assertEqual(val[0,1,2,0],0.0,"wrong value for (0,1,2,0)")
         self.assertEqual(val[0,1,2,1],0.0,"wrong value for (0,1,2,1)")
         self.assertEqual(val[0,1,2,2],0.0,"wrong value for (0,1,2,2)")
         self.assertEqual(val[0,2,0,0],0.0,"wrong value for (0,2,0,0)")
         self.assertEqual(val[0,2,0,1],0.0,"wrong value for (0,2,0,1)")
         self.assertEqual(val[0,2,0,2],1.0,"wrong value for (0,2,0,2)")
         self.assertEqual(val[0,2,1,0],0.0,"wrong value for (0,2,1,0)")
         self.assertEqual(val[0,2,1,1],0.0,"wrong value for (0,2,1,1)")
         self.assertEqual(val[0,2,1,2],0.0,"wrong value for (0,2,1,2)")
         self.assertEqual(val[0,2,2,0],0.0,"wrong value for (0,2,2,0)")
         self.assertEqual(val[0,2,2,1],0.0,"wrong value for (0,2,2,1)")
         self.assertEqual(val[0,2,2,2],0.0,"wrong value for (0,2,2,2)")
         self.assertEqual(val[1,0,0,0],0.0,"wrong value for (1,0,0,0)")
         self.assertEqual(val[1,0,0,1],0.0,"wrong value for (1,0,0,1)")
         self.assertEqual(val[1,0,0,2],0.0,"wrong value for (1,0,0,2)")
         self.assertEqual(val[1,0,1,0],1.0,"wrong value for (1,0,1,0)")
         self.assertEqual(val[1,0,1,1],0.0,"wrong value for (1,0,1,1)")
         self.assertEqual(val[1,0,1,2],0.0,"wrong value for (1,0,1,2)")
         self.assertEqual(val[1,0,2,0],0.0,"wrong value for (1,0,2,0)")
         self.assertEqual(val[1,0,2,1],0.0,"wrong value for (1,0,2,1)")
         self.assertEqual(val[1,0,2,2],0.0,"wrong value for (1,0,2,2)")
         self.assertEqual(val[1,1,0,0],0.0,"wrong value for (1,1,0,0)")
         self.assertEqual(val[1,1,0,1],0.0,"wrong value for (1,1,0,1)")
         self.assertEqual(val[1,1,0,2],0.0,"wrong value for (1,1,0,2)")
         self.assertEqual(val[1,1,1,0],0.0,"wrong value for (1,1,1,0)")
         self.assertEqual(val[1,1,1,1],1.0,"wrong value for (1,1,1,1)")
         self.assertEqual(val[1,1,1,2],0.0,"wrong value for (1,1,1,2)")
         self.assertEqual(val[1,1,2,0],0.0,"wrong value for (1,1,2,0)")
         self.assertEqual(val[1,1,2,1],0.0,"wrong value for (1,1,2,1)")
         self.assertEqual(val[1,1,2,2],0.0,"wrong value for (1,1,2,2)")
         self.assertEqual(val[1,2,0,0],0.0,"wrong value for (1,2,0,0)")
         self.assertEqual(val[1,2,0,1],0.0,"wrong value for (1,2,0,1)")
         self.assertEqual(val[1,2,0,2],0.0,"wrong value for (1,2,0,2)")
         self.assertEqual(val[1,2,1,0],0.0,"wrong value for (1,2,1,0)")
         self.assertEqual(val[1,2,1,1],0.0,"wrong value for (1,2,1,1)")
         self.assertEqual(val[1,2,1,2],1.0,"wrong value for (1,2,1,2)")
         self.assertEqual(val[1,2,2,0],0.0,"wrong value for (1,2,2,0)")
         self.assertEqual(val[1,2,2,1],0.0,"wrong value for (1,2,2,1)")
         self.assertEqual(val[1,2,2,2],0.0,"wrong value for (1,2,2,2)")
         self.assertEqual(val[2,0,0,0],0.0,"wrong value for (2,0,0,0)")
         self.assertEqual(val[2,0,0,1],0.0,"wrong value for (2,0,0,1)")
         self.assertEqual(val[2,0,0,2],0.0,"wrong value for (2,0,0,2)")
         self.assertEqual(val[2,0,1,0],0.0,"wrong value for (2,0,1,0)")
         self.assertEqual(val[2,0,1,1],0.0,"wrong value for (2,0,1,1)")
         self.assertEqual(val[2,0,1,2],0.0,"wrong value for (2,0,1,2)")
         self.assertEqual(val[2,0,2,0],1.0,"wrong value for (2,0,2,0)")
         self.assertEqual(val[2,0,2,1],0.0,"wrong value for (2,0,2,1)")
         self.assertEqual(val[2,0,2,2],0.0,"wrong value for (2,0,2,2)")
         self.assertEqual(val[2,1,0,0],0.0,"wrong value for (2,1,0,0)")
         self.assertEqual(val[2,1,0,1],0.0,"wrong value for (2,1,0,1)")
         self.assertEqual(val[2,1,0,2],0.0,"wrong value for (2,1,0,2)")
         self.assertEqual(val[2,1,1,0],0.0,"wrong value for (2,1,1,0)")
         self.assertEqual(val[2,1,1,1],0.0,"wrong value for (2,1,1,1)")
         self.assertEqual(val[2,1,1,2],0.0,"wrong value for (2,1,1,2)")
         self.assertEqual(val[2,1,2,0],0.0,"wrong value for (2,1,2,0)")
         self.assertEqual(val[2,1,2,1],1.0,"wrong value for (2,1,2,1)")
         self.assertEqual(val[2,1,2,2],0.0,"wrong value for (2,1,2,2)")
         self.assertEqual(val[2,2,0,0],0.0,"wrong value for (2,2,0,0)")
         self.assertEqual(val[2,2,0,1],0.0,"wrong value for (2,2,0,1)")
         self.assertEqual(val[2,2,0,2],0.0,"wrong value for (2,2,0,2)")
         self.assertEqual(val[2,2,1,0],0.0,"wrong value for (2,2,1,0)")
         self.assertEqual(val[2,2,1,1],0.0,"wrong value for (2,2,1,1)")
         self.assertEqual(val[2,2,1,2],0.0,"wrong value for (2,2,1,2)")
         self.assertEqual(val[2,2,2,0],0.0,"wrong value for (2,2,2,0)")
         self.assertEqual(val[2,2,2,1],0.0,"wrong value for (2,2,2,1)")
         self.assertEqual(val[2,2,2,2],1.0,"wrong value for (2,2,2,2)")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_identityTensor4_functionspace(self):
      val=identityTensor4(d=self.functionspace)
      self.assertTrue(isinstance(val,escript.Data),"wrong type of result.")
      self.assertEqual(val.getShape(),(self.functionspace.getDim(),self.functionspace.getDim(),self.functionspace.getDim(),self.functionspace.getDim()),"wrong shape.")
      if self.domain.getDim()==2:
         self.assertTrue(Lsup(val[0,0,0,0]-1.0)<=self.RES_TOL,"wrong value for (0,0,0,0)")
         self.assertTrue(Lsup(val[0,0,0,1]-0.0)<=self.RES_TOL,"wrong value for (0,0,0,1)")
         self.assertTrue(Lsup(val[0,0,1,0]-0.0)<=self.RES_TOL,"wrong value for (0,0,1,0)")
         self.assertTrue(Lsup(val[0,0,1,1]-0.0)<=self.RES_TOL,"wrong value for (0,0,1,1)")
         self.assertTrue(Lsup(val[0,1,0,0]-0.0)<=self.RES_TOL,"wrong value for (0,1,0,0)")
         self.assertTrue(Lsup(val[0,1,0,1]-1.0)<=self.RES_TOL,"wrong value for (0,1,0,1)")
         self.assertTrue(Lsup(val[0,1,1,0]-0.0)<=self.RES_TOL,"wrong value for (0,1,1,0)")
         self.assertTrue(Lsup(val[0,1,1,1]-0.0)<=self.RES_TOL,"wrong value for (0,1,1,1)")
         self.assertTrue(Lsup(val[1,0,0,0]-0.0)<=self.RES_TOL,"wrong value for (1,0,0,0)")
         self.assertTrue(Lsup(val[1,0,0,1]-0.0)<=self.RES_TOL,"wrong value for (1,0,0,1)")
         self.assertTrue(Lsup(val[1,0,1,0]-1.0)<=self.RES_TOL,"wrong value for (1,0,1,0)")
         self.assertTrue(Lsup(val[1,0,1,1]-0.0)<=self.RES_TOL,"wrong value for (1,0,1,1)")
         self.assertTrue(Lsup(val[1,1,0,0]-0.0)<=self.RES_TOL,"wrong value for (1,1,0,0)")
         self.assertTrue(Lsup(val[1,1,0,1]-0.0)<=self.RES_TOL,"wrong value for (1,1,0,1)")
         self.assertTrue(Lsup(val[1,1,1,0]-0.0)<=self.RES_TOL,"wrong value for (1,1,1,0)")
         self.assertTrue(Lsup(val[1,1,1,1]-1.0)<=self.RES_TOL,"wrong value for (1,1,1,1)")
      if self.domain.getDim()==3:
         self.assertTrue(Lsup(val[0,0,0,0]-1.0)<=self.RES_TOL,"wrong value for (0,0,0,0)")
         self.assertTrue(Lsup(val[0,0,0,1]-0.0)<=self.RES_TOL,"wrong value for (0,0,0,1)")
         self.assertTrue(Lsup(val[0,0,0,2]-0.0)<=self.RES_TOL,"wrong value for (0,0,0,2)")
         self.assertTrue(Lsup(val[0,0,1,0]-0.0)<=self.RES_TOL,"wrong value for (0,0,1,0)")
         self.assertTrue(Lsup(val[0,0,1,1]-0.0)<=self.RES_TOL,"wrong value for (0,0,1,1)")
         self.assertTrue(Lsup(val[0,0,1,2]-0.0)<=self.RES_TOL,"wrong value for (0,0,1,2)")
         self.assertTrue(Lsup(val[0,0,2,0]-0.0)<=self.RES_TOL,"wrong value for (0,0,2,0)")
         self.assertTrue(Lsup(val[0,0,2,1]-0.0)<=self.RES_TOL,"wrong value for (0,0,2,1)")
         self.assertTrue(Lsup(val[0,0,2,2]-0.0)<=self.RES_TOL,"wrong value for (0,0,2,2)")
         self.assertTrue(Lsup(val[0,1,0,0]-0.0)<=self.RES_TOL,"wrong value for (0,1,0,0)")
         self.assertTrue(Lsup(val[0,1,0,1]-1.0)<=self.RES_TOL,"wrong value for (0,1,0,1)")
         self.assertTrue(Lsup(val[0,1,0,2]-0.0)<=self.RES_TOL,"wrong value for (0,1,0,2)")
         self.assertTrue(Lsup(val[0,1,1,0]-0.0)<=self.RES_TOL,"wrong value for (0,1,1,0)")
         self.assertTrue(Lsup(val[0,1,1,1]-0.0)<=self.RES_TOL,"wrong value for (0,1,1,1)")
         self.assertTrue(Lsup(val[0,1,1,2]-0.0)<=self.RES_TOL,"wrong value for (0,1,1,2)")
         self.assertTrue(Lsup(val[0,1,2,0]-0.0)<=self.RES_TOL,"wrong value for (0,1,2,0)")
         self.assertTrue(Lsup(val[0,1,2,1]-0.0)<=self.RES_TOL,"wrong value for (0,1,2,1)")
         self.assertTrue(Lsup(val[0,1,2,2]-0.0)<=self.RES_TOL,"wrong value for (0,1,2,2)")
         self.assertTrue(Lsup(val[0,2,0,0]-0.0)<=self.RES_TOL,"wrong value for (0,2,0,0)")
         self.assertTrue(Lsup(val[0,2,0,1]-0.0)<=self.RES_TOL,"wrong value for (0,2,0,1)")
         self.assertTrue(Lsup(val[0,2,0,2]-1.0)<=self.RES_TOL,"wrong value for (0,2,0,2)")
         self.assertTrue(Lsup(val[0,2,1,0]-0.0)<=self.RES_TOL,"wrong value for (0,2,1,0)")
         self.assertTrue(Lsup(val[0,2,1,1]-0.0)<=self.RES_TOL,"wrong value for (0,2,1,1)")
         self.assertTrue(Lsup(val[0,2,1,2]-0.0)<=self.RES_TOL,"wrong value for (0,2,1,2)")
         self.assertTrue(Lsup(val[0,2,2,0]-0.0)<=self.RES_TOL,"wrong value for (0,2,2,0)")
         self.assertTrue(Lsup(val[0,2,2,1]-0.0)<=self.RES_TOL,"wrong value for (0,2,2,1)")
         self.assertTrue(Lsup(val[0,2,2,2]-0.0)<=self.RES_TOL,"wrong value for (0,2,2,2)")
         self.assertTrue(Lsup(val[1,0,0,0]-0.0)<=self.RES_TOL,"wrong value for (1,0,0,0)")
         self.assertTrue(Lsup(val[1,0,0,1]-0.0)<=self.RES_TOL,"wrong value for (1,0,0,1)")
         self.assertTrue(Lsup(val[1,0,0,2]-0.0)<=self.RES_TOL,"wrong value for (1,0,0,2)")
         self.assertTrue(Lsup(val[1,0,1,0]-1.0)<=self.RES_TOL,"wrong value for (1,0,1,0)")
         self.assertTrue(Lsup(val[1,0,1,1]-0.0)<=self.RES_TOL,"wrong value for (1,0,1,1)")
         self.assertTrue(Lsup(val[1,0,1,2]-0.0)<=self.RES_TOL,"wrong value for (1,0,1,2)")
         self.assertTrue(Lsup(val[1,0,2,0]-0.0)<=self.RES_TOL,"wrong value for (1,0,2,0)")
         self.assertTrue(Lsup(val[1,0,2,1]-0.0)<=self.RES_TOL,"wrong value for (1,0,2,1)")
         self.assertTrue(Lsup(val[1,0,2,2]-0.0)<=self.RES_TOL,"wrong value for (1,0,2,2)")
         self.assertTrue(Lsup(val[1,1,0,0]-0.0)<=self.RES_TOL,"wrong value for (1,1,0,0)")
         self.assertTrue(Lsup(val[1,1,0,1]-0.0)<=self.RES_TOL,"wrong value for (1,1,0,1)")
         self.assertTrue(Lsup(val[1,1,0,2]-0.0)<=self.RES_TOL,"wrong value for (1,1,0,2)")
         self.assertTrue(Lsup(val[1,1,1,0]-0.0)<=self.RES_TOL,"wrong value for (1,1,1,0)")
         self.assertTrue(Lsup(val[1,1,1,1]-1.0)<=self.RES_TOL,"wrong value for (1,1,1,1)")
         self.assertTrue(Lsup(val[1,1,1,2]-0.0)<=self.RES_TOL,"wrong value for (1,1,1,2)")
         self.assertTrue(Lsup(val[1,1,2,0]-0.0)<=self.RES_TOL,"wrong value for (1,1,2,0)")
         self.assertTrue(Lsup(val[1,1,2,1]-0.0)<=self.RES_TOL,"wrong value for (1,1,2,1)")
         self.assertTrue(Lsup(val[1,1,2,2]-0.0)<=self.RES_TOL,"wrong value for (1,1,2,2)")
         self.assertTrue(Lsup(val[1,2,0,0]-0.0)<=self.RES_TOL,"wrong value for (1,2,0,0)")
         self.assertTrue(Lsup(val[1,2,0,1]-0.0)<=self.RES_TOL,"wrong value for (1,2,0,1)")
         self.assertTrue(Lsup(val[1,2,0,2]-0.0)<=self.RES_TOL,"wrong value for (1,2,0,2)")
         self.assertTrue(Lsup(val[1,2,1,0]-0.0)<=self.RES_TOL,"wrong value for (1,2,1,0)")
         self.assertTrue(Lsup(val[1,2,1,1]-0.0)<=self.RES_TOL,"wrong value for (1,2,1,1)")
         self.assertTrue(Lsup(val[1,2,1,2]-1.0)<=self.RES_TOL,"wrong value for (1,2,1,2)")
         self.assertTrue(Lsup(val[1,2,2,0]-0.0)<=self.RES_TOL,"wrong value for (1,2,2,0)")
         self.assertTrue(Lsup(val[1,2,2,1]-0.0)<=self.RES_TOL,"wrong value for (1,2,2,1)")
         self.assertTrue(Lsup(val[1,2,2,2]-0.0)<=self.RES_TOL,"wrong value for (1,2,2,2)")
         self.assertTrue(Lsup(val[2,0,0,0]-0.0)<=self.RES_TOL,"wrong value for (2,0,0,0)")
         self.assertTrue(Lsup(val[2,0,0,1]-0.0)<=self.RES_TOL,"wrong value for (2,0,0,1)")
         self.assertTrue(Lsup(val[2,0,0,2]-0.0)<=self.RES_TOL,"wrong value for (2,0,0,2)")
         self.assertTrue(Lsup(val[2,0,1,0]-0.0)<=self.RES_TOL,"wrong value for (2,0,1,0)")
         self.assertTrue(Lsup(val[2,0,1,1]-0.0)<=self.RES_TOL,"wrong value for (2,0,1,1)")
         self.assertTrue(Lsup(val[2,0,1,2]-0.0)<=self.RES_TOL,"wrong value for (2,0,1,2)")
         self.assertTrue(Lsup(val[2,0,2,0]-1.0)<=self.RES_TOL,"wrong value for (2,0,2,0)")
         self.assertTrue(Lsup(val[2,0,2,1]-0.0)<=self.RES_TOL,"wrong value for (2,0,2,1)")
         self.assertTrue(Lsup(val[2,0,2,2]-0.0)<=self.RES_TOL,"wrong value for (2,0,2,2)")
         self.assertTrue(Lsup(val[2,1,0,0]-0.0)<=self.RES_TOL,"wrong value for (2,1,0,0)")
         self.assertTrue(Lsup(val[2,1,0,1]-0.0)<=self.RES_TOL,"wrong value for (2,1,0,1)")
         self.assertTrue(Lsup(val[2,1,0,2]-0.0)<=self.RES_TOL,"wrong value for (2,1,0,2)")
         self.assertTrue(Lsup(val[2,1,1,0]-0.0)<=self.RES_TOL,"wrong value for (2,1,1,0)")
         self.assertTrue(Lsup(val[2,1,1,1]-0.0)<=self.RES_TOL,"wrong value for (2,1,1,1)")
         self.assertTrue(Lsup(val[2,1,1,2]-0.0)<=self.RES_TOL,"wrong value for (2,1,1,2)")
         self.assertTrue(Lsup(val[2,1,2,0]-0.0)<=self.RES_TOL,"wrong value for (2,1,2,0)")
         self.assertTrue(Lsup(val[2,1,2,1]-1.0)<=self.RES_TOL,"wrong value for (2,1,2,1)")
         self.assertTrue(Lsup(val[2,1,2,2]-0.0)<=self.RES_TOL,"wrong value for (2,1,2,2)")
         self.assertTrue(Lsup(val[2,2,0,0]-0.0)<=self.RES_TOL,"wrong value for (2,2,0,0)")
         self.assertTrue(Lsup(val[2,2,0,1]-0.0)<=self.RES_TOL,"wrong value for (2,2,0,1)")
         self.assertTrue(Lsup(val[2,2,0,2]-0.0)<=self.RES_TOL,"wrong value for (2,2,0,2)")
         self.assertTrue(Lsup(val[2,2,1,0]-0.0)<=self.RES_TOL,"wrong value for (2,2,1,0)")
         self.assertTrue(Lsup(val[2,2,1,1]-0.0)<=self.RES_TOL,"wrong value for (2,2,1,1)")
         self.assertTrue(Lsup(val[2,2,1,2]-0.0)<=self.RES_TOL,"wrong value for (2,2,1,2)")
         self.assertTrue(Lsup(val[2,2,2,0]-0.0)<=self.RES_TOL,"wrong value for (2,2,2,0)")
         self.assertTrue(Lsup(val[2,2,2,1]-0.0)<=self.RES_TOL,"wrong value for (2,2,2,1)")
         self.assertTrue(Lsup(val[2,2,2,2]-1.0)<=self.RES_TOL,"wrong value for (2,2,2,2)")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_unitVector_1(self):
      val=unitVector(i=0,d=1)
      self.assertEqual(val[0],1.0,"wrong value for 0 in the 0 vector")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_unitVector_2(self):
      val=unitVector(i=0,d=2)
      self.assertEqual(val[0],1.0,"wrong value for 0 in the 0 vector")
      self.assertEqual(val[1],0.0,"wrong value for 1 in the 0 vector")
      val=unitVector(i=1,d=2)
      self.assertEqual(val[0],0.0,"wrong value for 0 in the 1 vector")
      self.assertEqual(val[1],1.0,"wrong value for 1 in the 1 vector")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_unitVector_3(self):
      val=unitVector(i=0,d=3)
      self.assertEqual(val[0],1.0,"wrong value for 0 in the 0 vector")
      self.assertEqual(val[1],0.0,"wrong value for 1 in the 0 vector")
      self.assertEqual(val[2],0.0,"wrong value for 2 in the 0 vector")
      val=unitVector(i=1,d=3)
      self.assertEqual(val[0],0.0,"wrong value for 0 in the 1 vector")
      self.assertEqual(val[1],1.0,"wrong value for 1 in the 1 vector")
      self.assertEqual(val[2],0.0,"wrong value for 2 in the 1 vector")
      val=unitVector(i=2,d=3)
      self.assertEqual(val[0],0.0,"wrong value for 0 in the 2 vector")
      self.assertEqual(val[1],0.0,"wrong value for 1 in the 2 vector")
      self.assertEqual(val[2],1.0,"wrong value for 2 in the 2 vector")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_unitVector_domain(self):
      val=unitVector(i=0,d=self.domain)
      self.assertTrue(isinstance(val,numpy.ndarray),"wrong type of result.")
      self.assertEqual(val.shape,(self.domain.getDim(),),"wrong shape.")
      if self.domain.getDim()==2:
         self.assertEqual(val[0],1.0,"wrong value for 0 in the 0 vector")
         self.assertEqual(val[1],0.0,"wrong value for 1 in the 0 vector")
      if self.domain.getDim()==3:
         self.assertEqual(val[0],1.0,"wrong value for 0 in the 0 vector")
         self.assertEqual(val[1],0.0,"wrong value for 1 in the 0 vector")
         self.assertEqual(val[2],0.0,"wrong value for 2 in the 0 vector")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_unitVector_functionspace(self):
      val=unitVector(i=0,d=self.functionspace)
      self.assertTrue(isinstance(val,escript.Data),"wrong type of result.")
      self.assertEqual(val.getShape(),(self.functionspace.getDim(),),"wrong shape.")
      if self.domain.getDim()==2:
         self.assertTrue(Lsup(val[0]-1.0)<=self.RES_TOL,"wrong value for 0 in the 0 vector")
         self.assertTrue(Lsup(val[1]-0.0)<=self.RES_TOL,"wrong value for 1 in the 0 vector")
      if self.domain.getDim()==3:
         self.assertTrue(Lsup(val[0]-1.0)<=self.RES_TOL,"wrong value for 0 in the 0 vector")
         self.assertTrue(Lsup(val[1]-0.0)<=self.RES_TOL,"wrong value for 1 in the 0 vector")
         self.assertTrue(Lsup(val[2]-0.0)<=self.RES_TOL,"wrong value for 2 in the 0 vector")

