#include <getopt.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/resource.h>
#include <sys/utsname.h>
#include <sys/wait.h>

#include "config.h"
#include "printerconf.h"

/*============================================================

  pconf_detect

  A simple command-line utility that uses the libprinterconf
  library to autodetect printers on parallel ports (using 
  parport) or networks (using SNMP).

============================================================*/


/*==========================================================*/
void usage( char *binname, int rc) {
/*------------------------------------------------------------
  Print usage and exit.
------------------------------------------------------------*/
  /* ITS4: ignore */
  fprintf (stderr, "\nUsage:\n\n"
	   "\t%s -m [PPORT|NETWORK|USB] -i <info-string>\n\n"
	   "-m is the detection method\n"
	   "-i is a string of method-specific information:\n\n"
	   "PPORT example:\n\n"
	   "\t%s -m PPORT -i 0,1\n\n"
	   "where \"0,1\" is a comma-separated list of\n"
	   "the parallel port to scan.\n"
	   "\n"
	   "NETWORK example:\n\n"
	   "\t%s -m NETWORK -i \"10.203.1.2/24\"\n\n"
	   "where \"10.203.1.2/24\" is the IP/bitmask to scan.\n\n"
	   "This can also be a range (\"10.203.1.12-30\"),\n"
	   "an IP/netmask (\"10.203.1.2/255.255.255.0\"),\n"
	   "a simple IP address (\"10.203.1.2\"),\n"
	   "or a host name (\"myprinter.mydomain.com\").\n"
	   "USB example:\n\n"
	   "\t%s -m USB -i 0,1\n\n"
	   "where \"0,1\" is a comma-seperated list of \n"
	   "the usb printer ports to scan.\n"
	   "\n", 
	   basename (binname),
	   basename (binname),
	   basename (binname),
	   basename (binname));

  exit (rc);
}

/*==============================================================*/
int load_parport_probe() {
/*----------------------------------------------------------------
  Attempt to load the parport_probe module. In kernel 2.2, this 
  will populate the /proc/parport entries.
----------------------------------------------------------------*/
  static char path[]="/sbin:/usr/sbin:/bin:/usr/bin";
  pid_t child;
  int status;

  switch(child=fork()){
  case 0:
    setenv("PATH",path,1);
    execlp("modprobe","modprobe","-r","parport_probe",
	   NULL);
    exit(19);
  case -1:
    return -1;
  default:
    wait4(child,&status,0,NULL);
    switch(WEXITSTATUS(status)){
    case 0:
      break;
    case 19:
      return -2;
    case 255: // modprobe seems to return 255 when it fails
    default:
      return 0;
    }
  }

  switch(child=fork()){
  case 0:
    setenv("PATH",path,1);
    execlp("modprobe","modprobe","parport_probe",NULL);
    exit(19);
  case -1:
    return -1;
  default:
    wait4(child,&status,0,NULL);
    switch(WEXITSTATUS(status)){
    case 0:
      break;
    case 19:
      return -2;
    case 255: // modprobe seems to return 255 when it fails
    default:
      return 0;
    }
  }
  return 1;
}

/*==========================================================*/
int main(int argc, char *argv[]) {
/*------------------------------------------------------------
  Parse the command line, then attempt detection 
------------------------------------------------------------*/
  char **printers = NULL;
  char **current = NULL;

  int failed = 0;
  int method=PCONF_DETECT_PPORT;
  char *info = NULL;
  int i;
  int port;
  int rc;
  int curarg;
  int retval = 0;
  static const struct option long_options[]={
    {"method",required_argument,NULL,'m'},
    {"info",required_argument,NULL,'i'},
    {"help",no_argument,NULL,'h'},
    {"version",no_argument,NULL,'V'}
  };
  struct utsname unam;
  int detect_style;

  uname(&unam);
  /* If this is a problem for you and you want this utility to
     run on your non-linux operating system feel free to submit
     a patch. -ben July 17, 2001 */
  if(strcmp("Linux",unam.sysname)){ // this only works on linux
    fprintf(stderr,"Error: this utility currently only works with linux\n");
    exit(1);
  }
  /* As far as I can tell there is currently no reason to make it
     work with any other version of linux. There is no V3 and the 
     1.x series has been gone for years. */
  if(unam.release[0]!='2'){ 
    fprintf(stderr,"Error this utility currently only works with linux "
	    "version 2\n");
    exit(2);
  }
  /* This test will probably have to be cleaned up a bit in the
     future. Right now I'm assuming that <2.2 kernels work one way
     and <2.3 do it the different way. */
  detect_style=unam.release[2]<'3'?2:4;

  while((curarg=getopt_long(argc,argv,"m:i:hV",long_options,NULL))!=EOF){
    switch(curarg){
    case 'm':
      if (! strcmp(optarg, "PPORT"))
	method = PCONF_DETECT_PPORT;
      else if (! strcmp(optarg, "NETWORK"))
	method = PCONF_DETECT_NETWORK;
      else if (! strcmp(optarg, "USB"))
	method = PCONF_DETECT_USB;
      /*
	else if (! strcmp(argv[i+1], "HPMULTI"))
	method = PCONF_DETECT_HPMULTI;
      */
      else 
	failed = 1;
      break;
    case 'i':
      info = (char *) strdup (optarg);
      break;
    case 'h':
      usage(argv[0],0);
    case 'V':
      printf("%s\n",VERSION);
      exit(0);
    default:
      fputs ("Invalid argument.\n",stderr);
      usage (argv[0], 1);
      exit(1);
    }
  }

  if (! info) 
    failed = 1;

  if (failed) {
    fputs ("Invalid arguments.\n",stderr);
    usage (argv[0], 1);
  }

  switch (method) {
  case PCONF_DETECT_PPORT:
    /* First, attempt to load parport_probe module if we are
       running a <=2.2 kernel*/ 
    if(detect_style==2){
      if(getuid()==0){
	switch(rc=load_parport_probe()){
	case 0:
	  fputs("parport_probe module does not seem to be built in your "
		"current  kernel.\n",stderr);
	  break;
	case -1:
	  fputs("could not fork modprobe\n",stderr);
	  break;
	case -2:
	  fputs("could not exec modprobe\n",stderr);
	  break;
	}
      }else{
	fputs("Warning: skipping loading of parport_probe kernel module.\n"
	      "Kernel module loading can only be done as root.\n",stderr);
      } 
    }

    port = atoi (info);
    printers = (char **) pconf_detect_printer (method, info, &retval);

    if (retval != 0) return 1;

    for (current=printers; *current; current++)
      printf ("%s\n", *current);/* ITS4: ignore */

    break;

  case PCONF_DETECT_NETWORK:
  case PCONF_DETECT_USB:
    printers = (char **) pconf_detect_printer (method, info, &retval);

    if (retval != 0) return 1;

    if (printers)
      for (current=printers; *current; current++)
	printf ("%s\n", *current);/* ITS4: ignore */

    break;
  }

  return 0;

}




