﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/chime-sdk-messaging/ChimeSDKMessagingRequest.h>
#include <aws/chime-sdk-messaging/ChimeSDKMessaging_EXPORTS.h>
#include <aws/chime-sdk-messaging/model/Processor.h>
#include <aws/chime-sdk-messaging/model/Tag.h>
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace ChimeSDKMessaging {
namespace Model {

/**
 */
class CreateChannelFlowRequest : public ChimeSDKMessagingRequest {
 public:
  AWS_CHIMESDKMESSAGING_API CreateChannelFlowRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateChannelFlow"; }

  AWS_CHIMESDKMESSAGING_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The ARN of the channel flow request.</p>
   */
  inline const Aws::String& GetAppInstanceArn() const { return m_appInstanceArn; }
  inline bool AppInstanceArnHasBeenSet() const { return m_appInstanceArnHasBeenSet; }
  template <typename AppInstanceArnT = Aws::String>
  void SetAppInstanceArn(AppInstanceArnT&& value) {
    m_appInstanceArnHasBeenSet = true;
    m_appInstanceArn = std::forward<AppInstanceArnT>(value);
  }
  template <typename AppInstanceArnT = Aws::String>
  CreateChannelFlowRequest& WithAppInstanceArn(AppInstanceArnT&& value) {
    SetAppInstanceArn(std::forward<AppInstanceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information about the processor Lambda functions.</p>
   */
  inline const Aws::Vector<Processor>& GetProcessors() const { return m_processors; }
  inline bool ProcessorsHasBeenSet() const { return m_processorsHasBeenSet; }
  template <typename ProcessorsT = Aws::Vector<Processor>>
  void SetProcessors(ProcessorsT&& value) {
    m_processorsHasBeenSet = true;
    m_processors = std::forward<ProcessorsT>(value);
  }
  template <typename ProcessorsT = Aws::Vector<Processor>>
  CreateChannelFlowRequest& WithProcessors(ProcessorsT&& value) {
    SetProcessors(std::forward<ProcessorsT>(value));
    return *this;
  }
  template <typename ProcessorsT = Processor>
  CreateChannelFlowRequest& AddProcessors(ProcessorsT&& value) {
    m_processorsHasBeenSet = true;
    m_processors.emplace_back(std::forward<ProcessorsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the channel flow.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateChannelFlowRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags for the creation request.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateChannelFlowRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateChannelFlowRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The client token for the request. An Idempotency token.</p>
   */
  inline const Aws::String& GetClientRequestToken() const { return m_clientRequestToken; }
  inline bool ClientRequestTokenHasBeenSet() const { return m_clientRequestTokenHasBeenSet; }
  template <typename ClientRequestTokenT = Aws::String>
  void SetClientRequestToken(ClientRequestTokenT&& value) {
    m_clientRequestTokenHasBeenSet = true;
    m_clientRequestToken = std::forward<ClientRequestTokenT>(value);
  }
  template <typename ClientRequestTokenT = Aws::String>
  CreateChannelFlowRequest& WithClientRequestToken(ClientRequestTokenT&& value) {
    SetClientRequestToken(std::forward<ClientRequestTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_appInstanceArn;
  bool m_appInstanceArnHasBeenSet = false;

  Aws::Vector<Processor> m_processors;
  bool m_processorsHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;

  Aws::String m_clientRequestToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientRequestTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace ChimeSDKMessaging
}  // namespace Aws
