use v6;
use lib <blib/lib lib>;

use Test;

# These test vectors are from the officially posted test vector sets,
# see manpage for URL.  There are way too many to test for normal
# distro builds, so we test a subset of sets 2 and 3 for each.

my $testvecs = 16; # useful values are 1..512

my $ntests = 89 + $testvecs * 9;
$ntests -= max(40, ($testvecs - 439) * 4) if $testvecs > 439;
plan $ntests;

use Sum::Tiger;
use Sum::librhash;
ok 1,'We use Sum::Tiger and we are still alive';

class T1 does Sum::Tiger1[:!recourse] does Sum::Marshal::Raw { };
my T1 $s .= new();
is $s.recourse, "Perl6", ":!recourse yields 'Perl6' for Tiger1";
ok $s.WHAT === T1, 'We create a Sum::Tiger1 class and object';

my $recourse = "librhash";
unless ($Sum::librhash::up) {
  $recourse = "Perl6";
}
class T1r does Sum::Tiger1 does Sum::Marshal::Raw { };
my T1r $sr .= new();
is $sr.recourse, $recourse, "$recourse picked by default for Tiger1";
ok $sr.WHAT === T1r, 'We create a $recourse backed Sum::Tiger1 class and object';
unless $Sum::librhash::up {
    todo "This will come with byte-only marshalling", 1;
}
throws-like {$sr.push(False)}, X::Sum::Marshal, message => "Marshalling error.  Cannot handle addend of type Bool via recourse $recourse.";

class T1p does Sum::Tiger1 does Sum::Marshal::Raw does Sum::Partial { };
my T1p $sp .= new();
is $sp.recourse, "Perl6", "Sum::Partial mixin chases away librhash";

my $recstr = "(:recourse<{$sr.recourse}>)";

given T1.new() {
is .size, 192, "Tiger1.size is correct (!:recourse)";
is +.finalize(Buf.new()),
   0x3293ac630c13f0245f92bbb1766e16167a4e58492dde73f3,
   "Tiger1 Set 1 vector #0 (empty buffer) (!:recourse)";
is .Buf.values, (0x32, 0x93, 0xac, 0x63, 0x0c, 0x13, 0xf0, 0x24, 0x5f, 0x92, 0xbb, 0xb1, 0x76, 0x6e, 0x16, 0x16, 0x7a, 0x4e, 0x58, 0x49, 0x2d, 0xde, 0x73, 0xf3), "Tiger1 Buf method works (!:recourse)";
}

given T1r.new() {
is .size, 192, "Tiger1.size is correct $recstr";
is +.finalize(Buf.new()),
   0x3293ac630c13f0245f92bbb1766e16167a4e58492dde73f3,
   "Tiger1 Set 1 vector #0 (empty buffer) $recstr";
is .Buf.values, (0x32, 0x93, 0xac, 0x63, 0x0c, 0x13, 0xf0, 0x24, 0x5f, 0x92, 0xbb, 0xb1, 0x76, 0x6e, 0x16, 0x16, 0x7a, 0x4e, 0x58, 0x49, 0x2d, 0xde, 0x73, 0xf3), "Tiger1 Buf method works $recstr";
}

my @t1set1 = (
# This already done above
#   "" =>                   0x3293AC630C13F0245F92BBB1766E16167A4E58492DDE73F3,
    "a" =>                  0x77BEFBEF2E7EF8AB2EC8F93BF587A7FC613E247F5F247809,
    "abc" =>                0x2AAB1484E8C158F2BFB8C5FF41B57A525129131C957B5F93,
    "message digest" =>     0xD981F8CB78201A950DCF3048751E441C517FCA1AA55A29F6,
    "abcdefghijklmnopqrstuvwxyz" =>
                            0x1714A472EEE57D30040412BFCC55032A0B11602FF37BEEE9,
    "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq" =>
                            0x0F7BF9A19B9C58F2B7610DF7E84F0AC3A71C631E7B53F78E,
    "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789" =>
                            0x8DCEA680A17583EE502BA38A3C368651890FFBCCDC49A8CC,
    ["1234567890123456789012345678901234567890123456789012345678901234", "5678901234567890"] =>     0x1C14795529FD9F207A958F84C52F11E887FA0CABDFD91BFD
# Right now this is too long to be practical to test.
#   ,
# "a" xx 1000000 => 0x6DB0E2729CBEAD93D715C6A7D36302E9B3CEE0D2BC314B41
);

for @t1set1.kv -> $vnum, $test {
    my T1 $t1 .= new();
    $t1.push(.encode('ascii')) for $test.key.list;
    is +$t1.finalize, $test.value, "Tiger1 Set 1 vector #{$vnum + 1} (:!recourse)";
    my T1r $t1r .= new();
    $t1r.push(.encode('ascii')) for $test.key.list;
    is +$t1.finalize, $test.value, "Tiger1 Set 1 vector #{$vnum + 1} $recstr";
}

my @t1set2 = «
This_is_done_above
766AFA25B4C4D82C835B8320654C93BDAC3BA9671A0C83AA
C2A72A4D6B22A5CD2750DD772F28F809CC15D40D552A2129
6C36DECB7CACFD5D87737515E25C79AE4D545CD487DE9CE5
CA3BE46BB668F8A7D60349A2931D17D53BB55F354BD985A6
46B9C0183B22D60ABE0DCB6A20D1B4632C2D918D32007C66
3B8A19B192E7559413F577C112B7515A2932BEACFB994432
06AA7ABCE8443EDBB4B583D7E273E95CA7883CD2E11606C4
5D9ED00A030E638BDB753A6A24FB900E5A63B8E73E6C25B6
ADA18985026A8D65B9819B2292C587CBB3D88435CF92ECE9
4C2B2A626E9D8089A49F4584070BD42D1C14B7610A2DFBFE
59065CD74C36A450CAEA0A367B32EE17F189820661086DEE
BFE432C13F794D4BB51E4340F2E2272E4D052F75404608A2
6C90EABF3F9B8F9C2A998A5EF0F7B739D7ED088FF7108326
CC7BA5FBB401A8F3A4E503C854AE6D2AAD250DC3869B6805
52C9CA4CF1ADB66993BB63979B437FE2DB32F3A7E6FA07E6
AABBCCA084ACECD0511D1F6232A17BFAEFA441B2982E5548
E3DC5C51C3BD9C5DB89C31785CEE78E84342BFDAA771986E
41613E1EE815D8C9AF5C3EA7CE2C5D55A2D6D76D6C739900
1B8575786EB7F87DDC2AE148FA2E28D55074F7B4F98E65E7
F963FA72706F3FF0F55CD2B36211A06CD190E149E53D9BF4
BAF5E11ACDE2EB6807403BD0012923775498D9B0C6ADB629
9272B22055528CCC9AB3EDC772D12976BAAF65768F000CBA
7D6FC776F1055B15D803D9EF6DD5C6101557BF0381C0C28A
7F7A8CE9580077EB6FD53BE8BB3E70650E2FDD8DBB44F5C7
6315472E27E41795BB61E560E552ABA13B35A6D1F2E9ECA7
CFB8F014B109374B249DA192EEFCE83AC7E3F40FD5317C7A
663AC22F5301AD1EF6EA81DF1730A95EAD03188D7E9464E4
9E657C65CE849AD1F50E8558AB1490A09713A408CD595126
0DBFC4A6203B362AA084480CC9848901CE13B7054F805484
3AE28B4517A2D8689127C3BA107E633F39F37F9D7CDB75EF
8C81474D9940F5410161BF3902C738888FEA5CB4873449D4
605D1B8C132BF5D16F1A8BC2451733F7F0FF57FD5F49E298
B2A6F45A6AAE093014855774713C5D270A6982331B7018F6
A08FBD9D743A063E1FD2F3061C7DEC96BC4A09AC2BA34273
89B2467A13BE3A4ACE341061935F2359BFB9CC96BCE752DC
8DD0EBEA7F94294E614660782762A18F37A2931769551D73
CAD18FF3B396F7FB04EE33C56BCA3FC3436040628D7989CC
F69C7F27A572E6653C782B6E1BB5146BA4C8A800D2CBBDB1
230B9F3B0BA4459990698ED7519B5D42B3234B4315F0EFB0
7F3A0954A5C374566C72370D1D97C2AC8FA9B1E3CB216E31
A80CED7C0EE0898C340FD366179019B66654212301221945
3A072F0455BACEB53763EDDB3D76EB867C29D7F89D180682
004717AC7E1FCBFD3C804ECE8FD228FB44C48179BB5232F7
AAC53D160368ACF2E74D962F45F3C8D70559338B5D6DB241
C39DD9C707D2812831C4DFC57C98F99ED5BDDB5A6136AEA8
7278E1FC893D215B384AD47A99CC1B678A647A2D04BA6647
C6F774270ECA51112046E06A8E5589765092E07D41D45DF9
B333735AFDAB30B4B597CCD137AC1AD2B30D4A047BAD0127
56CB89E287C0BCBC447C434C4E4C7D927B7C3A93A13A5122
D64C1E089406A1027991D7187E6BD88BE5E7B1564BF28640
624BFE886D307BB36C50CBDDF562B833DAB1B96288B568E8
51ED2437E8420E867AB602D439CD4E6839F4A69D57BD40CA
7C345779353C5E33B55CADD6F543A8CDFFE8DD87C158D12F
2DFF5DA66D3BFDBD16E576B8A5BED10AA7A59EF13FA06E56
9952AF5E64EEDE22CFCC24F22918FECF3B0C44DEDF27F574
1F27BB5926F35CF5D52A24817BF56CD17D79BB65ABE85785
7D7335252067E8E12F18BFCF5C1A77A39D5FBCF2FCF913F1
0396AE290E377B76F94F5EC9657F8E4F3420998D4849B26F
6AD76B3ECA7E90F56B0134B72EA14D998E0FF816FC5E78EE
AA5F69ED936D6BC41C0F0221980B66ED64117B6F0B2C474E
5DBF4C39EDA1A12A83BE9D82B3B7008AD88FA5B1F069C32F
2677DC2E7D3902ADF64225738948F7DD7BA7FA6450258A91
7DD158101DA48F6F9412B9E8E6628C91E33B836D5E8C2F17
5229DC51A494B913F8E04C4C729C93CB8B260CA4EE8EA9D7
2E31D53CD3D1ABF21EE0F99B481E2D52E88EF21D0CA826B0
73F693274A90C747D83F111D084321953CC1E283EE1C9652
47985F82E1B8EC4F3D6AD57CC6C808C201B5EB0C1E40ABBD
CDDB515A9895C525DD4C4FBE1A93B89191AC030C5685C501
264C3C9C1EE792943225F5C57A274BAA448B950E6854F9FC
DFDDCC7EE2DF96A970AA5BA756EA5D8A6E6B572306145B2C
CC693DBA58877C05239C2A0BCBF77212A6F637C023BC6836
5DFE02959A64534E0E679937B712F7DDE2C7B982597D315D
2EE34AE445DA43259FC63C0F4886A0D512AAAA9B340DE273
2F4E5FBD2060D6D18B9CD15FADB29F2DD5C9209A650678FD
8A21DA997DD898FAE2470276DBDA548D1EC63C0E76193C60
821211AD301E77C6CCB2824BA6DA1EC482F16BC7D58F867C
5508EE74EDA4BBC88247B0C833231274E3B739DAB0988A63
D733868DF262039B2B13C9FD71C197FF0A70CD866B8B04F1
3E767903CAD219A37AF1D393FFEC0CAB61FD90589F5239F3
654D00C774FBE74CFE4C0A6DEA5917823A6B20070CD0CCCB
EC4581D397B5D69298088B30E558207768E66C7262D919EA
BA1AA7DC59BB491E97F0206DD6571E28A4903A124DD3B72D
E40EE08DA140B3CFF129EBBE819EEC73819244D98E033BCF
A804CDBC07AAF509CCAA09D3E9A1F9F24CE682D507157F9A
84AC1C96B4DD107A577D3EE45793C98BFEF090249F9D2E79
76267CA67FC1DBBF4A842D11484B4E7D5220ED5AA908D307
B10041175C43C47DF24BDB7A8680CE2B2707CF3EEB846E4C
C82A6419844FC7028B52CAC8186D63B85121AAACBA622A87
DAD2E2635D2CE5B8721B37433EB43C308E26DA1324623011
73A42017104AAD84BBC78D0FCF9AD531F425AA71676D6429
4E9A41884476A21ADBB07E54F7C2886885B5B077B6201ACE
89EBE56D435D7F5C16A730B03E3069241345D10C9E894DB0
5BA7A613DDFCA9DE97F6152B74C02138C2925A1550BA35CF
21C1D3E7925BCDC70E59050E3B06BFB68E1F19112E82447B
279A783027CD546358519CBDB2ED467CB8A09306F0DE9AE2
06F6D65EF081AFE892AC0946F275DBAB9D3B6D9C55E517C9
5AB93CF43A19AF13B183C725452F6B8F5D430EFF74100F0C
ABA0354DFC6774D1D43610CABF2793A3041070802B11B37A
FD71AC4A03E6C0DDC64596A373B749CE1276E5E46B8262C7
E60C16B89368B6AFF0F0CEF144C206C69CE468A080B6D261
8CCFBF9EB58AAA436F6484CF0A566E0D41DE85F693621F77
5EBEE865394F1FD68208F0A921D73ABC4F4F0632B96EA129
3244B2BB44E68FFC04A358146EEF8BB89644C9307CB6BEA3
42C40537A876D69DF84FF36FBA3127901B112B9E96688012
8DDCFAF9B36C347ABA5314670BFA62D64A3C8D097DC6B8F5
9CA1C6ABD88E32CA3545410F610EF6A8588438E415FAF4E4
7AF5B38C8C863D714BC187400D08DB045C74F2A7DE046955
FAAA9B741A9D79B7BBBA11EDE1538D8CC480B9D788B67DD3
0F275011F9D406E2930F5328997EC3C8CEF416C589CDCF78
98F18618F5022F9FEC3A5250C1FF16DF590F8EF27E746465
2514050BF63C408ABFE6B31AB656B84DD4ACC98840671074
6FA00EB82505EBC4707F675B0A72A9F9795F647B550F155D
6D5627FF87C0D7F6AA45C601BAF0C59F280CD63A73D26C90
A587B6BF983BEC315CDC974AD154BDB9483F4FEDAD76BBCB
2074971BA3054150FB36C46AC43183A2B128056F693C31E1
030F292EF4743DEEC8B0B29F80279AAAD6E6722BA6817692
072065FC399EA1F562DC1225D40E69671987D79A3D14CCEE
D7EBAC7EDAB4ECCEC85E072574EF3B7BD940F093C879E6D1
E08377D7D8BBF2C6B31112B24EBFDFDCF0FAE8D872C87D55
239549439CCE539F304CB7A95160A033529B228CA406EFF9
FC10CAC34417F86387A1792540B1C5F8B8A232FA5FD1FD9B
0DE1DC3BDE99F25254EDC2CDF30C012E36B04766EE38B917
7C585143A52234DCB28D103289F01DF2D277287F3AA3F114
ACC4B1539649408FC256F268D5CBFA5C74B434C52A0121AD
BA664121D81F25551079BFA58C69393FFC8E9BBA48267076
C962B717D1C15E8EF7908A148054C6CC23EE74D52231258A
6D95BB3A41801A653FDB1F56CA8FB5F352A3B781AEDFACFC
464B87921CCDAEDBC0D6941610D1EB19E536036096403F32
D2230EF4D52AF33B975DA8999EC7ABFE7713D9D8B5DE54A7
9E574D7D407BDEB00A3D784F8B7791E34A350914A20CDB7A
52C3130E642E64EAAEE9673DBFFFDD3029DF46359F885E4B
B8CB4B7425DA877097FB7DA13AC1192F4DA5A89633B1A5A4
B985E6537CDCD919732C6C4658E74CD2D45496E37A0AB42E
BD767AC08E82D9F3752673BFF987F077C21770F04CDCDA00
AA2BF0958BF94CA190A1EDFFD7D17D69AF4E90F347020C68
F29CF9B53DA4F9955E7BB766EDBAD59EA914E5354DDF9ACF
6A1B4DB723F4B607F1CB773A2A72ABCDF7F41B1E64EF50CA
EDCD0CAFC9B6F741F46FD8668ADE02D957965E3B6AF69DD5
0BAE83A29A6B9695E7B19DB730769DACFE487B5A54DB19D5
978FE24933936BD50E395C290CD6A4079BCD1976299D7DFB
D382F4E925FE899ECCC3AE7BB59198813F6DEBE101D185C7
3B6CFFAE0479EA6A67F6A6C5D803745CEEC7F1C7437462F6
01AD724959CA32D97989567F492C87006C82A2E89DEB42B7
E9B6BAA124BA93A4501DB9A92F212A14C538CF82C17E89B4
52F79881C80B39B2D8A80E68848EE702BA842B44CAA80CCE
F596453768672C67D26FF95FFF16E0EC55F2EDC993ECA2DC
8043337BDEEB6C587568175D432EAF6ED97293908F0FC9E3
31504BF60258347DB7E5E77C836913F1EFE61122B9BFAAAE
4A3C4471643DC81928287DCCFD5746977F171A4D225399AC
33E75B6E73EEFF0E8033298BC1441E4866E6228258860778
59C50812D6B63907A5523BA2A8742FB1808EF33EE435E55B
03EB6D4703EFCF2E088515D3B902A1CF78711B7E7CE3AF74
082009BCF5B82FBFD7925A6BE4B54D71A384A843A246E4C5
7469C96361297C4440E12F8BC0CE15D131CED2A3E3D8B3C4
BA94E4559E7E3C54F228112D2E2A49B4AE31DFCE0B03C4D1
8C96E1DC34240E666012957D9B5EE0B423CEA57CE4BD52FD
79823387B86B517F0994CD12972DCACCED3EB94A0A3B9522
47F179B6B6DC6EA6C418558801F6E5E6ACD1F1120478C40B
F3E38856AD4D70F839FC44F344DA7217D011F48EB94FA66B
53BFCB1EB5CA9B2E8A4225E6C00584023FF8AA0017AD3A0D
4B2D0F151C848FE28BFB27201CA242F1B863C59ECA5737F4
6FD7B38DBC4DFBBA91A7ADEF5805F924A88464FC056FB2A0
7F094A9B6831B9929D03AF95BCF5567DF49D0E1C15360A71
3230A6EA60C990966F8345F5C68304613BE55FBE3605BA8E
F65F0E51FD4C0B70671A9FB1B29CCD877F8509FE4854114E
9E4B9EE43DC30458BE7E1D2FC51A12225FDBE60ECCD2EF19
6831B3D904EF81FBFFF2FE7ADC30EF09BCC8A4C552D02E11
67F6D2B74B13251432469B9CE6A7C91EAD6E72B9CB51CEF8
EF232CB9815202CB72DFD9B8E37AE9560FDC0221C9F3BE14
BD78688E71A3F0C70F1FE608EC27813C792B923CCEDC9685
B585D2A299AEE3E10504A453A7391986A13BC385958902DE
FCF11F04FF7DDF91BE2EBB2D244EE15E5A1AAFBD8F05C95E
6F33F97A4CE83293ADBAEA9DC940936C20DD78F79059810F
9ECA5B82AF301BDB45DB2AFA3647DC0437E7E3DC4CCAD275
25AB437B88A8435568D1ED483798878DF6ADEB0472EF93A9
81A68B237DDC60C9C31B34B52A754A8A7B1F51D40781B988
8B3F8554323D325E7CAE1E4D555660AD0B855ACCCF1E09CE
1585CB0936BE8C7CB01DBC72CAF67ABB845739EBA5D53C3D
7943970FCAFAA1394A3770BCAD50A0A5369530884D79FF47
48EFC4A26AB1620074FDF0EB246EB37A4EF1254D03E506D5
348A3EB8FED8212F7E94CE30AB164250CD9FBC7276951573
7844EC5944F9CFE976D5B59C85F83FE0800B70AEDCB0D1D4
D8BD30F4817F6AFCAA804E8953B5875373B35DB0ABC02F9C
E99E18449AEB4421CEC31D1521C23445BC2598EA5DA14A09
46A53B5DC0DF07A7A242D2E7E880F66A405667335863F285
4D0D93A77C41B4F8A2C0A65A0183432A34F0BAAF6A6C670F
39660BF03402BC44588263409E907F7DDF5A367926D117BD
6A6956D1549165712122EE87EDFFD263BA5F907694AE5771
DD1CD1B51B221A7551E6C81B108A060E247DA99BF9216F15
9451D3D12BD92FA84F8A72D6A078B2E5EEFAA422690D1326
27197D44EA2B0E5F5FBBB775DE113B81B08E173963746813
CDDDCACFEA7B70B485655BA3DC3F60DEE4F6B8F861069E33
D507E20EF2AFFE67D2D8BEC5FEDD3C007729049CE31C53A4
D0EBA5D323155E31AFCCBDB6BB52D18718197D2508B80176
29D678AD5B64735C11DBCD29D5D63F8F39040F22DDDC9F3E
E1C3030EB0A37646147257AFDEA5F0CD4212E9F708EF42D6
4C545BB03C662D4E25A2D0768F808952632AF357EB94E557
875B923A3C36B226C5F3FF101DE1B4B9823C53056D0E9279
CD9E6FE8DF4D801FBEF41A5240D206D0EAD97131BE4E52C6
424CDDE2E5478A29A71352AAE538EAFE9F1B6F07AE05137B
995E56D393BB35480B66AE84256EE86D3261164949A1E8E1
11659EE7372F8CFC9537DD28E0D7DAEE1F180A54A341620A
9E3DE768550294AD75853C7D5A5F433105E79DE5B8A5EE20
09CB03BB7FEC2F4C63DED38B01A950D14BB5D858099CBAF7
1702EB3E6032178A5B60825E8DBE4754105651C49B18CA2C
49FBE65A50747A1D8088B1DC0A79AE197A056394A5F85860
D0F5B3BC5D071E03D05F7C09D4953AA289FE10736951CD88
996B4DD292782A19D742B534916870023F32E72F1C197625
463853DEFC2F5AC1C60FCE25747F2A99CB8CFAC8DEC77875
9E44B238B5C34B34D69D8018E45028856B53231FD9F25DF0
995B707087373EA186CF9A2876897D3B1F67069A296FA7C7
5BB269F595183A5ED732ADF97D6554F06578AFA866E7DAF5
B3A185B73F912A04DA88518D4601D27346CFA5E0872DEC26
9547130F960DD5A682FCE6E01D8C8516AAFB2A0CE5DF9E6B
88B1D853AE7ADB39C8959CA9B6EBC13F8E336C4DC4136E7C
2C4B719E5E0BFBBA616FB8AB35470D84B676C7D3A6395262
86770BC6BC120A52E3AFA6E45D4FE9A3F72622DD0E2202A8
751FA526AE68FF8EC6E8BAE548B5916115A6F6E7DE68AD92
64DF6C1874F7924033E8ED56BDE011F8533DF2F8F809DCB8
CE186E5AA07D65B2DCA8889A03DB729F141A709793415ED7
E0673B2995E3706D779A79FF8876A3FBDAB27F615ACF75D9
13DCC16CB6094283D1FE7B2E66AC75BC5513716A08FB2999
AF9874E0C405C0FBDF88E99E8F3B06730A77BC823946096F
F178E06FC3319BA18F2EAC3DCC5240D06BAC062BCD5CA1CE
ECA63E62EAF2BFA0442D3402E38DA2079C13704C3570109D
AC7A917CF8244BE8AAC81C2C7690AA37D8047BF4E3EF472B
B46E474162D3EFFBAEDF854D54724F68BAE618F13DF1EA20
F118D23A3759BB3924CD28E4F56312E59270AB2971432E04
A27CE999BD56A6E64BE9D02FEC7000DEB9B6D54D2BEC6E1F
E91A899EDF44005D73E36A31C082247225E4EBAA2D400C55
DE716C5BA7E71690DF91852593D16BE622BC1EF5E514ACFE
E899374C34AED37FBC9E0E4A0D37A13045639D5CAF24095A
EB926C1EFA2CD7D222A9DA75601FDFB7CDD15A6E3BFACD7C
29C4C6EF8738FC8E580232A5C320F08150E0273E502A386C
D9A059917B57A175EA3FB1AFF9EF197598A8C3DE96CFF80F
128C51808702CF4C53FE5ADE29C880196960C210884998E9
89A8673B8C41D9E00CDED45FFCEE30C63D96C10A5707C47A
068B643C582179DE8FB0C0B0BB828D78D6089CA3673EADE5
4B4E83DDC7236551415145F9818462BA25ACDDAFC9B94717
3D8E74F26682750FEEDE30A305C9CA162C6BDD9B790BCC5D
CF57B0BAFCBD670D48D52D22E94CFC9DA1A7EA1F80923A06
F94903F2F2AB91C9491EA66C7A98401F869ACC1D8010C710
DF58EE17B266D0F132AF119179BFAF48DF41331DDCD8D464
D74207B746B3DAFCFC8E38E0325998FB2E1219720DCC643E
CF843577B294A8177F18BFD08B2DD8158E1CFCA349F99063
6E932AEA840E75D2C4E3549DDBB4F7DE030C61E29A3DF067
5D53EE60FB695539CCE16914AD2A8E477EB94E0803728FD5
3EFCD915BB46020B8999C9AD56356FB91F93C4907C4012D5
98A1A4E1C0D37DAAC9119697E1A5AE59BC8829340A496BE2
293DE385ABAAAE0702A6A6B1D6FAAF852529C0B1CEEA87B8
C41A3D530973C7A12EFF41D6D7E0EB2D305E21D74C68FCB3
182CC87B2ECB0B8262792B868EB77D7AD86ACA5A605E930B
D519F8F5EA54C9CCE2756C1941275E0F99996F76D3BD5682
1B7B457C663E971B4CBA01D03FD5A4B114DE021FF7C13ADF
A29BCE5D377367636E0F137E809D869BA472B1F798488344
739414BD4CD6AB967CD46A1D943412757D858B24D1C4ECF7
D265FAEA2CFACD294FDD8E9C1687C2D0DD8095DA77045362
72E2C3BF21DDD1D9E9FFED653CBF27F21C1304B1A979280C
1ADEB44B5DD6144F37A430AA5CDEE6E87A54608BDDF86FED
693DF9F1D66ECF72041F44897806DB1FEDE9485C38D73680
D90251247CF23FC5F210D6E8CDB9881DC3B26134B2562D00
F887A6A5C7D6ED702E5EA0774BBD301600EB0A41397E44B8
1B9EEDFEBE055430699B64D3773C0ED1785372001D6A94A2
3615FE2BB3B94A00245A1B3DF68B9D43BBAD82B16BFCD6A5
8AC9AFB779EFDBB5C2CA2C9080FFDA39763F74A973697889
D871F30BAAE8582CA5AFFE92E79E5F463CD5906AD086441F
1FDCAB3BAC2B7181A34F3031DB46003981192370E6BBB953
FE49517B506090C44543320397A11D8B77CDA5C6F13CB10F
FF4D54E6589BF784967D3F02BAFE01D4B2DEB7C01A086015
84A140DE357F866F5FD4527C66CF1EB1825101F7ADCB0948
AB528CACAD90CBA6CF89EB3F63573A5D4332F882C7CE71F9
9E9AD6783D3775885E65D05EDA4262E179836ACBB9867CE2
D5ACE92C4F333668EC8FC0B9B940844BB2ECA10BE0AE9156
E0FF3D4712C5713446F6D5D997573631A06AFF779D791E27
57A658AFE0CA0BF103ED10CF6E49E3A94FA3C89E51DA140B
84D4A2986640BEBC1BFCF3C4AEAE8EDC91A94C0E69AAE51E
F3926291AEB75587B0B8D575F5F1A27C89D9AB24A4B6221B
4847B57AEE24F48811ABD2CB9978AD5DF8501A3DAE286AB4
5CE3715F4689C9ABD5F180A11E22A18201F044FDEC6644F9
5D91C8314556534FF3DEE0515334F09E16935538E347C029
B13A7BBC81055580B02B4CA3DFEBACD66A178098BC94E25F
D902017A6A6EE71E4A0F71BD78F7AFAC7CBBE2E7719A716E
26BBD441CAB9A50A9CEE2398AFFCFB745AC5946B010ECDA4
6A6697B6DA1A669F9477D1418604DFFD4EF4F00025369064
4753A53A1E8FD3A836888089878C79EF52AC4839DA881E9C
7EE9E4268BA0E5003F27BC070C52D21E995957A52D435210
06B4D0E4C5892094EFA987A2037EC844D4EA61AFFA2C4E60
E48325D26CBEA5C14AF5D2BC043242DE2A5DC17EF5798BA0
C0AB19DAFADC7090E5A43A66D3ADEBF9201204C0856E056D
2A39E534C11E9232B5B46CC2F75AC715446AC532BA42422F
307067862B277E69ACB6EB49F7CF182E74F41505CAAFFBAD
7AF0C85AD9358907F2A191B0B5CF4044003FE50D69921EA8
17EAEF6C0CFF6930B34DC020F331FB657BDBBED88FA6705C
6BED7F7314514B384C139811BFAF08121DF6DBC9550EA21D
73163CFF1F811E8C56D8E5F076412321CFC1A55DADEF1855
0DE25B928CF3E4FE85F46C18334D9C9DAE995E4889068C4E
E3A4425BDD588E8415293521CCB6E135E27960252E160D19
8261DC34670705EB0250054DCF6062D9B91E083F09695CC0
00AD05F76C41974C04A9C3D5CA5E4D53E74F24BA0F181984
07C4DA723EC91666017C5B7235140FA07976876D05EE534B
9341A937CBE62F7B0EDF4493209741E1491249A584DF4A29
2612C32F1C29AB6B58159EA81C4E0779C13CDFB5307BA77D
4056019CAFF2472981A22A256FDD1A0C6C70028C454665C2
1BD4360038C1282E66538B156E9F604AA0262D4608184A5C
B96EA7517576851B2BC13DCB5C6F7391AD7122643F0CB571
EEC34BB528EC15D14412EDE038318113DB4AD6A24600684D
D38CCA82EEA448448DAE01C300762F26367BD25BDD0E7EFE
E3702EB44AFE7EBC18DD0845256F34121AC49F6B7BBBEC85
D5F6DE3CBF98622D792F26404E3C9C2CC1FA4AD5F028CF11
9BE9550E3373C58077559B598FDD20B381DAFC4A9A2D4537
C9C471424AA6BED291732A74A4DF1FB64B71837DFE05E43E
FCC72686AD012FF48D1471CD0EAD606EC21A5E496DB66103
158F6F66FE465F61A134DAFE329E70D49F343A658DE77532
7ABBDE244ADBEBEAA31399955155E30F69A6C7E5AE35EF40
EF3A4B8AA5800F434A4D3085D9C2232C50A7F77A9FF04049
59E7766097F2D40016A02EE4357C3011AC21D4B50F71AFE3
66B6407BE96D8A1008503AF829CB8B340CD1AFADCCD2C154
5512C01D4983348720AD4ACBBB9FE07AA5C6CD9C134C784F
ED36467A25C9B6055F6CA9C6500457743D02289FC9F3C768
F11D39950658DBDF786703E3EDAAA3A654B4428AD419DF11
FC44AF74FCFB97CBFFFA8F49A305FED6F0CF091C5555EDBE
52D8C7EB89A0ED303290A5BEC7D9FE939F1DE5011A73A24B
1BC88EE45F3F639D35C7B2A7C1945F0A8E11A0D631BACF15
2080C3D01E26907A6E9536F3CBA4B83CF3CAB77959D33E50
41572C7FF10D1C77C5A9754E31E8D4DE1125E2189F12E662
7A2F4FA19134A80C04448E7010DE975F0AAB61A9186265DF
1C73ED616AFC10A06A484BDA0F3E31C043F14B0941C63092
2AFACE18DA68984D907D1C0F647E9244D6976F4F471064B9
D6440F08472C445077794D59D9F7FF690C476D58BC099C7D
A8D1985EDE98057EE1490F3FEEBE0AE5FF9072A5F4CB1D25
683FD77D4E9A591E4051EEEF767161AE1F38422583CB7E12
8F8FDE1BC53364E2B6C89ED3A4D80B8C69BCDD73D025616D
40917AAA260BAB49CEFCBAFC473A896FAC9D5EFAE171A5C6
77DB0429DBE25A41DA836808895BD948826C7A2C8D2B6E32
9C5FCB73F0A437D9BC652EDD5814B1FC55D6E8D0FDF80465
E2056969720C08B5C9C8F903416668A7F5B2D6F95046C5FF
1F9DAD81EDF9595A3872A6980AD4E2BCAD219E4B1893D107
F232A5C5062C6B55D26FADF4DB7510E37537EE5039D3E458
E1B0B6D9105482A76DB5852F76D02E432D6F1C02EA1FB43E
A0E27771E51A3976E7262D092CCB0FC65184CFE30C85DD9E
DD1F563470929BFF82A9FB979748DA0FE54698E7566D87B0
3C4D5C146E28703E01A7202202619038E44A7637A28B7FC9
5A2C91F9FF8B5242FFDBF35689FABAD91228ED65C5EABAAA
D95BC879A68D549A5296BD34344C5ACC4D826E54ECEC346C
9A175185D3913FF913DCFBC81070DC467D8E1C2E4799FFC7
557AAA123B10DFECDE0440CA11FC882C48642BB5844E87E9
ED78BF444031317EB7BF41F3008604E52EA287E88CF401D0
69BF72F8422455645C841985FCC00B545D4CC3C87FD597CD
2FABCB8173C04E739971113F4173AECDA305CB3E673662F0
C2C747303F258C981D214FCE462D1F1E24295C724A2EFA57
A8130996CEB9C1C021E5C8CEDB55A44DDA56EF2EF4001EBB
D33A7246729743643BF4B438FBB9BC2374F9E346A0868DC6
E4EC4AD2CCA6183545501ECC33C78F895B8DCFBE8E5EBBF0
D72FBA37CF967845B9D7D5441C1E76DA762E559D8CD0790D
D7015D9F168D9E90939EF15994A3A4CA60A6A36B2D5F6E3B
2F6A5DDFAABB8D6C4B29276FAE46B14FBF444D29C3F03E81
1E7D14CDEBDB776789F06A394ED42D272096C18405F72C38
F91073D2422729ADE2D84ADC16DE6E068F01BDE41314FB19
04FAF52202275080A915E3ECE47B7A5C5B85783C3B266D3B
1201A8247225F31B928390B35787C046653249D0441F0AAD
EA00FF4C51766DAF8F05832C7704E4E5F761593D70DCB3BC
8B5AD5C3FC03A5C57550D00514312B28B21B60247DA6E8E0
3B07F8BC2E48AE0DE40743B35F90D24EAACDD87F9E8A103A
2C3AF9BBF42BA704AC1CFF50712300C0208B996BBA2CD6FB
8F0492DC54D3013634F3C1D15020B0DAF91290BD5708ACEF
DECC27A83F6DBD7A11E97362CBCE393771CF554A6B29313F
F54330CF39C14C1AAF67C754CB46C7455A119F3024482B76
1E63B6124EF57A92B36C8FF837C38CC9072048ED98EE853F
9FF29043D626941F044FCB85EF3106EC79295549AB9857F4
7A18CC98E15DB828734E19F2A92CA67FA7B0A76B41B37DB1
4646BEB63903896484AF974A7909F7E5E37EE713794A5682
BF501E591AB9C2224F40FDB7E3CB2F772641A30981BABFD6
E02F96821F8A80C974C3DC6FBB2261A5089BB8248563D9F0
9233D832B93149A9F0C7B4598FA5A5788C2B0A677BD910EA
FC16E5E7AE7F18BC973A6A380D783745921E269F822764AE
F2E4B6637FDFF18F785894EECEA514E7B96B2409E62DF3FB
96B6213F10B67BC775FBA5A1A7AE6044FF9D0801D24662DC
1F6AC762BE8B8B4E4FA998022420BBA027174DE7D260D570
942FEC93915205169E00A0B3566150593F64F3130B937B5F
081E9CFD455A715FE487B9C1758ABD4ACEDBA7395FC479EE
11E864E0138433B360ECFCE823FC33A3BC6481C5DE5C95CD
965AAFBD0832E5C1736518A9CCC8B3FCC083B18472C2F93A
D88A92D373B875143778B905DC2C55773EF4F851BD38DAF9
10DD94B66BA6AE0498C9C7754844662E5D8B62E27D2C4D26
36C383749AE2D62CD8A88C363F3CE84B9480D466CB848908
367998CC80C1A97689FADA48E3F468A0C0D2BEBA3DC6173A
B6D8846050D4D6BD95D5D2F101E0E86BA975103653BB7AD2
C87528E06CC91F6B68BCDDE94917709AF10F8EA6A107AF36
3B4638A52AB4ADDDD549745CC19EEEB6DE416F66BEED28B0
CA16D435CC34DFBFB75275CF585E81DE5DFD985555615D98
AD254F17CDF0DF0EEFD4C5FB027A8E8EC41E43453AF27777
8BD37E3CF05E59537389672DF921392A45CB57ACFE9247A1
6BC753092DF48B959F6BE285C6AD319D331FADCA850C20D0
8E6F31F4F49CF2926491B83D4600EFF9934A7375DB395F2D
44EEB169A54F26D4B861FE39FB52359551512B767CB07592
274A4042B6F4B3E483A77B31185D4665C0B0B8EC2BD58C91
EAE04ECE750F890ECDA1F4704F0435AEDF9A8F2CE0D1AEC8
900641C1AFAEB902F0934EFB2E8AE820D9E20442D84B5BA9
BD939BD0FEEFB953D0019194531012433E2634F387119C8A
41621DF28B038C0032508592F986C7C4832927CD1134B2B4
33F19F0EAF0EC42E040F5DF46162F083631C8FE00EE46162
E11E916E145716BC9F80B6273134D8C5143A61B378EC91CA
57CBD28EB525504C5A272E3327BCD2F4190229B8B9F7F279
3B3434C29240AFD5CA355EDE8B74A19A1A0BC0AC88FA65B2
2317B6DCFFEADF813E9F1192CB930247C1E241117C844ABF
6D99C0AF36EF7F66558E95832BA723E9FB2C1EB3FE8686E7
A12CE9B7A5C8657192C6BC5855288AED86EEB87637CA5CD8
978365EDBC762C10187F212BF9115D79FCBD489EC135AC9A
690940EBEA1D40FE2BB6822973FBE0119C242F1635931DC5
7982BE271D2691F6CC6AA906714750C2DAEFCFA8C01C8BE3
2B00C2BA54B99BE354884C0A3050A543DA810615F02B42FD
883DDEC3E4C6878F27D866B8A8685BA33EA08DB12CACA4AD
D1996CB317DA1386D3FA3DE1DF8B95164700F351B51946E8
218C1ACC812BD8EA24C2C50E7F4131412795B8144C3A5B7D
1F0E7D9174E1280D63957806A344B89A5BBEC53F206F490D
8C4A9C341016B2D74E1DC75478A20A57EC934C38E6D60DF3
0B9772283AD857C9B9F1FDB81A826FE4652F6213A9046350
4AD781E0F0E5BE2EC6ECA33F8D183CABE7FCE59B42499887
C56ECCA0306F0971F98DAA967AE7AAB046D84D3A9770DEB9
E4B070DC98F4BF10DF829C1ECE2BDD30A45D53B6F3F6BA4D
239CE0F2E57B6D19769C669D245314B1A7208202406D1439
4D09271AEA1915A0B1C2DF6D883B238D71CB70A2DAD1D04D
9842FE21812F95C1D8961C8ACDC69D34FC8DEBC33EEBD953
FD5A1DDED7D38B987B582FD91E95FF8ED84DDF5D486A00EB
6D712A0F71A8DD9BBE7962CC771F63A338CF5B7463B677C4
CCE2900327EA3A34166D9FBAA768FCE9EAC3F9A97EDDAD82
0117053F4A107000411FC39134462CF155F83AC7F7D08F97
A93AE5F7041656EF301AF20261DA6140DF3B148CBC4FD2A5
74EC899ED93078EC109561314A3969F0A109D4672A1BC327
117AC491A916479BC439A23B64A42E45219233DD0AEC4EC6
E98C7B0B3ECBC0BED36E00FECF3595FB24A66CBA09167BDA
F5845CBBA386319361D4042658606947FB96D72934378AA3
1AD9B83226DE9FF28ED8A8F726080EDE26DB39CC9A33D56E
834A29F3BF5E4257861F0535DFCCF584A790EDD4C0906703
D491255D7BF23C021FE4470DC4B97B422992EC162E39970D
678C80A5A5297C8CC1526C6BF319AAB9F8968F89E268327B
3EC1C1C7854004273AF8126C939F1CBC1A4752D686C4C394
DBB0B3A32D6D64A97FA5A369AB154DAF72C139459C6DB03E
ABE0D133C8F7D99DDBF5BF677C45E639F6ACC8DECE3E4D1B
A4EE394B2A208E9B0A74C6D57568E470F6E658C44689FC63
8ACF5A7D9F485CF8118EE88EA7703A79726791A35418E2E8
E49035A4751D0A51C783FE5C2AD28FC266B5716B9CB0EE66
619E9240F0825D658FC11BF7FEBA29B4BF1DE81CBA839922
5155E4A772C8345A0ACDC6D4D08B12DF21FB1363D04F00CB
FB331A6B32B8669EA5F3BCEAA32B06CFB0377C5CB7BBF955
1937AB1E560F4BF2826B2E3342F90CF11D1090A13BF56C97
5685188A6CE67610B69F2E78977FE9BF327384F3F34E19B1
19208AEF976EEA1A1296AB46BB8519E4E35CC3D26D2B574F
B7C001DE41AD0EB237D223995419354C2854D1BDAADD24CB
09736469AF3DB080DDBEC218B19C0DDE7D1E200E27744BEE
9AA7BCF1582E4421CA66FEB80D6E803E15991F1C0D99DF73
5BA881C421F8F98A382FA76B6C63C41EE53A7BFBB189D74B
96CE8DF6DBE539BC389741C68EAE9BB86C6AC98C827CC62A
2B06A2DFFD677E75AC0336108FAD6FE8B43F80E483141C4D
856A4F6AF406C1F663867E9176835214D9CB832D22A8DF4D
A0D74C6090806AE1B4A9A676950B940DC37D28F762AF66DC
1A57480F81096E92AFCF2B2D6F99AA159AB836F3AA88D5B5
64990481CCABBA7EF3B2B9AB4DBB0153DE52D847D1D3E0CE
442F5F122950F44D0524953070C68BCDF53FF7D978579724
54E0734814946AE8044479A73BD9D16EB86D8B980A58CB98
3EC0CB725707E9E932DC2E47CDE7B3A625D3D7AFFCC695C4
0966707C33FA06934CF0411349E0B34A7D4B8E676A6F332B
0EBA22FF036A6CEB3171A2EC4A594A763288CBB732B125A6
DA1DCA0F6E414B6634ECF34DE132F3320BE35AC8D6AB0BBB
1E939670516BDD2388994FB1B3EC9936BBB29D3DC5282281
5710E67498B2FE06EC199D82028FCF6912070F35E21A7CD3
64C908483F58E3BB317384AAE8907B9318D9F159DBD34541
C04313B59CA26122E79552FEEA3416E5FEFA1F52993DF8C3
FE7DA75A8BCA322A2CD1E2239B0AD2045776ADF9AB16CF4A
6A1EBBA8217854DE4E61B81E3246E7908080CF4FB29B952D
3F08E23E82773EC74D03D1D19BF608D75D3012A679C7848C
49D948E0A5149A7B41F73C30B62FEEDBABA7DD9FFA9DB383
BE9EDCCC18BC221B5440B367DCC31D2EFA77F01F05A5EBBC
3D9F6E880410A2AEB8974B7C2A8D31B062225035CBC35993
73EEF67383FFD6D4728D27750644D8B8B34573D26FB1D15E
0F8ABDF2B9A98DFC98E52F2B25BC6D49F742FE8CAFC6AC5F
B6A47F78AFAC6677C5DFAF3CC70090794E6F36B12A04F8E4
453EFCFC72DFB7A69328832BE24F6FDC03DE3CB41E0F084B
52643B42AC8D23B82F8D15704B0FA685F919BCEAF4A493E5
95475FE7F5016A6FBB175EBF3B13EFE41FCFE9249586BB2A
DA7AC89CF91CBB2394F52DD8C851BB6A8139EC2E1740D2FB
3E1E886D303E4F8B4E22C14B0DBC31BEDED2A1F983C82ED1
8AA2C1EBEFAE434A355D86F3303124C25F51B6AD51CD5E90
B70E6A8BEF61A059ECB8534303D05DDA564A2E102DE9E1F8
1F521182C7E9121DC069C1F0C825688EF75A11F8FFFE777F
34480B2F2B0BD9D23FC8F4640AFF4CE96FA837BC77739CCE
7209AC7B344407983A3B0944B025FBE188B88F256223DED1
8AEEF6FEB5BF02F8ACC6430FEC5CC7858EEDF9C1DBBE8F09
18802584E6C52DA289A61C4164070FFB53FAAF86673E721C
A6E3E4DA7CDF50842DFC2D944E701AB06180AACEF2B9AE12
92816DC2ED34615CB496443B8B9B28502CD7D02F48D699DF
4D4C4B29D48A6158703135260B5333574F76286C40C6E866
1E84279A19CA61D40E1F5197BD4DAB963A27F910F4D34776
50C9010ACAFD193271C799BC07DB1831FD2132C0D9FD5460
38760FE446BB63A6ACDCE9A69717AC29A7A04193E1299FC0
0E425380B928052574D82A3F604162F3021361517560271F
2A74E081FB3B8C0F2012E1CD26DCA9502F11B2E2DFEBC98D
19C312C65129AEC09ECBDA2523C3623668A5883E1F0B3DB9
005DDD8185F861CD77643932D00DB15E5D1E366F527CA344
B53CF919B35C8D3C54FBD2ABF3285F354A569B10FB84D797
5385AE84AA747F74EAB56891550D850AA8154BD82C04E51D
9CDFCFB1500A92E143FB1FB2D1F455539FBFBB918A6CB54C
CE2EF2314A52AF40AC697E942A88850562D8E6596F2CBD05
A857DD168B22B65A6DD2EA8035C4EDC4B890453D14BA6052
491271257C696C1F5DD5F48A6B77591486B7F90A63D704FF
AC8C3570DB6D358733E701C799BADB84319A41DAD02D1AAE
92D278265548B709549E112335FE56A885CCFC9126228C72
4D22FC080CE4FC22AF165A13030CCAAB82D564470D2018A1
EC1E62E3B5D8D5099CE0DAC5F4EEC04227B814FB74E1D1B3
49A947C2E93398A60B4F4B942A02CA6E40F959ED170BCE47
FE51BFD37834DF1FDC705F8D4977733B696E2C28792615D2
33FF9966DDD692427A9BC4D611F3C74CF629A0544A1A7ED7
4B96F09765730E4BD2F423950E42DB153CC4621DD9E3E52B
D04C8DDD186362D9E173584ACA2863DF9EB74CEB702B6580
CB6B662D7DA39838FA98D1D0B3EF302F7C5AAD30A3EF9719
303EAB880412E45B0E73F20F6B8A30AE9ECB8D010EA275A1
500F3A3F9106BEFCED7EBA94833E43538F72AF465B611FD0
941F7ED08356987364E25CD0D3B2D14FBCF2E3C109ACE130
FBECBEE4188735B886D40E1E00CC4654E8F9841A58B38854
759D79778FA748F8E828A568C45B7E2774A6052E0A4A06A7
C36F70B07ABB4B6F238D460CE6F28636D4AE5AF314891885
B9E4D16B5B00B755F179E9DE0DA7537BEA9FEC6304E97633
F91A0A3779FE5C52B3D7FE1828B466728BEF769498A2DC9E
85694311FAC3110E162AEC1E7BE310FB12216694AB97C9CB
BF9655A0AE9E8F7E3CF3FC9D9428A2FE9BA415F6E2F790FF
E74BF06A00E844A3DACF96B8A0314D118210508968227333
A1545BEB0C24A6FED11B30C6C0E200F0C86CCD9849EBC482
93E4986EC1BAE0334662495E3261769214DFAB96C6567A12
E7A446EEF954A261876BCAD3DF55A9087A4D5F754A38D8BF
90687664F0F0DD609B71CD19D7F71971C8A05BE234586121
7EE44B7DC06792D333FD7863D364BEAE2CAC36445F381BF3
23DFB3A3FA1B555B8851D3B4FEE5084FA4F8729B22FBF3C1
5F2882A853996624C790F91A96BFE0AD3FA822305305BB9C
71CD8DD909F1473C92FFDF54839D15078DA83377BD3BB12B
57EED90F06FC7FEC2232B78FD370812F974C8406AB86363F
D50D44D48A7519B21C7ACF59DCD56066C7C328F680F2FB7C
7AEF9037C94EEE912FABE32B9261BC3D77F54EB49299DB31
61DF45072DF01B439B6FDA3703C8012267C872E60DB16EBD
868D911B661C7B361E268A75D545655AB38C0CA7B592896D
A8CADDDF307239226CAE73144F6B3037E651E2DCEB25BAC3
D151D12A79DB0B4A0C4F4AA8BDF42CB75F570F65292BFE2E
CE324FE20175AC4656EEDB92A052B2B9BC43CE30D54FD9CB
9D3891BD9906CCB4F5FE4A4835373D483DFAB350F684B2CD
6C240CDCC41A5DB56F7A88E25A04A2F3D10D33B6908D0F4C
27D66AAFCCF04A8DBF147A6F951865C8570BC445CBB67D32
906B5C15BB63F7BF43A9D30F4059E46C3B28FFD9045324E9
B6ADA6CC0B981BC8EDAC11280563197F8496F48B09A40BEF
29C5DCA8DE1538650319F259EB7A1E8A4AF694C0D01CCA90
3C2DDE5BFE7DF9994A79B510CFF53A7A0DA6240D532771B4
9E4D9FF06432059B15BD98F2737EA46C597EFBC519A40419
00E7EAC89153E940EFF31B5CF2D10BFB9C893965BC4DC125
D8581CF122C23E06A9A8781E6635113F76CBF21ECC520BE7
C21E194207946496EDA1A5C76D1AA10BF3A8CF79B9951713
8F9A758E67E3B8F45233DCB37EAE2C302BAD085B5815B241
F779055B4EF3CDB07DB23AEFE6FA301B3C5338980A64458E
C1004D19844F3E08145989ABCD41D3C501598963E590C563
82796EFB8F0CF9790A49F7B42AFC44DD94C7753AF4F64534
4933BC26239BE8E544765B276B851DE85770A04CC25AA09F
82B9AA3A1EB92D8DF8233E5C17A7DC7B86BA6D98186A862A
1E763325671BEE3E17FD894D7A4E1405315004563BFAA368
23F636DBD773680BBA7F736DC77C03A2BE53125CB51C2992
2917267497B3600043AEA65E940DB70A12D8CF6F777DD61D
AC3DBDCF6DC91E68DA9D8F1454E47AB6CB3AB3CD7DF5937C
E9703CF79085CDBE599A2B7ABB4C299B52A6083BB17C7C64
ED810920B896CF635E04E675493473431131ACFE68F01190
DA0300E244937AF701F4D93B6F49F8EDAFD3BC89659F7443
78CE95AD9CEE4FDEE7D5BD1F22A8547CEE7A6F7E31046145
57B581CBF0EF91E38F4A306FFB3791A0BE6E1FA96BDE0703
0459A0F3899575F22AF3635B651AA17C14CF707DA800F34D
A1764FEBCE8F1D47A054CAAE837142935C100B0497A50BC3
35D1463594BC38FE8E9890ECAEB591F4A0DDEDAD0E52623F
2D556E5D05C09DF185E89EEC36B40B1AB54E592ECA35FE36
699286EFF960E738A4777087F2885461376DDC597F323FED
E65D9ACBE1299E85DA75A95670458B74BFDFA9654A81F5A4
61CC17A2E4E079F41E43FCF5DE1C60AB2110CD2B5B71C14A
0870537F7FD0A5F600E2B02B31878C2244F2E76C8D0AB421
EBB03FBC1B5372398DFB8AFE2FB2548F60595438E9B2B2E2
86B592895D7204B1CB5070D26A0BD648CF3E7B0C5F35E969
4885C79193F110271B4C0896303EF7DF196A91BCCBB53EF2
1F1A334D1DD150256E8BF56362CD08458BFE4A72D97E6088
061ECB0BCA13550B390960E1CE630CBFE8CB1A39BE6BB4A7
DAD6B19A11F00F92372CF4945AED62D01A03EBE8212755A9
37FB327C0E10E98AF3F4DE96CB1DCFF3DCADFD74D4E461BE
6F1D28AA4EEF347CAE41FABBC528AE71345BAC83D6572BAF
4DB73B1AF685CA34735232CE78BCC57584925239319A85DC
8DD8ADEEC15B7367DA19F4372D18C09B8962B2C8F2CD765A
38FDDAC15B13380AA42328E094308053663B853EFB0EAA6F
6D4E2FE1F48AC04FC7BFF48BFD0F0A814BDB764B28D779A9
7CEB13DB5C6BA7561E4E9E2412F73E9B1331B1EC219D2ECB
64284201527E4044950F3FB0FD65E6C6B83BD6FB98ADCC8E
4A80A80AB780D19BDD44E4E50FB4D29B174B7DC2D1A4F992
704CE7D9895218DAABAF9BDF944F7901571D0C24873984BF
ED40F7F428A194454D181D9977DE498BCE127575B182DAC6
161C0F62F34FB58C81153E3175705F7DC00C223F3D3E37F5
DB4FC9CD5AD6DB3F5868220ABBC8770FA69968579BD8EC70
18AA68867CE6B62222A6398DFD6ECC155B64455C8EE18B4A
403F31FDCAC2C7F26AAF02377359C838619EBE525C9FB4C4
0A30CDA0012B2F3FDC6F81D277E9416BE934FCBCCFF5AEA1
5646AB12B341734EF2B92E395D68F2CFA081018F748DB62B
1574D6AD7CABC7DFAEE1520289E0ED8B26DC6195C5F588BD
BF2D6B4034ECB4E1B19D9FD74DF01A66640B807487B7FDFE
65EF0FACC3599CAB31DE14CD30978C62469BF408E8EE7E8A
C051B31EE5F8EFBD8D772149E4494AA966F1CC612849386A
96A1FF454DAAEF40CBA155E4C3814920CD87D72D6566DDFC
1CC846F2A3304F3F1FFE730724C5EF0A402998A6F5BDBBA0
8F0238DD571DBC3E7C6B079FD21A902ABB4572D49F16E6BD
25CA497BBE71AF5B77F724201654FAE26589D32067766A25
257196F102E891E04ACD046CF099A2191ACE7F878B1C5CE2
9C2AA68D8E43C5D63DD0383CFA5DF98580A31CAA553CFDF0
1EDB7833ECD9D837E1D3A866A8BF52AEA81C80F5E1CA9A78
F0D3453130D337B281825BE9B9E80556971CBA825A669ED4
E9B864C9752F9C3A6AABCFAC7F5219BF08D96857BC8A45A6
2E6FBA543331FE07692528EA9172B41BA7D69AB6FC95BF98
D51D4BA457ADD451531E688D7CAECADA2DA2CA2BE7A888A3
A965998E9B19A975C912C7E0B0BDFA26CA2DB7820CB653E7
359F480A3475F89093F33289C3EDD28867C0E0F11AF79939
045B6B436C15BBC09C1E761C2DC7FB7007C5866AEB095B76
359924126EB52F43DF0FD1E185FEE3C49AFA9AB8E4D4D200
F85CF1B8907763039EB3A6E99C04DC210A8BD7D40B3766B6
D8EC8E67F43457BB035C39AB3359E25D91F9832D88CAEBE2
DAA3EFB9DA1720220C00CDB06411DE36E7A6767E184F5EC6
E99E114ABDA8DFDCEE9F404C6ADF620ADF313CAA04E5289D
B7F67BD650D55417BC48B4B54B32B34F7C3622DFB92C8FB1
8CA6EF62DD4E1445C729E3FAFF0B57DF5ADA90D714B906F6
9548A55DE683215EB74C181D8A991304D0355E5916283448
B883B29D2E25487067E2AB464A6F9D79FB6A0A0D6B13FB98
6FE13D9B04E2FB3AC67F9669C24B67EB32FC819BD787CEE3
53F4C5EA1F79FD6BA0BDA032D47D383DDFF4BE7CC8120CE5
F4297B82707D0F7D4C2728B79256EB06566CC762CF9F1D71
74928E13AE397D48000F8D8DE156D25CACB3C7487F0DAAEF
B932F001044C33BDDC70A20E05BEF5DFC3443C29E6FC1E1F
D917B8C5CF18FBD88482F6754EFC9D308EBCAB912A609D5E
C4DE45FA612D7A1652E3115D5897CDFF44FEAA26C11A5A9A
1B4A90C037A477A40100FD47E787AC0469F5E9764C2ADF58
DDE0B996955889E159C43D2D87E0137BD95429A51E9588EA
C9576BDB1387804FC5ABEF9ABF5E19143F1D22D438F8B987
EEF71A3C9F19A10D437BB6FF6D33C3C43BFB24DA0E8A879B
4500DF1A8D677E594FD9A34DCCE8D79B4EB3D268C7358414
6DAC60CA67ABD06CD6BE7262EEC26EB201EC49CB7494F9C8
B3950A9D299A5A732CC0841F1EFAE62F3DB20A707B98F3F5
F5B48FCE86CACD028E1943CC031160E2D1614773A9FFCE81
B8516E8D5B7C83E4DAC48B5E711E0E0525537AEA3DD5E446
008358A5CF4BB4A9D58CACE0C57910BB5D948DEF189D8C23
277DC2D0229F2154CA794EFCCACE247098FBE00B20634484
4B92E6489EE4303B6327CC868C42830B226B3FAC298E76C3
361CF6E5F5A9770FAF28CDD4B12FE9A8E911381FDDC68E1E
BEE6F265F2AAC6296CE5F2F278FB8258C065FBBE68740683
4112B496A4BA7C67F040A48A30C3F48496FDD3AD2A4A9E6B
59872FF734E631A2D1538DDC02DDAC2C785288A819A47EE8
BD8F9769AD884851642141FAEDA3AC0A95EE321504D34B65
AF017D1438C23D8D157B3479BF95180B203FA179B073A7ED
65BB58B117121271E32CF3700B1E51AE949116081D8F739D
6D4CE0B4EA29E7EEDCD2A8A817FA61E0BA340D3FB947087C
D841D9FFCCF37FDBDC9B32D49E30A30C98437EE47F8EBE87
F5D25892E050EFBD462C15FBEC91093E4115931A973D7738
D663C288232D8512986F0C2F37F20A764AFABF068F44CE62
9683672998AEECE69DA6EBAE99201B719384956C9FA4AAE2
F0E4243337B409002BC3B46B78C4D464A0928854B34DC6BC
BD8112AA5905013B1CD3175F8A9F133208D4A420CA898CD0
41FD1545DBFE0DE81857E82A46778ABA68ED0D69AB768675
B96DEEA03C7A08B45DD850D75D006372CC0FCD9A77FA0EF2
12EFF5D746116A53ADD60B5DFEBCCC03615BE5D78F0279FF
58A429073D0E3086BEC5B9566F4FF7914B45506F8BB9675A
4B0292082184E5727B98ABCFD57197EE14AB5893E2CDD370
93B745390D94BAA400A67A1DF78382541F5C663FB1E0EE32
6348DD5F84B091F85C67FFEE809B8807A9AEAE9B0A20BDF7
3F0AB3E9192A4117A013594EA6E671110262387A8F3CA73F
06119AC879360E6A42F96823D085F90990706370B6094F13
33965CB30E14E89AC80BB2A870B9D02E2B30FFA52A93F436
BEE4F2B5A0DE2D2253038491448C898894A35E1998DCC0BC
DFA8F315B46061CB5527C26054BB32FF6F5D3A548880B565
9EAF05D45A302F909E6850E28AF8A88E51799E20EF75D0E3
02614D90ADBD3A6B26802FE73D7E7DB53B1AE1E3A9A5DC98
6167565F208D140C120C866B29D885931EB5DA0BE17DD614
BCD32399A4B534A8C990069F302A737D1CA5E24C9AC4792D
7A4617C22861B6F8801C260191B05A006E3D0B00E409522D
42B26CB0B5A611E2DFE76250475F19FFEAEA0E8244BFD7C2
4BD3B6E3CA54577ED9EF5167CC3407FF0DB621FC1C375E50
A02BFD957CEE0446E5C79C45429269D4E0E70DC68A27B7D8
AB8FBE9C09E9906200201F3AFCEF79AFF29267BA19C63A82
11F7D467BFD00D8C1DBE090C98A8502C13EC0B7696D81FD3
1648624081FE1221BB402CFA17C356F50C7622F98621652C
4F07EBC64AC586EA050E3E66FD9F4966725A7B5CE118F01B
AACDFFA77E7F5257E7191A102C5944050EB884D688A8A873
73356100092067E4868AEB99D977BAE183483BA08510F26C
B4B087C878CF533B53E9E2E40DBC0015FC053E86F2FEB6B4
2F102233F92064958C779EE8FE5CA46E9CB1D83CDE89BC19
DB76DF9A0D2E9D21B57EC44C99653F59348981467EC2476D
9C83E73A6AB712FAFF97BA9D1BE7AE0DFE2192E83866005C
1CAF2A01197B8C4527F9DE9B401D01DB4741DF7DB0815CE0
206ABD07A7BB966A917125B50010280CFF8A2ECBB2858CBB
6F56EC2AAD6D916CB0C9A4D6A9B4BA5CF3C19A04D6A77431
EB71D191A03A31E3774EEA21CFC88290C021651780289A5E
ADF060551C405EFC8DDE00BF937F8E6862C956AE95648A8D
4937D383035EAB2CF617D1882E5B035B490761645414892C
0C2CCBAEB70C5C1F85FAFA01B1719F363D902716810F582E
DB38513D9F03C059319A626693344C99A51FFD0F1D9E773A
C4A8E8E8107EE307927DD8B3B063DA9267359D7D7EFAD004
EE223A391D53D3EC9138BD4E2B7759FF03D27A6C24EC00D8
2C8A96DCD659F9790CD521F779FBA97675B3F5C570E98506
A007030ABCC28A1D55DE02E7EC28BCED18C1F55E6E159D69
BF3F5D9BC21383F15DC81AF9753CFF1DE404B7ADF1599582
2428DBA7C9182C307419B58D65F049B0F02E1FBED8E6BF7F
FCED365E31C7E4F3B06CB6720875A33E1AC45D047D071342
14AF61CCC8414C6777B8693EC9C3DA0B2294B3618FEBB6AB
3152343DA08584541FB91884067C5204086FE9CF25045B63
8284FFA336620D1DE090C50665B4735690414F41AF630082
08B07367BA5021739FD246776978E355C20116C2781AFD62
76AEFAAA65DD608452D2478DE0CC5F48D7F9DDA1E9CD1CF8
A334673C6D865A267213ECA114753006A78C525D3343E5F9
7853B7188EBB964DB741A309E51292CE330B3CA4B2267FD1
2FEDFD926783A5C91D51BCC3228F2846243B95E5D858AF07
152DCC608EEB4DB38A04A526F7D3ECD5C28DAF7A0FB46456
C195E4B987F7017846663B11E6A2AA6D79AAB73495D87D93
09EB427CE32D46B1A9F92AC1C71502F10C6AD2FF070B0240
9AA0A0690672E35851D30B2F7D6FFC56ED63EBF01D6A0E0F
4433C2BEC783FFE50F52ACDEBBC9C819C45AF369E0D0D4D6
31D2CF9E0335CC194288AB74CCBD8A2BBD91663069F290EE
131ECB9C325CB03E24B910F9CF32BD16DE34E47CDE793B86
C1789746C0E04E42F2CEAA374FBCEDCFCB17AFB816D3D9B5
AF98F44129C560CD2DC5C8F869CC2A753905AE872B7577C8
3398368F41E18B1B52BAD154653A13C93228BC2C23DCE276
2FF0F3450C1AE1D7A8DCB83B55B5717B0204E45754095825
A157556C3CEB0EB035F44C3344353870F9C8276500E77959
040298A47B48230FBD0422E3BC671A42BBC7018293DA7902
A65BFD919FC4C8D0D1E063252D5693CA0DEEC9ECAFFA281B
EFEC1B9E91663F15DF8057375F36BB4D0423BC3AF3084D6B
C055A3A9EF84BD96AEBAE31C08B74BA22BD2489C70D25672
8359437F0D35CE717001F094EA6FA517A109E1622F75BCB6
57447C4630A238A0776AE4EAF12C356A81AF2CB0CEA59DE4
82C7F5A1D12EBB2E744EC2ABB3AB3626A15080FB7EA91D96
FC6A36281D976C7EFAED0B3711AFB25C4C527CBF07C94FB0
4762DE286A18A73A42AF4D5117AD8CDAFC2339B6FDB8F2B6
907F74828C567DA898A6AE1B04C7CB2110E899B97B87F39A
8020EED49FE1CA20317563B1BBA5E0595E443F04D34EC230
AE1AEECA1373D4348E4EDC2B7F393A8F44A351AA952AD7F7
665B2A1A727E57E6C01EC695D645428CED0F13E98CADF899
63221BA15C1E91FAC1C92F394250E0AB0D0A790945A36EA3
DE7D9673C1D22D88EE65EB131D4B791107649352403D9F5E
3F1197F078ED576BCA4AFF2884F36C90DAAD8AE2D4A44FB2
E1E07E5FF47BB806D94B4F86B5DF98FCD540016CA93C0122
F24B252235C2BFE8B59C65328258B8117A1A15CA040DE2D7
50AE2A92F034581CFA656DF65C237BD771F44C2BF6714902
60CA04458514989969B5D3A1537F226CB54E3FCE9CC9B134
89DDADE87F6D5014D5511C5F9350617633DDE5739F877F2C
F236602DE10AF723E14CCB421FC91353F374CFF4AF79428E
657A49B0359DD9ADB5482AD898F920A937CCE04569E6C5C9
84545DCFAFFB4D820D3AFAA6A34187814476953F7DEFDB52
CC195221D0ACFBAFC4EB3551744F111DF7C064D56E5A48D3
D73DBE7C4D4C83460245904D1359D90CAF8CA0251289838C
F045F83ACF0B3DB819E091A4A5DAC9BF6D7A3C1F9763EF62
20E29A4A4C36D7C2BADE368564765EE0353709CE47D0E36A
CF58C29D18E29B6C5F0292CDAF698E5366CA1B3B7BFAD78F
DEAD60E074F4A64FE5A1BCD0DF0E930259679600EC2BC337
ACF6CE7239649EF568A369356C76BCD3C16D40FCEEDDD215
B418B4CC233F0A069208C6E4FE15007B5EE98C0F6C351068
18D91DB63FC3849F9FA924125FE8EA1CF31110432C3A5DC2
FEFAF23907DAE6D86C3E0944B610DAD8016BA01609E24FCA
11130B384A56A218C1CCCD837B1675C3E8ECE715A06E869B
8CE7840450B26DE74B798056D30584DCC436AD79B061729D
DF5F29EB44234654485C7B4AB2BBFAD827F6F6F92C562C78
F199E5A725E25CFB226D5F52AD18FE7DA30C2380DA46B2BE
AA227D0D6C56A7CBC6AD53BFE99F8AF609A4BA5E1C19401F
F367C625CC2644592F959095E9CBB2C160FFDBE609A3BD42
9FABD207618D722C7F433137782D980D2CE9764B4301BC5C
E7DE2BD8460EA22B677998007C30CAA11099163DEFD85AAF
CB722C247F8FCE7459DB84E5F053A068468AB3327E63D11C
05D2A28308C44536C591DF21B02AB542AF982B81A4C5E129
E008DFE9BED4937EC85950408428D3A07ADB66EEDE29272F
87CBEFF490C4CB52ADB75B9EDCD63B29D07AD01CFAD5F5EF
C32A917FB42DCB857883780CEACE71F53E488EE57F63A0D0
0C828CBF002F9890E23543BCE470C91C4C5110890CD5258C
7463E6B410CAA6A1B100BC6EF5E146239285A43F07518403
411AEEF7B2999BCAB60CF7154867107E287690AA3121840A
30E718A0B54F911608ED9981A3670C8317B87053C55ECE82
196ACD3484E2F5FD51AC0CC667596AB0E764936C37A345B7
D322494FB91A4C307680125CCB5CC650AE5CD82DC730C659
09029C785ACF04B41E0ECAC2823F8A5F11BBFE240C0EE944
BBD6746F03B71CD58EC1342070EAD8727EE7003E192353FD
CFA3DB86B27201D0EF3008C9FCB88A182EB17C9A44F16886
C95132471E4B7C7F7167EB9C7FA7C92CD767FD1BA6E1FD6E
489755E829DE6F9AB28BBC63089B37B56A44A5B19AC4BAB2
8FB95BA35D5235D068E7F6CEB31DF7A51011F441BF7AED4C
C4F29A86A5415A0086FEC47CB3D46A71E8D224880657789A
B9397395E2574B004A64262A36C1C3319F9E28EA6413F218
B6F26251B5CD3E3178C1B31CE0B9F61F5102AB7B53C954EC
E810E857E06A6560B0199973C59B0DC075008BD64F9648D8
71B0A84B1212BAE33430BB15DEA1F86C78F8B5BCF967CBD2
1E4BD3ECBD1B66E0EB42DDA29826A417FEB5D7E00B9751D2
D0B541027F58642956E576EBFF19E1698613C167BAC09C8C
0E9168F36ADBAD5D084FDE06A265A75FE57D31A6E080BDF5
C5AD956C7D06EC8651665176733FB2EF0D01BE7E81E0F775
AE676E722D416F9A2C9A71115F78373049A37A2DE230924B
D1C7A48C6D714D3A81367C7040778549C0C531E9E9FC6BCC
4D8D72ED5DC1121BF5E700CF603C5AD0FF764B71A9468AAA
D0A4BAE93E55BE9C87D6ECCAD8F6F451E2CF7A4AD30A4858
24478EC1D846BF2016FE3FF161BCCA7B9E99EE6DCE14EADA
3973FC4140ED0FFA38B892A3ACCAD5F5FCF726C57393E8B8
D25EA293CAE971BC11615FBB4299E67659CB765AD4F7C69A
A5A54E12A9538A158E78AE09896DCB2CE31F14150625E615
5124551BC6F2EC932AC5246C6E36578F9FE89FD64DD8C1D8
12D12E9DE6479E5B7772EFF1AE04F70DD17B22FAC4781CDF
5EE944FBC71C076EBE758FCD38722BC6B6FAA93949E68C06
234514C6E1A1F394E5D068A09B3E840815C20E68CFE2ED2E
F8254FBD3D55BCB072A6F89F5294CA43CE9BB07E1A78BA43
06952E7A2A1D9560B8ED418EBEA56366CF7478030F9E9A31
5DA678192BBBC15116608163F70962873A4276AD547CF1E1
239045220EF064D13637D503A6079EF21B5F2D02CF98FA5A
B8A86449A4F75A8A7ACEC855AC94A658105908F76432D70D
739D8FD418243AC2A794B3F771F7C6BB0870CD56F5D854D8
2E545E4E97B118AB88FA22A0F09B751F4989B042317BD596
F3CE9D17AA396115C25DB18F5EB7A36AEE946054E18EE78E
7200C7E935E5FA40CAFA8E9CF67FCE3134F82062AACF0584
1E4C55A649C5D6D0ACDE8AC5AF9FCD871DCFA177A46340BB
1C56193D6E23AC74F0DB64DEE0913C24CDADDFFA323EC4A4
AAEA04D2B11931D58016A9B68B3FF543C7BFD87823EA6ABA
AB067AA6F2621A518559B29898E3BC9B7E74A94B671ADDF9
5BFC790896C34A050D05230E17F96C32DF60824419A0AA5B
19237CF766ACF6F09833CB64CF7D3DC666A1A6C3DE598C86
CC94EEE6D6CF47CF5129A0E7B3856475A9F699FC8C253322
9F8DC736AA8DBDB3D2BCEC81D713B719DA7A11E6C92532D2
2BD7EDA5ED32D239BF16116580A245587869587C0D5A1E15
30CEFBEDE0899440BF1887EFE92330AFE13159B25DC59897
37838610BB3944FB3DFBB14B95A4007705B3773148E58AC3
875F3A4CC0CCB75CFD9FBE685AF3C891B8027EADEF0A65AA
B2C3F1FFE0499C1CEF7FA485F3B36EC937FFBCD6F45B657D
0ABA97F082C6AD63E0E0D5FF4925B04D47E55113240EEBEE
C40D0FB374F55FF2C4A524662E65AF04EBBA9F1F2FE411EA
94AA7C83CE1A5A313247DA3D04E7F5E913E133D10C0F06E3
71F556E9C2C6AFD5CE110ECC548B4E2C61180A6122269DB3
DBBD13DD68AD0172A852492EFD3C5BA42AB6B2AB4E9084C5
3DE1ECB5FEC022C7CF8B4CF4AFFEE41D85648EAEF6B0F5A7
8B926ED6062F9C67DF686D1CCDE0ABF93A60694B64D6BE28
0C45ECAC55510A994395BB0643D7F94717F3ED533CBADA70
4385B3786EF3342D0A4119588B89AE8048C67E97101249F6
571BFA74B68DEC571B7DAAA0AA684FCD54D816006199BBE2
893ACC7C5E053D2D6A53B019A42151D23534D1F0BF24D89E
59F9983726C300CBB7F6575F3345473DFE04318CCBA1A7D1
3D21F69E1CFA3112A0E3CB2534769547D13308C5F3DA3F17
40224B9C7ECBBB69D1A5E0A3A0F249315AE1E7DE859FD763
AC5F6B5B407E7C09F0E09240755E8D96B9C1E4DEE228506A
75175389FA135492A540664C8544C8B4A8AD19CD685AF350
8D33C870B0D3B61E73E8A66E03ECF8067162384298AE03AC
05E46C37F3CB05BCC68D7A2EA1AED4B5618FD09C6E8B00C2
5D741AC904507F1BD7F6264D7373A54FD2DCEC28839EE7A8
369884F0A6F7146697435EA6E862EB39B10ADF7C703CFBAA
9712CB31FB386198EB78EF9023C5BD1A1081EE72B7AC0AE2
7EF8F229B824EA82F8E8834E799A67A0B946086BD9AA582B
27C15AE37F283241AD953233A831FCB48B249B16831D24B6
374FDF3468856A576B72D25CBAC7324D592ADEF50F0CA40A
7250739EE5E44B428B9F2F8B1D43352AC1E5A54083720DB4
886B83A6BFA8D5F608BD5E625C697BD701CB3D34B7C6E19F
F0928A7B6B65CC3CFC01E034927D5B4094BA04E66A3BFC91
E8DBC10EE210F168697B6B8C7E866B0564733B597499B8D6
FF6F1669656A8924546ED215700F91A4D511191935690D6E
2543ECA4058C02EFE62E2131E9A7627D6DCAA6C51A13A517
2A41B4A4E6B7621EC424266E7CEC019AD114D717A75B0B23
27AEA8910BA345F1FEF67192E962B96C08FBE9536073AB65
42B666685F8B3F3B4628E7B689E261F7C2B3217E9A1A4645
CD47B41322DCCEB5F0734C0CBA55201621929FC16E3D8FF8
A0370B7E0E7FE2FD7AF27B32E11436DBE60D85B6F0B085DB
B069DC359CF738F285FFF7AB6EB79F1F58F5B3693CDFF769
C59EA69553AA4A97E01F077AA7C26644CAA01CA97C74A3FA
51E2B3A3D56D88BDB4F77CEAC0D146B08C269EB50D7914D4
7AE40B10C2564C0F975FB280A9908C01ECB54A44D4A1725A
FA547020C95380FABE2D98E5DB91587C938BC4A294F97DD5
5F62DE8B683CF0257D589B5BF067E2C5E63F805728C5E566
76597114099D0FF16C78AE64E62E4E6659891A288E86BEF5
EBE402FB0485FC6D9F4232FAA19577CBD08B75ABE182D9E2
01AC2D59DCEC9AD8C06A9591F71780DD2E9F0650AF23A079
FED2A77F2EDBA371ED74CEE7CD59D4A39CFDCC6E4F3B79A0
9A3A264F48D2BC4B08671B0D6EE58B4E0E87D19CFC561738
37929473A58993ADCAC09ADAEBF6A352602126C8870B12DE
194FBB4CC683604008A42929F96E7C3BB2A48F201A8019AC
F57B61B59215B570F17E209F5C06EFB8FE1072053DA57403
3A9589D456EEE28C372675DAA9F364F8034406B8C072D86B
CDA1DB3AD4F3369F0F597EE3D925C39E55CBB5913FAC51E8
9AA3C9D6EB62026B8B12C3032709E52CC782F0FD0BC44321
821D1BE303E8FFDDEBF59F684D40EC98714A4CB5657519EC
F8F8F7095F7FF1111811FB2B63F431B529CA740EFA9A3350
57E6E8CBBA61AABAE7E0459C5E8955953C3A5C9F414F2B53
22FE84FC8E699E217807F5D0199F808E3A32DCFF3158B6E2
AE44F6A36900C52F2444681BA484DFCEACC7DFD0D6A2D7F8
40D20C00E12583B87DD2214946B05A3F33ABAA7D9EEF075D
8844C01D1EF1FFD58BE4BE7412A974BC0E35C61EF3FDAB7B
5F7D81FBD5A219998ABBE4A40AD15810E00E0A1A3D2DC9E7
44B22F004D7A8B08B5794E3D94B07CAD00EFDDD87D0A68B6
E0BF4F81FB633EBF0092E6AD4EC8B6834A85DF4F639F94D0
E8BDFDA08887ADC1BB6F240FB185B8D0A375921AA82B6694
440253D277F9E97932FED4DE240EDF6F74BA373135B480EA
549F18592C1C6417ADC0620491AA0FA727274DEBE495F7BA
FAAED9A8C586879E8AF8F1DEEB4F0EB7A681C58D9BD69527
CD495EAF48EDDE5984C3CAFE2CE297A2C174776DC3AE3825
EE1EBC7F5A326FE52B44F4A0DE6FABDA8E9D10EDF962EDD6
4995C15FAF2F80DA43B819E776E5D31FFFB3A9FCDDAA8B79
114BD7F8FCA026BA9874DD79174EA1FB7275B8C7F2042871
A52EB64277C3E193C1E4D4EDCBCB29206040810A2034BD41
7C937ABFA87B4E1AFECF2DADED584AB56BEE6BAB335704A0
054DF15BBC2276332B819212A1D4D9D7EFBB87B586B48FB7
A474DB87513795BE16E2A83826E8BD6172DADC5580D8892C
A33889112918869C07E383A0493A0D88782E8FB1283BED4A
3622E243F9BCD0C6AFF42988B9C7FCD35CBF9CF5FF311441
A2502BF015FFF87E14CEC12BB52150FE301EBBB54DF91033
BDE5124EBA58CD1929DD63E9A4A987296D93E78378BA7E21
5304DFAD2AF734333B82743AE8AA10A647A4BB4A95611CE5
368E76B1FFA704BB5EC0143FAF7B1EC0264B02F2471A8A8E
0EE1C9FA0C3E4D3D261B4185AB38E58E35CEA8C7A5D7776D
9CD1B0D3AF7F4A697DB4C0705C39E6E06EB81A51CDA9ED3D
ACDF3C6AEEEA9185245D5B4D7BC8526698DF9670B397A8E0
71012A9F082EA541C3762DA042F16F0418F03A069BD80CA0
D9C7C94329B348C411E21750E7B5E0188ACF677884213C8F
FAA1FA47DEDC888043BC35BD97288C3E3E8702E3790A5ED4
0389078E174E9EE8095FAD41BE4891D4D6121F63665C9BC6
8001C890229E8AE714682AD08CD11BC25B883E6D35C1AC2F
00D3DCB71064BAEAE25CFC1317B3665ABA285DAAE8B1A5E9
7BA68B7CF5A508B49F57BCF8A8A9C96E47BE2A32E8C7F3D3
95312E89F5ABD005D8CF5D763A9AF0E7A964E3E2D5D5963D
A2150E7770743F20AB099942FD07BC6C78E32D04A6A81E19
9B769E4C244D0479991D741C0715557488BF92CE0EE1F487
4D0016CD63CCB3C917A9CAC0A656ABF1BF1F79A42D9FB4E5
3FA979D7CA94A5673277A81A55B8DF2AFEA1BB03783CF6C9
B0F82E85D9E9BF74299A59185845B5AEB967F1A84BF25D45
142D54F33AA60E3784C0A71FE3E89CF00D1F45050F86C00C
336D2159C49EB770BDD020AA74A876FEE149B2C076AB1901
B4F923DB1BD7859C158079CA3C67D195769E89E7BEE332CE
9D00F289C962B1266C7597B9B12975F244DA8B9F0422794E
EF69B5993A01B037965F2834C944785FB4D6050CBFBF65B4
1D9C9AA4BD15F3B6DDDE9771259AA550F17CE01383DFC3E7
61B0C2F60CAC0792E38DFF5BFB7F92D6A10C07BB9895DA51
5D098CB9A3A7CDF25EE70D2F183091CA341BE5456E4986C7
7D5C0748B9AC9B75FB51E882FE5390FC65B600BB8D91A35A
72E3E3B8199146E1EB60F702ACA6DA0D02ADF40C8C049496
DF240A10A2F1FEF02FDAE685A3973500389D69F01CE513E4
DCDD2CF23AC7AA3D2519EE44637C6DB443B694A033E5D9AD
E591A28130ABD26D11BD2BA1CA5E6E39CA3D28D63DF4B7E5
D70BCAA1810D6213A77935D751F52C8743C809C37DCB02CE
E67C0050D6C3DEBBFA39340A050A9D027E7877692E62A984
E2414AE32219B21C68743B6F1AFEE3E1404A3FAFA98A7FBB
B1DD7FE39ED11FD16C8F151B6A4228D54417B9B9BAB88C74
617EA03C0238DBAE762058ABF591773FADB4123A446EA41D
C46F5E9D3DE419171E31BBE6CF841401F5075785C8B9E4B1
D34EB71021AC4EF1D15E8E0B7A707EC8A1BE9BB86FA4F80D
A905880A72B09BFFE2A8F5913A8387D198488B4DD839EE61
9759AE86B5CC71DB9406A2ABDD0537940D717E2953E4573F
BB15FEB841808032E9DD625A2E9134947BE7DBAD011877FB
766F7212528F0513A95B91687770FD435604AF23C707DCE4
3D73370B4B8C4718910BEEAF025582D7CC5CE951CE5398CF
536F558449ACB5007A8634E303B93568E2FDF684FD828A3C
FF0BCAB96FD7666790B8CF26BF0D69486ECF67BEFDDF8B42
83F59F7EB2AD0EAFE24DBF6DB09B1D81DDB7805263635B46
157602AB22286DB308C8543C122BD5B6A3F6367DAB09F5CE
A6452EF736D5CFF79BC3C3DBAA7FC7B4B992AE7DBD87650B
A9D7DA3BA333C2FD8EE18589DB2A984294C0504787A57FFE
3E3377C9B4AB0A7A64DF073874B9E322B2BD3536C741F5E8
73A0C487172CD3B1C9AFBA9D587F928B0A9A379ED64DFBDA
45C3EE530D205E4D46D5D35EA1D32D9385217C5DC30B9E18
04929AFFFEFD318622A0AE1E7300E80103AAC778DE03FCA3
02A01344BC4CAA53269780F894B79C37C59C3998B1B73602
4905B6FCC9F01B59FAA17858B244864A7F0D872A225D9D6E
FB7177FFE0E718C91348A8302C57129596149F042FCC99A6
F46D594A1FB8F5F5FC6B332705F891CFF41AB3251A40FA98
7763A5AE2B3CE4592CCDF64BD51942F3BDD437E995A6ED78
B3B8F20652D349F5C93F9579B9BDEAC300882D0A61BE4745
9BB79648CE27C21321FA6138EE6CAA81AB467AE6F04B166F
F7AAEE1F059E711CAB253859584750A4B418F70F3EA58C89
D060B1380EFD22C62F4CFCCB0EA9F38107082268C9D2F1DC
D64C3D7391A05DEFECD33AB76CD8C00FC0322A514F43BCB9
A6FABD413D9FAF90AC0A8D97F82B1D28761D9B3A7A3A2E2C
0743EA171C8F7E2E5652F8F3CB5FA6160B6C10A7B2163A09
F1BBF6AB567E8A418A0640011065C108355B73D9E8A13E06
0C011CC846B6C1D17748370C507A6087879724C77E6F895C
C73288676D7386884F209E397AE2F0D644F0B58F5AEC396A
B569BDB58E1278E882EA79ED03BB25BDBB6F30EC198FD075
1AB097C34955500E0F14A403CB274DACCBADDC5B282ED20E
77343AAF547903DFA492E433DFC3813CABE3B9675CE3F963
93E56AE559EC7BF41251B6C84700465BF78E0757576679B3
9E7CA24A44F2A04321346AE94D53C63928D7BC899BED8A82
04FD1D8D78453F5B76130BC6EDC32A809AC2D8687CBCBD25
8D827EBCEE3DC3AC0E4FCFBE7FB983FC0E528C182AA65A7D
85C63DD5B13A570D3CACEE930FDECB732C1B871F046AA2C8
1F719EEC174264D8F7818E7C2EE8721941C9CD2872D5A15A
320057361FB2290EB402EFAF0E166E92B189E224DE370043
7EE50C9A0A119614BAADFBB2FF1F3716717682D83A1A9635
24F342F67741BFD5CCA5E9C5C0D902CB6F93DF12203E891A
3EE0B097849C633AA6BF497BB2A04E1EAB97E21756C9B491
4DFB2BF0FFEA3B99AC3E8B18F2723D4A9CF38E7257F05064
557844D3C791C5E13CEE914C1159C0227665DEC5A7585C52
E12CAA734CA519989A767086A1FCCA376C2A9B71740A7637
CC58C605B4BA16951B3DF7249561E221C468D3E3BDCEAAA8
B2AF478115BC6A4379E124EC1A8567B7C3F391BC0FE1EAF3
11A3F472D1FAB6634316D279144BB75B1B7065AC6E68CAE6
BE7803AE8A8C84B7567352C2E3A78BE424784406D33F6930
A0186815401B6806A9195F1FD2256543510839753B4D1D8E
11C6281ADE09440B1E4D8E6A816BEC60072CB5E4BF5590CA
5A529ACF90CF7298EE1955025C19BBB58A85875C849290D0
0C86082F2F99EFB85BA059EB713A74899D427F8C3C3A3587
2C1E11FF5A6DF6B418301CA51E87241F08D5C6ED2AD4D619
A65C07D1025AEDA56C53DD8D2A216C2725762242B882BC25
F3F7DBCEFCC20915945CE89F9912752663637B559044DC80
43A59ECAB48D3784CD83B0887E92D208A0E7CEC6DA3A7146
942D19B88EF907364D60442901F7A0A221B989345E42A176
717ABB96B6B7DFF066B33468B25FF28702C59E13149DC31E
1259C6B02D3705740F4ACFF45E6533D0B10722F060B09746
6C5250DF2B3D049491B2D118CD85C1D20AD75E21E4A33E75
45F92071F99A476979F8FBE6BB46C105B0B1C36CB539D0ED
B01EFEF2DE4884484C5154C195E4E1B76AC7B1846D3E1628
63F3C21D32DC45978E2895DE7010358341E89562CD0FA5D3
7D16DC5E00379CA0F13FA261AD5FDB56449ADF9DAD3810E1
BB656BB3435E8CD66C9B8C751586579BDC951D0188A5753E
40EB29886A607B99E6D5D8D828976B7F752DCF89EE5823F2
3467CD5CE610146F8D39D876145AD481C95954279A1B745C
337CD30C26D92A6560D82CC552C41A5780AAEAE426835ACC
EBF4D50EABB98924EE598D739079924B5E76281115C30C50
05D6183AF648A3508F70D883F89F71EFE7157F7D8BFD95D3
440894BFAE62ED76926DCE0EC00A81EAEC3BD2B2F8740B7D
00A569CCADB17662483CA36230BCC6956BA5C1D5595C044A
45F7C003B6C3F28E3E27D6C3C704D149BCE7E10103730D99
017E404BB5D8D41733E5439C543E8F1BCE608D3CD2649BC0
7A5BC59EBF4755AEC4C755663C6381B949C40B856FBE28A4
67187125CA3CD630863CD90DF96C5A863B8E37D3CC75165F
F350816A79C72F1D8477CC8FE60D81C139606EF73D32DF4E
469D6127ED99F0C606944971EA2E953DAABD98A4130D2988
5E5319090C0C136D1838A01DCEECFF79896A942C1921C983
»;

for sort((0^..^($testvecs),440..449).unique) {
    is +T1.new.finalize(Buf.new(0 xx $_ div 8), False xx $_ % 8), ("0x" ~ @t1set2[$_]).Int, "Tiger1 Set 2 vector #$_";
}
for 512..^(512 + $testvecs) {
    my T1 $t1 .= new();
    $t1.push(Buf.new(0 xx 64));
    is +$t1.finalize(Buf.new(0 xx ($_ - 512) div 8),
              False xx $_ % 8), ("0x" ~ @t1set2[$_]).Int,
              "Tiger1 Set 2 vector #$_";
}

my @t1set3 = «
B99F2BEFF1E64861ECB3D91B04C9BA4BE6FC0FE7CE9653F3
FBDD470EE15111676B477532D293019EB182CE9EF17C0B59
D68BC840B64746A212810784CAB2BBDFCC49FDC743AFF5D4
D6B280F151C286E2B8A78B3C80F181E192BEEC0A339F0453
02FDBD79A641FDB807DD80699019BC97DA69D692104FBFCB
E30F39FBB7BB8C191E2CD9A970EC806EA64825E4CCE782D7
0521BC64D5C188320CB9AE9D2A89BE291352F77C3E44772B
60224F0B05145D00FB3C09E78522FE17FB7C64AC90AF4BE0
3104BE53E8A19ED90BB03DA1FC5FBD052EE7D80B4AB31AF2
ECE34E9D1BBA44EBAAF18254812DA74E1B250D874049471F
7FBEF692D8F3DB59912960E1C4F255D65CFA93E64CDA00B3
180C2B906440036CB48ED7DBD670CDD06054734CA9E3E531
EC1C85F1C4361CCC979FA94565CE47BA766B26C03E0155ED
0CB103FE2F16046BCF473C0EF5F30510094B327A811303AC
35E499F9FCCE1B0CFE442FE493333763C8E5F956A255DDBE
E8CFB0B7651DCF73891553FE1AD3DC0F1BF5CA89E2243771
9F375EFC406EDB4DC7008C6ADF57CE7BC6072BD39593651A
C40960EE520A3F6E38938C4D774695FAB9AA99ED98724686
C925142A228B90DC4EEE8D740480095B72EEF71717AF230D
49F67D2EE9E5E07754EB3FB587A5DFC9A5A5F1AF952ABB90
8EA4A53FB597B8612B751B25B4D67A3211BD9B55810830FB
7F8A9779678C20D7B26126CE54CF40D586D639038E0D898A
45B2D2247DF2BE8762F7C2BFB1D282DB99686B8C61F12DC2
5AF3F43A65A123F25318B108322F1422FBDD4E2088B3FCC5
255BA44E43BB186AC95A8BA873F27C17D4209A7CDA0743E8
6CC1C5FE50A6A3E1A8C9151E529627B7825E797CA2996763
4C8E2574514F1908500775D706E66CB5E098A07763CB29A8
80E15B5D7DD082786B73BFA7C11ABE846CC08992D82082EF
9AAEDC20A2B5C9B3AC1FEAB011E0BC63C3F14A77716745B4
20FDDFC28496F4B7C474D1DC0D2556B71A0BBE0008FBCA74
6C5437A6A7ACD3882300183FBE7B474CB7C0E45DF4BF126E
3220E4DE2B99FD792057CA379EC8E0FB9277B2B0AFCAF7E1
5605262344AED0F46F6AA94AAE288A936B3783D7B0C96EBB
79FE05ED00EA9FBE5C5A411B7BE9D0AB7E95FCAC4944FB5A
5632B77A86B06F60FB38F6F8D7B8ACB076610CDB17C45F3B
6F1C90B7CE7D352A1EAF908EAAEB69DE70CFEA924FA1F322
B0582D1882AC0B5C534293ABB751072EACFCBCCD7760B739
D56CC1F5522E00EF7F25DA7C524CFBEF639E04ED232FDA74
FDD3E72BD41951FF6CA729207F9EE320F277C240C5ABA6CE
25C341EFBF8E0197DD7AA0D098BE9C262EA24A5ABF1956DC
3D8E11219C1F4A655DE818DD5BAE94724292D1FC1F2C0451
78D0C2AEAAB13E6957111303BCE092987F38B125654AEB06
76CE1A38D5408FCF5C60F231F8624D22EBE4F3C2A77833CA
A1478F92ACD8AFBF1383F566E8074249C576C6C01573C816
EEF978C6730A5A51E0C67CA0CD591718F48E586924DACAC3
D244E12581769DF1260AD3B9BADFEAACAEC10F08D0BC20B8
6C88A229FF9769C257AE208427684E2BD2F2616FB11DC61E
F9AA7D877F5B192A751D6956225FBB166E70BB5373EA03B5
8E5FC1110E41E417B771F4860BD7180A8771EC8BE3F51371
16A04075744A70693EDC2823E6D3F53F8BD7DA82F61C988A
DAA5DCE2246AA0B16DFCB5A6DE1148A4667CF130CB0FE57A
7935C3EAC77062032E4DA41CFE266DB7B1CEAD6A92F36531
12FAFE2DD9FE0F928585DBFFDF2FC940329F4CA2C428FFA2
B5AF46B22173E04EC7C73D201D230E76AFF7E25E7C64C9A1
04A39346609CC870A34BFE4BC381391E5096EECC0420CBE0
8E30346B3CF6A33A5DF960349E79E318A559177A10132AE5
E010FE9A2A139AF552D7A22D5D04D59ABE9F26099565B139
7E3A17F0489C5D1CF249BDFD6D839D17797E5F91655061DF
DB0EDCB6F326FFC5EC6C23A9B23630110BCF3EF1623090C7
0E878D8D2984F9A638DFF1A309EA4B436D24FA40B3BC5F4D
AE09F204A5E6EEA22BADD0042D00D48E60643343C951C79D
5411134E8112EBD21A89CA726D6C3293BAF1522031BFED57
1CC1D5B8489F213F6A0519811867CA70AAB68B93768B2EAB
A16D7299661CC6978AE949CF9F607E286913CA0615E12CEB
9A2A8EE8963BE88DF83D5761D490FFB0450E9C450542CB08
FAC9C5FB38BF7CE0902DC3E447D3D4D6DD369DBE94A8AB67
C430CAA3FD05B674624BB549B3A565C9F1A93D8E4AE429E0
D03C16013ACB204439B0B84C5EC0507C07EAD360114BB6D7
44F6F5EFA89F2302B9E4B6710EFA1BD79F2589575F57B7A0
C392AF0E93D619A0FAAC3FC2EE960DE9464DF3F3B87728D6
BF1C5CF1B59A39C9EFD3D46F9144E7E7CC75F53C83BE87AB
F4DEAFA84B6859F0B32D04DA3CF8DED931870EB8EBA086B6
68B3511E9A930CC1E465DDADD3813E79E5235690E279A53A
D98B1C76BF4216804783DD2A082F5722AE7049717AC6BEE4
8FE146611B75E5E6527F0E2516F09090FABE1AE0F51D9965
B6F4518C9B5A082BA4951EEFEE8991D2605395D04FFD9784
0C4C268E0243354A31C576B053FCECCD320D864892DD4452
4FCBCA86DAAD5EE31427339F45DB698C56F4EF5977C62346
D63400C91F46CEB1F73427A1DD61920E5D98250CD7B21D61
2002BE33AA6783F40109E747E27231C652FC90B304F91218
A8FEDBD8A393C41857EA09B457FDB42078049B3F269C1431
D2C06B7D56774863AB6F3A9F8FFD79FB56FF130906B16596
5D8339715D342E8DDCB7CF18F4878D2221BB05FC4B2563BD
06DEF00E4151BBFA011D87096B16C09A955B1059EBA03393
36386BE0645E7577BFF63BAAC50532912EBB7026B120E472
8E4CAC8FE8C09463A5726B0FD3335A09E8971B498CCFC14E
45EA2AE4AD4B1E0C6958FABC69939E1A19300E28946638E8
7C97246A7834ECF7074DCAD08A8B4BF3E8A68157889B97E4
0F040C8CB0C777EE04086DDEAC53751D037FC41A62DF3D64
011A9D0417A070DA0D425E835F03C68D0BD8762A7174C0A6
76B90E3049F98281369B0ACD1D2193EF2F63A25F75D9FEA1
733DBC18BB0821CD1B33210BD9018F2EAF405E0A281328B4
4C3B42FEA3DA2B23AE3C723B22522AECC84BB1D5CD2F9E51
77396616E87FDEEBDB6276B30422B04C2098F85F0085371D
6BD882666B2089F5A3904F91E26B34263BEF463456486ECA
1AA8BC49C40EEADA030EE802C336D83D75C3EEECDF365AB2
6E9615A0291AEB33D37FFF3BDDAF628178009820457F0737
F78CB6642756E3253A46FC55E5C852B5D209CB886F1953D4
3D900B55C9CD9E821512415B734B06A28B8EFB0488F9EB4E
1D534936BE6E2831E83661E0568B170EF52712899E2F8A8C
68D08DCA280F71F6E33FE81F56BF5D98BB498B720F8294F4
D4D1339C8797918AEB4000CE0B75EB741EEA852B7E5FEF35
F6204099D4C1B7F82BF0A1364A6322C3578A46F354A68622
FC94F4DFB37B4674843E0CD9DAFDE149D89667915DEFAEFA
F64C25ED176B388E022A3AA4622975F1CEC7A9A359D1D28A
CE3230D5A7B57E82F92D8A36485B0EF70B3A8430391907F8
D9AC7B94226A60FC7D99CD796FD5B6EA36AE798047E1B53D
8A9F12D8D9DF81E43E9DC038FBA927A3A0A0D6FA43749D2D
CCAE282196A07B02E896FDDBC71616298BBBC436BED3D424
09503A7A95B31206D16FC3A3A6730CB379F413C255B0916A
0D799B12AF89A92E0E91237F37236BA484B5192B468F92B4
04A3597758A82EE3EAA34F5B0FB1198A1177B3B3B8DD583C
BAC0B8BE5A41BB8186CABAD7A32F17559DB8579D2A32ED35
96B5AD3931A54DF4C0754FFD17E2ED27EEDBA480D7DEC700
4A3E600FE13F444B31E03CFA0863F9E316F83F68F16D509D
B5DA026AE3BFBFAC6D2AB00A16ECFBA40130E1817E26E6FD
635799A28E0E31EF6DA157BE71327548E1EBBE93D39A40CE
00A7C94AC7D6DA73AF3537816D7535F6501C81E9D0B70C30
CEA02CEB43D209ABA6217234278931ECB509813A516C8770
460F8608C1CE8A7ED701C8D46D800E5097DD421321A595BF
595B7A837EA74AE2A9C2A4DD9915C684E4003A78C908EB34
40A34377B74D9C7404E69898DA030F47C7B4F5125379799C
011A1626ACDE5490D40D6D3B212E4296DBCFAA310C0D0A7D
0C38A676E739B11BF2B5F305DFC38C1194B67D5BD9ABDD3B
B2BDB49CBB6A27D90DC43CAEF626531E3794B02A4B557363
925F73F9BF011425B3C855EB93FF2B3702994957C90A6E71
C5438E4F6F20FA52579EA70DBE976E8ED0CD0CA79F0FF667
9C19CE6E144C81EDBEFA42762CCE7687D82D90C80E6E4DD7
F6F514983734179B56A2DB5810D2FEF9FBAAE792DF757EE6
47EA4D78437E145E56490DD5C47788BEC899AB249A782AF1
751FCB36A8B567673E9571B4F359CB7452913DE68458233B
6B86FAA0426879F9C1A6F4A9F26FF19DADCD0BA2C34273F2
8ABD32BED2814C6428812B67CB7228BF02AC80EC80C5771C
026EE4B31888E474E20F1627B30475FFEE90BAFA2E214CE2
6EA00649517124D6598C2CBB109B8F8AC445771A9550263D
9029BD1306510D7BFCEF39B7E3E6F3A62993A810141919A4
802BE35C1D0356AA8D7E7830C1426AA9789CF9B1B75C7145
78D1FE031819E727172B1FB5A0EC87AC33C494C1E37C222E
E87A76683F7B6D8CA2B47C8C41F346764053D8841FBE610E
E3E18819D97CD24E6C82AF5212876375134D9401E1CFB25B
528D81D4D77B3C9360D1CCB3D8D906FAD4BAEEB8C417AF8E
940509DDE91065297F51C95D6D61196BCADDFFBFE43A0490
1F745AD4161EE006A4260D7249E0B68B0453B01819EB2A14
B1AA443FAA0B47E51A307ECAC7FAF07CA8732E715513C738
CB30A43A2DE29002ECE22BD0CA7FA5C40E0F1EE3C9784249
A5B6D8400303E5118489CE082241808027945C143D1200AC
FFB82281B7B2CDFF2E83440CFEC036D03C26BD5133AA56C8
08F21816D77A3D5F7DFFFC5477C4EB78AF660D7C32B1B6F0
286FC4EA0BA3EE25CA28CD635DA1077BA00B7BBE1D319C9C
ABD221DECE666C1E0ADE89ED127001B1AE8B37A3249BC237
EFD037EAD7168C5B48E18DBB0BFE94E7A7337ED9F9DC0B60
17A915BB90D878000A07BBF2F7C29605541CCBB2BA33BA9B
EDBA011909174DACC7276FB748DA3D72037688E09B471CDB
18646AB2DFAC64E3D1B8298F9DA48F8ACD3EC8E9B6C091D2
5BB55C940C48C5F173356ED2F76F135BB33C49C6FF401CE8
444CCFDECBBCAAC55FD538C84FECE4AD05E1F86CA2183046
0DD2986EBE58B62051C5F8558B33B2150E2162273A21D2D6
BBD0BE785D950FEE7272A6A42E20EEF9FCC96DA0862A9D4F
C6EE3736BED22F09A3082957D5D2D9939E926DE56B23E540
F272AA5CC30126CA15ACF94904EA44A28068BD3B3ED74BDB
24713F450026D6D9717475558F33A84CEC5B8EC74367A1F4
021A9ECFC35A18BC91568F76A06EA7B6F068B1B45E725F2C
54027E9B61191C3ADE350615D809EC6A4BF7FE926946F93B
27C362B26734609E760ED30E492273A2EC0786CECE1B62F7
C56A7C83A9F780DBFC564564D60B110894B3462BF202B31B
E46DAE97BF1D1433B91E7C4B30A3D11189BD1959F363C8AA
DFFFD3EE84CE33EDBD6CCBDF9FDBC4C6CD9F6C3663AB3124
CA9D313C0A9D62198C2528B856627BC9C70DD1C6523AA09D
DAB676CBA9ABC9799E4416E703CD4C89348F3513D93BCE58
383C379EE484CB248AF0008F339F2B76C25DB122ABE46C2B
466C5466009BFF5B4D9639CC2DDFFBEC31663189A11A3F06
5C2457FB54624D291D0223FDCA7CE3EAFB85B6D492408A9D
FE593DC31045C782EC6B52216FC64B35FD33EE3904BA9966
D7174534437E6168C2EF60B0EA9DC61ADD8098C0694539EF
75FDFFFAF4F1D44DD610CEDDB215E16593340EDE583C875A
36EAA91B577FB20178B0F06FBF369D07FBCE8445BB36774D
9068E188528160BC3B2C2036B4B6140BFCBC98A031847132
42E86596DB6EA0724CE131CC379F5A3F37646C87026DFB87
733642DD7D4F08A2303B7517974AC6D9E6EC65FBED56CB05
92ABCE48F750C364C5FC649E3E2168E5BE83F6176A2DC552
BB841126EBD38F36A63FDA47FD5971DFF6D9CF0432811535
1A5F8B7F34376121B1BA35FB19DD2DE6A1FFE55D22703940
A4705AA774708F6D6AEABD6B527C973DD3B955E144812BC5
ADA04C7993477041403A144C15BBFEE889A6F835B9F968DF
2F4179071804EC9280CD2EA6E265AD77633F5196CF8EDB32
4378BD6082B465ADB4007CBDC5F764C402ADCBEE4347C08F
F75B2A3267B96CA1EE4191B99CE78B079333F10EF184D91A
C3891913D02202A3DFD2BFE70EDB992275744136452C566F
07B465A5110B6F1D8CC50B4716A5B5CE9985CD0A7A857EAE
AA3E818EDB7190F12AD58D9A4A9DAB53209AEE83F5B3C554
C570CAD73E05B2F8283A26726F07BAA339EE78ED7CA42F62
B48EC775851C9AE303F05AC8E51E62FF169C175792C2627A
1E3047A18CCBA9F21D1E58F90FC2F8100D94227E29DF84A0
5CA88D9C7D388BCF7877FE0F8CDD5EC767DEC73783B60195
F0AFF7784660E13949EEBC8A817DB227D7DC2248EB0F659B
9B5D9AB548CE2CB1B4F121B5BCACB724A02A655E56650014
DA5D70AC87DB9AB9A19D7E79FA3820B9A6855F148A662E9E
0DA0AA0A4187C2AC05B220AB88498FC244FCEF7FCC59692A
DEFC6E4E0FF252A652CAD76C06429D7D902F80D0C132EC2C
FB8EA1191F9272038017D3D10B0F901514490B119C3C086F
A0F433C8EF9F86EC52989CC77B999EA02827EA0D57F00766
73C51E27F033BDCF09E29C19740AC05676C7DA219C83F019
C1A41E1999A6E7ED8D2407005608A44999C0918F009F12BB
C1DE2FB59A6A6CC45D3E1A9A502F0327E4E0582692882E01
FA371922F657875B03268090C91EF8A7D968486EFCBE853D
D7786FFE8F9411E982D591C63A364483A733A005D8DD4F52
EA71A84A2AE43700E2D6CBF9705809F38EB01E3AA3500981
AEE84693BEA5105467DFFF30E694772F31148CCECA362D0B
02C9565A480DFE315479D62BFFFF915768411199596D0067
0EA5DA49521F8C298EB3A65B157D3B0E42E0BDA0EED10EB3
A8F8E5533E0025DDF4421EA099D9C553736B86626649BFAC
76D357BCB224F3B59DD6286A3EA0D4C2D09699E68ED7D4E3
202794850A8B2A8D611473C84F18835F322D32E04214502F
1BED2B5DAA818E27EACA9EA5BC6B5D88FF7605CBC529C423
6FCEF79BBB8B556D597071E6E86815575C36955DB95FF0C9
5C76AF9054B56A78EE4BAE2F69E54EB4A4DCCA9251EC323F
8645904A860B9227CC2EBE31269BCB033D12626D4E7B1E07
5CCAA2D96A4B90DD33CA1A8F63D9A32785FCE91FE9D261FE
644AEF13CA34442715AF3B71F6F40863B9FE5B68D9AEA69D
BB8F2FCECA6DBE2233D94CC9617C0C924251503E15ECBEFD
C0CE5B9858DCAC7C2A7FD41EAD46C1D1F4A55C612EB87095
C3209B535B1893D292F24AA705A54AC6893F5C34DFC0B89B
4A9362D913330BE01F8AA9A40621CF2BAE0AEE998182BFD8
6757B6725238FDF3AA6E92CE05242DC7A138F55A8A2B8CDF
9C1EFCC39E943DB17478F55C2E8F5AABE665A257A74E7CC7
9E48EA08AF2BC00367D950D2A0BE14D0F26B71FBF3997B10
A6BBD6177AF40DDDCCF8CC8B8218833B8A89D6BE122E12AA
F7BAE51827F6C16F31508A28D362DE9BA4984E47C88DC037
65DD7FF5BB8CE8D5F84480BF5A0E35127CD736A1CDE7BC12
E8531E2C7329ED60B7246E6246B1FEE13B1B70F1A119CE54
7B26F57EF6D56F8966AFB3AF6C9BCE1C1948396CE4FFB1F2
D6EFC3AD5E62E9055C414D01E11E0EC9374B5F4486AF92D9
B4000A22B7ECE2B3BC2695A2E98A9F4FC981DE2F44A80CA3
30F506642345E45D1116318D8F9849A640B13C27291F5B83
E02C8DBAAA96F301081E0F6F6FF3163D11B0340C5154E06D
4D176A285408CA3E69FAA2F90B980CD10BED31F3B7C31F15
8BBECF1AAD25FDC805039E0B0AE53C72B2D598400FAD8A0F
68213C2CB2C200AE8E130125F0DA1A81AF46DEA7A32564B0
B65CAE529F4F5640EC8E61D7B18DCABF5C42D09D16638E02
6D9BDF35483F61B8B41DF2BB3DAB479CE960916A91768BA9
C7065FCFBD627402B557D08EE245E4B792079CBA32E7E956
FA7CAC5FC22F2AF0FF1E052D28817A65966B2B0C1C2F8792
204A233E044B2C07EBBD2C529A8D578A07B21BB674EE4D64
5D64C8D0554606505D54BDF395D6608AE15D23144284C05F
C90A9B1D92220CA0F3F4A8E65155FE0E1D152C89213E96D9
3C8673A6ED62C5FFC880D24A000053ED7042329424181DFF
732D68C9AAEF89D27676A87AEA0F975BC31E9B7184AABE49
61D3540EF01107CBEDCA147F615B0DA929474344FA8DAB0D
FBF56E336FF510ADA9A801F84E4792FD8E9B1C3B44B4D666
FD5EB3F4BF2E475CA451C856B2F576FCE1CAC3B700348176
E5D6F6AF3BB3D2D22975AB500DA24B5783E651C1AFADBCC1
22C7EFD25977630807C1D0FD7DC220FF426D589BEBF3F818
38C0B8ED87706F4A7C2DC5EADCD01C72193261CF37C196EF
CAA17EB9893A85EBE650736A0B56FDCC9DDF6916707D1B99
1FC43409C387A59F2702D1120CA69AE1EB8BCC072AEBCC89
9D5CAD387705F97F784279C85B5C09D3A8216C3151BB8BB5
DC1EBB94F3E0E6703DAA5715A7E826554DB9EDE756B2EED2
5E626E0FBB4CAE926AC717EC009CAEE23B5CD3062359E20A
F348BF048DB8D98D0BD0666B520AB255B12CA6047B3166AF
8F929C79DDE5DE2794998170EFCABCE511637A29810ABF47
FE4661EE9DD9FD928D2AF9B5898EAEAB617E88369457DD67
6E0C9BCE677EA50439E02731B301300511643AC1FE00125D
4589434D3BE9D44EF45DF1ED91AA97D812ED55DD869B9067
0C0DC9DAAD3351B7CE17F8D1F6DBDA62F91FCF6C93019C45
07FD597941C0A30A137256D65E5D2430EBA2C6EF24979E00
EEFC74ECC53C334768D59EEB02B49A462A976CAD32FA9F4C
292EE351E25622EF85A07C0C161FC0FBC09229C176998D35
7ED81D7B5AD71E60F04E388D2E29E044EE253D701D213ED1
56A2F7E6F85975BFCE0705A124AE4CBF7C3015690304521D
B07EA50183596A2430F3DEC13F96845D1070A3D32F5EC066
526F6BBF1EBE525ECF7EFBE693A06B746307726F2011529C
89EBE9219A9AC412BEC88BA321030B4AAA257B6410F7719B
ABD5FE2E6DA870D5B894B35E72E41D590A01E40303524335
1E3D537D9580A74F2D7FC71EE241923AB3848D11A1A5644C
F580664AB99A6B50BFA21019261D91B3A1238FB129B0797A
2DB79D2513802F939D4608E4FFE38D1CD5FFE7B667CE3EEC
13B1F534DACAC2836C67B4B410151AE550AFF9C977B6A3D5
86ABFE6988DFE9D6C9A884942946BE74115987EA3C6AF477
9FB58A6C6822D1F7BF3AC52BD210132FD6DD1E823D3CE95C
1F5E0C29EBE8A28B934E7932C0D120F2F01CFA87DC089E0C
BEA84869592DF1FD12127F891CA9DAFEE767226868F878DB
6FC0DF95D191F492399AFC32C4F536F43C14E189DA93396C
33E9415EA0EDB2BAC1DD210B9CC2F5D7F46B5B07C5EFCC3F
CD933BE80058AFBBC9FFD6590CA2F53B3696E7F86CBDE8CD
15F89B44FF00A4CB930233FADB49A824F0AB540636A0E8A5
33C345BFFA20C8F0B92AC6F3F063E477F1CDA525C35E68AB
617EE74C600A5B9D5C591984222DC000642FA97ECAF6948F
E50FF576A78A5FF73038F307740FFB190F405AAF65909C7B
10F16B3B9BE9762F603DC156A2626D0A92C49D6788433292
65C75607C3085EE0DF99AC88FCB71FEEA3E95D1995027B90
A95A236CEAA6FE6A33DBD6B2FF2CC61DCD06B576D34D8BF9
94701D90637F543B83AC736D448D915AB56ACF688AA2A9BF
C005D8C2069CB7F291C7214165383ECC418999FADF197ED3
F02715BA20879FE16CA382F3F33B35DA09273F524A832CE0
925F4C6B45F07D01F1033AE426E554FE30AD5664F89EB521
DA89E876647B2B17B726DABF0620B5791B8D910CB9E7323D
A3817AEC34558CE5C4554B4BB652D52E0FB834D66173B40A
EC4599DB388F66C10AF7C24987AAAB6D02CF431602C321D3
D7BBCCC448440CD6BFF93D8CA1DC309BE47E51C823B85EBB
2CAB4F11BE0BB64B4F16F707DF30F465A279C6972FBCD7D9
12E4D1EF75CBCA5369EB440776F117EDA8B536A35C26F139
17444B27596CBC519FBC8F2B90D68D05BCF51608ACC3D304
D76A8B2E986EB16CD6121CED58295F0CF5EB02DEB60F98CC
164E3829B70E59570D2A5B24A0F1C67A974DFFF1EF5BF2EA
0EE8B5DD7E5A285EBE11965B574517B949C3487937453B2B
E3639D864D20ECB48D8C2B8F892D97A7FECF83937C48C2DC
4434C996B25FE43CA24746396E50C157139C57787675CE17
6D1693269BBA7630B6C083EA6D8AA16F0512F07C214251A9
A1D5C7ABDDF9FA51FB5C9AE0008FA953EA83063C93E49319
9BAC4A7FFF14B34F32F10EB2B6A3A61A13C255DB176D6FCB
C6BD779EE2F9FF6A4ACCDA2E33FD0C62D641733E414D5B36
F0D656506DE265EC28F1781DACB76D1C7575DC55C15A40DC
0E95FD14633037441CD9AFE78391FFE73AAFEFE532B8F7D6
AFA01B52D549FF4C8B6DC71BBA6CFA27E8EA03046B95F6EF
9E63417D3DF651A3353FB6573FFAED9DF4ABD7F68AAAA051
CE51254F812107B91727838B619D5C4F17FC2C3C9A95233A
89F63C422BB85635F432D4A6DC9F940D79E0B5E111A24F4E
999534C4B3B25F32FFF05B893F4E26CF8F1819FCB5B908C1
7FAACB92FEFD12045C77B4180EA95EE7E1775703E7BB4C3E
2BFEAD2B7E1C6BB0F47AEFB9909E3ED961EFB39A29794515
A52AE6D0221773E42FB042B7830D28A73A590562C38C802B
5B3007DFB62B78955620A0F17F58B6FE55B360C0F7A9F4C3
77883951F130BA75FED13621DC61BB766E475DB7751B0D3B
297DEB90367720058318C994C0382B83D182110797A29A53
0BA968080D4608C3BFBE43682CBBD63F246F866AB0A0070C
328F20CDFFD905862A11F70B6FDB4803133CCE8ABB8D2B73
74AC023139CF3557F49E2F8A674AD0857AEEB9EBA3C1985C
BD588F19DC358DC37450A6689E4D7DA413206159F3DD2A4D
DB626F5040AF44803156FF86860A1367B1B644B021C15460
ADE8785F2F65B67ED1740F8693C5A1CB3BFEE9E3B211B83D
D08599E7444393A3C5FF2493E1B6647B297708919163E05B
8AA57D3CCCB899CC92A44ACE801F54A13E60A45BFDFE7D30
3415A0699A0D14DBBBB2280A9C8E3C91B954B769CE606362
EAD03B5ACC5C86FC8672AE14330BF93A9489A5017D3B9354
7ABE3B7FD5804A23B937C0EF5ADD8869188BF842F68D21F5
739004F8FC7577D0BA954187F300E441F4A99F62E0F3C367
30D99CB386570F45B8F0EB75868E8EA9EC7DBB04C26D8FEA
7278F6E780965FB5C02AC4AC24BD2020D2439623EEB901C8
93E7E84B803152CE17491B9B9DC6502041DA4D56C636B4A7
32D63F3C642173BDE55C2A7E8BB6C942420C309DB2E7210A
CECC92727AAEE2FA74B90AA728F0D3F0FC5E8A42DF283713
51437CED5761BC3FCF29FDE5877675B5DCCABC85BC2279B8
F6A494D7241543CBFE2BB3C68F2DA9FD9915BF0A14BF82F4
DD8BE4ED5FFCF836C977352ECD2CF689BD94F9CF0EB66EB4
1EF1B22652B07E0039DC257B30E88A2FA5D3FC64C79C741C
775E4B0447AFD1C960320C042EF2AB6C879088956435872E
A40FB52C188AAC639CF9ECE72B978C812CA7991997CCC3D0
F6F305D3ACA46CEFDA5FCF420B49C9C13F14C3247E4214AB
2A5611447591A609BEDB9B99AF030D1548C25C1090FB7252
6BB8BF7CF510DF644EA1771146EC6E943E26598734BB2022
8508F80824415D97E468B19EA305B0473A915B996DC83A31
8814EE7A02164290EB431E70D50490DC2313BD30EB479857
EFD4A311C5F5093AB23F2C775577EEEE230643A75F7938C6
6FA980DE170E4708F4F59BA46DE0EA340BC61C540B0D12BD
C699821C7E342804278CA77879FFB9332F274530C673385F
FCFA7BA4FE1736CA672914AC98905FCF235CB82F10BE9981
BB1DC4CAA892B46D15C44125DEDF64C63EC4DE5B807F93DF
520F25FEEC8D698D31C164D291A54B906564A75B7B693239
772CB58B4086454862B813AB25EA5E33BE530D3DF48F8BD5
CBDBBF709ADE1701ABDA7A258528D675B505678E6B643FDD
A16CEC411677F88C1F751442BCDBC9F013AA1F4E829CFCA8
9DA04A1EF52DDF86660B9EA7D8AF385BBE6369B10AF2F3B3
049F7036F40FB51260E4B8CB0696EA03D375B401CED3E2C7
EEB8208784C4C65DA497729C948063BD9F51FE5C026CBC8B
F39BA71DFAD235C2E0CC8BBAE4CD8D5D1EEE9247086ECAEE
8372D7953F7697BE48661E3718196082B06EB448EDF3D839
D41785ADE9E2FFCAA5B9D44613CD23CD9C3A519CD9C89BED
620D6396A467CB957740B85D5D51E53279DF78D56D6CE8B6
89905029EF0112631ED9DFDEB63BA26235B822022B686D59
51AD7F7D9611ED7C24A84525832A2F8CFAB80F907C02533E
C645E4CF66DA4B343FC0E9221B4616E8CAE1E65E50195E64
38EB1AAA6CEBBE8A21EBBB9C0976D29F4089EE47ED1D1F4E
EDF66F50CD0F54FB3094E3412E28177A53B1CE56D0FDE4E8
11CD219A1E03532D2D4D61F1563C2AAA8A1586736FEB5CA0
01D50B226A3517A3DCE79495FE1410E7AB6FEEAF8FA24BB7
F981D517F259B363A945D2D19447378649E597A37641B911
B36A8BB7E4018ADB5DC72B4961501F9A5AD5931F5E97A6D6
D78D1E9D4682A0777A6EC0D8C2F165CAB925966650937F89
2F6D9783D5BF5976DCC31C7F00B9FCD04B6C16BE92417C9B
22B9F7C41FCE5A4638D6E6C61265B58D18C2D5FBBE33E9F4
37B7BA6658F835D817CB114D06C197D23C47708352EAC551
D8E97C4A5F14A1F595C041D0ED59C4D4631E7AB673484824
AC782146178A613B3FEA97DBF2018A6C7E1EE3789BBE2A51
11FD098F69CA3BE809474769C7DC6664A37AA26358A1EF08
93856CBBE6404B2EF9D4240556810833C73D6D049FCA47BB
C8466665EA45CE4CFD88F819D5C639777640B84D3D60C4BF
9B4834F18ED69D519D497AD84204BCB9D749F122A86206F1
7571A4B267EB04E445803700DB6A72242E1DE16B1677C410
2A6518AF7D1C26F81852CABD90366371783BE49E94BEA42F
AEBDEE389CEE275743FD1DFE10716DA734049874700D55C1
8EF1808D19882CCB413F4B432918984BFE61D3419999642E
26B8CC222AA5C53969D3DB6DA64743B71063AE978E1F5899
2B8FA32B543ED0E1A8EED1165806602675D21D4A330AACD5
7BBD8EB068F2EFB116AC6EF6FE80BFB0E8023702D17EFA6E
344265DD1FEEE145A37F268EF0877581998CF323CFECECCA
9B606437205BD1DB249D3566D7AEE9D289C9CFD0D57FD058
C28D239C59BD76719345840E7C491748FAE7FCBC5EEA6C51
BD1EFD0CFD6283ACD2A95B1DAE33ACC47A14D21AE3A74DF1
B0EEE8E5A9256A73A98B641F3D29092D2F1F1B75509F5455
C46B3EB0F3B3C6D61CF0E5A263D4499A91D68055B1EB4996
C14215E03D57E2D109D88F48500F0D2332A819F1716D8885
DD38ABA719EEB2B7B78DD4455AB2DF07A5DA8608BC8250EC
1175351A36F8F2769BA8A206185C10B241189FA4D6220C83
208DC327BC1BE3B55EFDEFEF89776DA5EA979CA479CC970C
AD80043313C929A421495430561C49F0613C195D3578B518
8384691A0DCADD919AB610D69A4C478A582525D5C48B7F17
2FF93D872E9E6AE1783C0EF5B1AB9510050F0276563BA02F
D3C7B2D23986E191D95B19810843B63360578024F3272DA9
29F56FA57392FF312289DF4A88C718FA56D2C3357FE6D0D8
175884C528D5584EF0E0AAB61660EB9684242F00F6911A24
F5D5EE0BFBF2186BD6CE3C63A9E19852EFF62D39E8215068
6712197F2D9E424139C93292E9B684A11579BAD5546A9F99
EDAFE550B5634A040E23A008DD3B15C476907B3CD596725B
230EC8B2DDCE48BB1ED8928509CE6659981DB303813C7031
02371A2CA23F8781DCCBA0672C979EBF0EA8F17F4CAAB443
E0EFB4CE4025CD25C760F9E822F954D6B38319E71ABED25E
2030785CC6D18999182F7B6A9BCA9F72F86829F89B306C37
3490EBCD433D9FDA74EAAD97469345846E190540C6D82642
A9C3E5AB60F1666619A899B8F3AC2370A5BF2298B99BADBB
804320365682196340806665DC55299041D3BD85728BA0F2
53A2DF8B0C83C7120BE2A9608F1B5A4528BEFEC9D7970487
2BCD607D529B39E1429F0C54B94ED78F826389086C7739C8
0C466A6ACFA584D239367547536323A32715D0DBB5867B0E
2FC3EA605A33AC26E78D8CA19F98CE7B5827E62961D89212
8761107DC3785DBF710BBBEDE5FEABAB6F175A5ECFA961B3
F123F30881EC794674403328E3CE7C80BC95DAC12D7B360B
341C9DADB601F2781F6ABEB933B73AAD103A81E8D96F6730
38E53597534CCF0E60DA817C62924453E193D61861043244
3DF84CC5EDC3D732948BE841302B9AB708F5AF33DC537A31
B546891818224B14E90827757DE18BABD71449E57F8FCFF1
A7D6E7A48C1C14CD8DC58D7EFCD4FC8D407F15A1D4AA41D5
0B178B35A97C06D9D4D2A5838AA867BFDBDB36D674AF33D0
8C22DFD72E5ECBA3D0E0B54985D01223032939E7B7B33697
A52E67BE70E116BDF6EB61EF225DB94DF06966203DFA2E31
4AB1B7BAD46AF64C8B6203A8889CF56A079F18F688790896
65D1A4BF8DBA94C9752913A4C15FCFB7F18CCF257D48DED6
090CFB92322D482D9D8AAFF716F10FDB491BE0FAF50C784F
94F1F66FB01D719F04D36CA30287F7E534DC31B0D91CE1FE
D8689D811424A59B870F84D11BFA1D202BF57896154405B6
D7F2171F5B7523C54B4D1A81BB6048C99CED1E15087A9C49
94DC8F6AAAD2A07CA81191B613291EB06EEC66A1B628F9CC
534D75114EB28DFE082BBA4285ED30E55CBE4864278FF182
F5971C8523421B68BC3D9FCA5ADC5D2C76CD0C3797232950
D274B834360952A27265C12C2B70623CDCB69C4815B8AD9E
D25074898E465F024B7D5AA542575176B885998694BFF162
100B09B1A2778E07F0DEE2E6DFB28D032FA35D7EBAB2E87A
96FE586BD7D472FFA884C5724561D517C1126957FB190096
45EE36AF320341688173D7C41A7615BCCD0A431F834EFD18
F73847AEB45D714CC5756DBDEE00A9E580EF91101AC8FE64
30A56E8B8CDAD753A854A1BE4B61DB04F303B6CFD0D76522
1BADACF65A1F790B448DF08A100E66CDB8B404DDC7865016
D0DA13AA2AFC006A97C537541B505B96CE798BDDC07B9D6D
FB4170AA2DBAA3844FD8485D4325BE987E2E589B58AB3B80
50A0FE8090A0AA2EA503CD40DBE894004852584FDE11E3E2
F6311445087C56C3CA179EF211FD32762134EDC910D9270B
A292F008A7BDF88E61FF866269A6D6F6AC9A81F706D57C27
700C29CF917F3634215C58B9B3B9C147FC0AEA52C05B2841
2071C5313B0EA39ECC258AC29ED6E6CA8E0511C726A528A9
AC6F08712123FF72E9C04D1C5C343BC2A356DBEE9223B8E8
01F46A7935579B7A7F5CDD00B5E7F1D62C37726F4216DC1D
820320BBB243E30FE35395FBFAF46A5C8C0CA41E4B48A80B
4B3365B4AF9E0C1AB703F297ECB82F9DB6ED44819DBC72F8
DF69D9A819C4E27ED1B7D66678FDEBD8BBD1747F892D2B55
C0F17FAF958767C65EBA2F68EE245205CB1BB73290AACAB6
D62C3EBC661115CDE4F4BD646D745ADB7225304D2712837D
37E9A837E07B12281F4740D8353DD6008A077F6EF51978B9
9826470A5ABEE6EE66BB799C9A9206703174AB77AA6223A4
506C60EAC24C18D955364759BD8CD522157AD658C7B1652E
110923F8A4E154057211F273FDAF0F565AAD0A0962C4E08D
30CE0B97A6578CE2398BF04E63F78B2CAB5DC97A140426CE
3AE75A6039177B44B18B2C02EE148388F93E6EB96A989D3D
0DE8D7C180C5FB4E90ECF4165EECEDE97316170B7B00D681
A1BA31195FF34A411C5D9A72A84C83C7BD7C6C777FC7DA4A
1D7295FE7B297D4B752159B7DD71483D53C7799FF82054E4
F0210F4416BE56B157EDC9A76684D1F1653EFA6B7D06079B
311229D7C5AADF9CAA0DF39E441D4E119D0B3887194B9751
7E5E875547284EE4952C250DA3BB4E3BBE59086FA638CAAE
890371D12AF03062698C2F4F5F2D7272DF652EE99C4233B3
BD3F48F09AB50132242F762A61904E945CD3628A925A8B82
6401C4DDCDD7441581AFF57D13C1198FC2347D886D319FDA
2446A367AABBDEBA9EFD1F8CAA9805EC1A9B45200A90E2C1
37C1C64324D78E1F7549E6169FA66D492184F3A5A524810F
91A620E120AB9C4949C6D3E27A856F079F76DA00D56C3A22
575C19C550408D8454650AF1739812870E371BA8F34B6861
5AB6FF5B263ACFAB2013C3068C03A82979EA6DB287A3ECDD
BB40E88E5DA7D4B12356124D9ED74BA5283C502FC68B5A1B
CFEC678B1CD75F4249D536C9399F11C12CE9938EE8CADCFF
8A26E3715E0471DD3616A9E505F4C73722BB42FB233BF641
ED38E1AA4B73AF6024471252F8102E77ECBF4BC315DB1C31
4D5E37A37AFC8C7DBBC25AF2C6670ED2DAFB56B5376E2550
958603BDA98C29A87AEDB0C1C8F559D161EA4FCBE9EED392
A7F58251627E2C4C82ED1AFADB2FFF3B3483CDC279117E1E
4B003F86DA13C24E10041F934DC02EAA9E86C4200550DE8F
6139AB107BCD192E8419E869F77738D2624BA81FD774828F
15E286064681AB50343CDFD1E178F6DAD9071E3727997DA1
95A26E99BFFBB5D0027B380688D3E5C567454F4F2F283C41
45326511C3070AD878172A7DA38D17ED65D17BFFB0298058
C127E412181EF4734360466A38245409E44B2B208FFF4D67
4C9A55B7BC50B8D7343F1FF351F13A89CA53E8AD2F056559
9B93DF18074112C26FBEB1EA277070EF4D5A1778376DC12C
79CE5EFD59A9A0ADBC3661F278B3C540A0332A9E8FA30E07
886F0AD49176EAF30EA422CDA7D1061AF2D62DA67C672407
54C7674DC49D49E63DD7296241C3F2F0B022BCC0451400F0
9DFC07C61179B9DEEA1D452D3073C2A12CB0342EC8FA25C1
E9C6E911956A0D7C3958F4E0252A4DF98FAE33AFC834EA10
9DB1556485B4C5C67FC532C4C6937AFA3A889FDB09B055A7
5875A1A391DEDD84155F3805A81EF68AD288A9E77DEB4324
5DA3FCD577895036D0AB8D6BC654C59E76C0312C81F3A374
3DB3B9731E66FEF6039A2723C7CD7814F2A5F8ED20807361
C784C70B8F431D8AEA3F5ADF2BA55AB4BEC9C580558CD0BD
B256AA37A7DA84F3599B338E2B853B92ECBA5FDE6901B620
1CDEC1D10BD749090B1F491A4FC8E2F8B150D64F3215CE8F
»;

{
my $msg = 1 +< 511;
my $vnum = 0;
while ($msg) {
    next if $vnum > $testvecs - 1 and not 439 < $vnum < 450;
    is +T1.new.finalize(Buf.new(255 X+& ($msg X+> (504,496...0)))), ("0x" ~ @t1set3[$vnum]).Int, "Tiger1 Set 3, vector #$vnum (:!recourse)";
    is +T1r.new.finalize(Buf.new(255 X+& ($msg X+> (504,496...0)))), ("0x" ~ @t1set3[$vnum]).Int, "Tiger1 Set 3, vector #$vnum $recstr";
    NEXT {
        $vnum++;
        $msg +>= 1;
    }
}
}

#Test vectors -- set 4
#=====================
#
#Set 4, vector#  0:
#                       message=192 zero bits
#                          hash=CDDDCACFEA7B70B485655BA3DC3F60DEE4F6B8F861069E33
#         iterated 100000 times=35C4F594F7E827FFC68BFECEBEDA314EDC6FE917BDF00B66

class T2 does Sum::Tiger2[:!recourse] does Sum::Marshal::Raw { };
my T2 $s2 .= new();
ok $s2.WHAT === T2, 'We create a Sum::Tiger2 class and object';

class T2u does Sum::Tiger2 does Sum::Marshal::Raw { };
# NOTE: Tiger2 is currently our only test of an indirect Perl6 recourse
# so if we get a C or P5 implementation, we must make more tests.
my T2u $s2u .= new();
is $s2u.recourse, "Perl6", "Tiger2 defaults to indirect Perl6 recourse";

given (T2.new()) {
is .size, 192, "Tiger2.size is correct";
is +.finalize(blob8.new()),
   0x4441be75f6018773c206c22745374b924aa8313fef919f41,
   "Tiger2 Set 1 vector #0 (empty buffer)";
is .Buf.values, (0x44, 0x41, 0xbe, 0x75, 0xf6, 0x01, 0x87, 0x73, 0xc2, 0x06, 0xc2, 0x27, 0x45, 0x37, 0x4b, 0x92, 0x4a, 0xa8, 0x31, 0x3f, 0xef, 0x91, 0x9f, 0x41), "Tiger2 Buf method works";
}

my @t2set1 = (
# This already done above
#   "" =>                   0x4441BE75F6018773C206C22745374B924AA8313FEF919F41,
    "a" =>                  0x67E6AE8E9E968999F70A23E72AEAA9251CBC7C78A7916636,
    "abc" =>                0xF68D7BC5AF4B43A06E048D7829560D4A9415658BB0B1F3BF,
    "message digest" =>     0xE29419A1B5FA259DE8005E7DE75078EA81A542EF2552462D,
    "abcdefghijklmnopqrstuvwxyz" =>
                            0xF5B6B6A78C405C8547E91CD8624CB8BE83FC804A474488FD,
    "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq" =>
                            0xA6737F3997E8FBB63D20D2DF88F86376B5FE2D5CE36646A9,
    "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789" =>
                            0xEA9AB6228CEE7B51B77544FCA6066C8CBB5BBAE6319505CD,
    ["1234567890123456789012345678901234567890123456789012345678901234", "5678901234567890"] =>     0xD85278115329EBAA0EEC85ECDC5396FDA8AA3A5820942FFF
# Right now this is too long to be practical to test.
#   , "a" xx 1000000 => 0xE068281F060F551628CC5715B9D0226796914D45F7717CF4
);

for @t2set1.kv -> $vnum, $test {
    my T2 $t2 .= new();
    { $t2.push($_) } for $test.key».encode('ascii');
    is +$t2.finalize, $test.value, "Tiger2 Set 1 vector #{$vnum + 1}";
    my T2u $t2u .= new();
    { $t2u.push($_) } for $test.key».encode('ascii');
    is +$t2u.finalize, $test.value, "Tiger2 Set 1 vector #{$vnum + 1} (indirect)";
}

# Now grab the code in the synopsis from the POD and make sure it runs.
# This is currently complete hackery but might improve when pod support does.
# And also an outputs_ok Test.pm function that redirects $*OUT might be nice.
class sayer {
    has $.accum is rw = "";
    method print (*@s) { $.accum ~= [~] @s }
}
my sayer $p .= new();
{ temp $*OUT = $p; EVAL $Sum::Tiger::Doc::synopsis; }
is $p.accum, $Sum::Tiger::Doc::synopsis.comb(/<.after \x23\s> (<.ws> <.xdigit>+)+/).join("\n") ~ "\n", 'Code in manpage synopsis actually works';

my @t2set2 = «
4441BE75F6018773C206C22745374B924AA8313FEF919F41
13B0BD588F0D687BCEB645DC8F0F3F3A5D311F5C8B13DF52
035907035356BABB43D261DF4D2352665D841F35D316EED4
CD0CAB40050BDDCA6A6C3E17BE1465B3AA6068881D62CD82
639F90E38BFB9D6441590CBE2EC05B432A66FD6E975AC1A1
A16DAC1E698EBBE69A169C71D8BE03296CECF6B605C01760
8C0462768001F1DFABA0E1AA7838F65169D2CEEC899FDDF4
531837FE5BBBD343B98D8FCFD116B4C724702CCD8AF40DCC
860F73A57072EFACBBEA35F7DB67D97AA829C53463805093
329551C66FBB17481781270B72FB19C18226E672F79196DD
468C181D8E3A74EAE7B9DAE67CFB0379EC0C30817FA4D77A
7FAC446C8CDC8FB9FDDA88291000F95A62E8D07C0B59B712
69A0AE5670171843645675A0A65540E6DD23C8DDB8553D7E
7E06E68EC2C583E8085EFCD48599ABFC5DB383B7130C1344
F75DDBC00FC01BB281754890E5E910941308D10E1D346AD5
CA522249D401A7DDEA1F1AAD682C75E4626792D832E34047
46DCB92C4217C36222915C85BCDE0EB0250A9B76C41CC9D2
FAF524CD6FB33B17C24645F3768E7263367582CDBF3942AA
C49420F94FC96460D65A2DC25DD55CEBCC7C5235EBC882F5
33D73A5F30BBE075AF6CC4BC402B425E91D9E93040E30F24
3ED5910FE228DC32C433A8B9444B8514A50DDC57ED2C190D
E4F45BEE0C22BA9F30FD191141966035D89D00DD1B15CD59
BF178FFAAAA81A5A1186FE4C6C71EB8AD2F506F14FD0F086
8F79CA3889F21FCC7C2B8C63F59F03C1F445230ACD66DC7A
834B5E6E248A6DF4F35B363E29CFBE1D8CDCE9AC55728649
25A5A35C6B87F1303969B623B14579A6DE8CC7FA3147117C
6D723A4AC8425A67C82796B64FC11D77EBE9C92670F0A48B
5D6CB161CD2E6664EE5A2789F43F27F962223AB91B13DB4D
3B04CD0F621A5E91CC9F5E6F9BC369B3DAAD5A46F93A4B7E
A203E5C31A0D84CC65570E5A3802203165B7E54A8CA5D9A5
178FFBD837424F29F36BA02A43D610121AD0FEE71A69C259
5D68A8DA1EA16B637DEB9DC3110F55B015A250231B1CAF71
79BC152719662CA815EE101856EAABED6BD2ECAFAEA2E21C
11696C61C977D4BC079ACD4CC73A7314072D63AFEDDB1C5C
B4C5E67D71B72DC9B6FA490A3FC2EFD307D345E1FF8109C2
170DE283D4A836AA6E72582F2AD81074B124DE6D891847F4
FDD18A9F37453B5934383CFF09197182087D4C2A02DE6F6C
2E407B015AF26B47CC57D6924DAE71D0E46C16879E9ED717
9B7A6107C77CC5C46E5D68F0A0E13A5AD1CBF47739F5D015
4889AD62547AFE00D76BC333A2C1597B948A949471D2C5D9
A8AF6FB4B6EBBDB0C498471189422F2E3E82B9893B4E5260
7F4A1427D718D9F7473489912EE85AC3160554D1CAEA8F42
B1BC7FC58561E2DD41C46407D219C460244B3AB59E0B0727
F7F8537F998BA8152B86722E305146BA1C62A5E2E43726C6
F92E3206125DEB249122597A162144EEA4914ADE625BD171
D96BD2DC8113CCCCB1050C13A05B958756DFB7C638FA6988
FC3D11D29D23A00FF6AC3D0849CAFE2483A1EA1BADEAAE7D
AB22ED46EDB38BE0BEF2C6696382FC074BCE81407F88BCB4
B721E30C243329CFBE0E88DAC6D0C1E75D1E21286086F667
FC2868B725BF573FA0DB713D54CE4892ADADF7E6E6249668
516849846BA286D0FBF4E45D21A16F6C62D98B3C3E563DB3
094999E735C27F6DA54BBEAA44929183395EAAFC720C67C7
00193D2E86C24F8A11D4E131526761E7391FC86760A34DFE
6B0B39629ED1A8AB8EF0C90D5F625E2AB5A030CC9BBC7400
55AF7D7AB5C344301B4560AE28586D654FA705C7EDC5017A
CCCD15060163E59343746D4BD0AB999B17775C0CF52F32AE
085935327E118496D0E8EC8F1C1FF408A61358517862B603
AFE9BF01912F8E454232D700129A810E8C5ADDB19B048551
12422D77D895258ED3270304108F33B5D9356C4253D74E9E
19863B04D731D3BA8BF12AB0F56B433270E8E7016E03BA35
19357E72EC181C6B1822882E7888CB1127A456D2F152CDA0
8A4AD6E20EB75FA06E0A8F990C2694F13D0E4E433BE46EDD
A4DA42F43BBAB72D067DEA087FE6AB75505196E56DFC920C
90E63C53FEC66E58255859184321B87E8CFD6E838EA42BCC
F7EDA67C1820A912B17948A82D29A32A6EA1D2DDF9AD0B73
32EA5788CF7394AD16F41F89CDBDB913BA7C6028D1C75B95
D167434510ED7F1731509B49725B054C30991E55EBAF3751
B9E0E8204C90C67B3BC81548FE3924D3145E4719517F5FE9
33A0F583D187D3433B2B3FA848B9A3AF6F13F2427FE96B95
25584E6087BFF43FD42B4E41020D1033AE922D6D2C939A97
C832329D7A59C361F1E9C22F3067DB0F8DE5608651E74663
21559E08C544097A4254B1311CB15110FA2FAA198DFE57DC
CC2FDA945A2AD33670D9A592E4AA66F42DAF6BA7BFA7E1BE
B34F0744E2CCAE93B4C676A372A2BAB7DEC7473CC6AFBC76
2A01F562A6F2622DCEFB5C435BCBCAD7B0157833A575610F
1D55A9F65279411801FD2ACF3B4EAD9109A9AE4440394AE5
558BC871842F8E09173F0CF4566CADF71767FCAA06C0BD50
D0D6C90473C7E5ACFA8B166AF949F96183CB9F572EFC17BC
336032C0B0325772FD5100888E617A280DA9F8B64B3E06B6
D39E94FFDAE9902CFE04892F19D1BE116D6BBA75692DBB50
BC39A3520EB561062FCAFF4088A0E57D8F9B82F6B9289086
95E012F956781F140740A36013AA964DA702894B741DE5A0
00A75E9762ACCE0C8EB7139D9517E0A999CA3EA6F85D488D
116DF9BB4CBA64C9D0C002A3C4D814B5A01DBEA15A639C24
D5F19E590223804E40F55C465A1E0EC85A62A224B72E4ECC
94109A01249ADAB63AD9F21646ED35DA5CA4B9DBA0ACD384
AB904DD9B1DFB524A8133871AA3C4F93005D27A8D83D16BD
74FAB70AE7EFED4A1D4DD648EAED040B7A25246575DB8DA1
8A87E97E7040B44CF412EE1168086055B8A04E54604D0DB3
1EFAAA077B783C30969E9CB2701A6F45C7C607881E3B1C9A
EBE7222127F5B5335431831A221D01E01118C1D1D55BB821
4B742681A965BBBED80E8B27DADF1C9EB8EACAD599ED78AA
63D8838B5692DB10A8EF680486086B2E98857991F9D63E93
3DD14095A5737953A18F88077C85CF1D97C13E657D219CA1
7CDD4366C45750F3BE36B82B9E242F8609E469610E175417
E244457547C27F6902ACBB056CC03477193978D43D5C3708
24EE19B808C497D9A7F49EEEFCCB8E627A52B9845FA6E004
7A1E29EBDF3FD3F165F56613735FA5F0626931D440EAFB6F
C20643C7777E776993CFCBF60D76561068A50A72C3218A13
C0545614330D4DF43ED58172567452C70A10CFDB4F98BC4B
AF8309910C060D0B6AB4A3728DF9C5723D3577DC4C87AE1C
EC079F29CC0D2A52C8EAADA7720495FB806056B901A20A8D
DE8ECA6A979A5D688FAF17B9D66F77A83BC9B8EEF5922B5B
221F9B7F50920249F46E85B44B4565B2D246F355919B0A64
EC33464F76CAFAD58E4BE0049BDE2AA5601382E31C39B116
215EA5D3EBE4CC739978AFD8ED73B41EE116E3CCA23A3AA4
722E4A5FEA1D6288812AB2C25B18459FB139CC19902087B8
F7CA7CD10F422EAECAC3DBFC6282592E86CB1A1D9B891383
A34714A899EC4FD8C50D777937931F7438EBA16482A76AAC
8AB15D4297F117B7E0CD9985BE73E41E3381C54A2D99A15A
89FF8241070B28B820EC9FF2BCC68044D6600D727B6C98C3
7EDE41778576C729DB00E25F9BBA93CF5DDC140D2169698F
1094BB827A477D8E33BC7E114AFB3ED77D4C49A4A35C5375
2A21F04E05F73686F26CD727A82B3661F4DCE9A851FFE6CB
1C64CE93CD2D60CB4BE1F6594D64DB063B84732E7D971619
1ED799B611A0756A1196D398562CCD7A3F77E3319D74C603
392F7A885B8105A1C99260785F497D40804B43241C9C12BF
23E4D28C23903DEFE0CFD769CE55C77011C07C96596766DB
1C9DDF573A9E8B80C7EF0375ADB15CCC58A53CD6C5901B35
717D832C45BD93153667471B0791F0BC1E1F5CF01D26CF1A
345A07388A58F14E95C061285D16B073605E27AFF9E91CBE
D39D9F554C60E3E8967BD931F3FB86D7E15AF8590D372BBA
F62E0B84C50CF1C9C61386271036A7837373124D1950BA56
0DDBAA4115D96C95686E78FF65316CF9825C3099A40BC607
E3E21ADC5D36F5BDC9865DFD86191BE4623EC935A8E0267F
E60BE2CB790C66719684BDEF7CB46A8434D5C6F7830C2EFC
B690A0FAB7C72347015CFAD60BB3672A0A78B66CB1AEE157
3F4A50420E5EBBDF624CF4FD1A20BB9555548BD0E51112E2
2134110B93B62F7507576D8677C46222BCA4445F9DFDC31C
4465456BA3BE43307476F6202600F7851497CCA690051AC9
BD852CA460E6F732CBDB12B982ABA8AABE5DA7F405CBD416
1D68276B09140F14B7356AF85711B588C1B313144B37A4A8
66FD7307EA4B76CC72255B90D055CE5DA162AD6314BF3671
E22F5D835C065D018F0D0B110B79B452AF2722A84D9D6939
68795B211897F0515E50D24C0842F50656826EA1DC7724B8
EAF56C8A6CA280D65A4E0F2BA4D7D3300D1D98B6C725B90D
AE6F4F45C5C939540D60432CD39987A64ADBD228CC2EBAA6
5EF0D7780B4D76A57AF5AD8E84D526B45961EC3FCD7BCC4B
BADB2820DB6DA82FF18E67B4F8E5F62FD76CB230D0BD2649
74E5A2906AEAB7FD04C33ABAAA766E803C966B06A7131928
5E56821EFB0D85A243D1E8A701EEC08379FABEEA1F61F5CF
49F4F3BADB537236DE5EBBFC67442D708D83D2AE24C92704
8C42D490604F8563BCBDA5ADF8BB2A6AF76D5771A217059D
48CBFC95D93CFBA681222CEDCF014DB93846C29E44CA1CD6
7CDF4C3C8F5169AA62FA312A4DB5DAAED2803C20FC36E5F5
AE435484F9CB6D1142531AF6B99517C5664027680161403E
34A41D83BE373FA99DE2CFC275BCAD67031E7916E830A866
C7CC415F973AA6EE24EF842A89CCE9753F272AA90395C124
63BD7626491C1FFAD40EF77C4F401B730D46736E62DD1047
68C01DE37123936C201805901647FFA9B0E9D9F338CB7C46
83C21C14E686F13EF5B89CCB340DDC4514BA342F2E7055DC
ECBFCF67A8D1A6F1107F90C12A3A792436CD3A9DC4F95C81
AC1A1A6913C226B059B3611B28FE890AD5028BCE2BF33C14
0A9056FA407EF7336EA913328D4A4723328C38FA45473A14
0CBE1E636BA120294EF35F5546A6A28E87BE2F5544C9C35E
1A049F7BA80E052F620C19EFD7D7B395158774E9E58209DF
A7CFB4BD0A5FA119C43DD3DB327B692B2F4F316025B66AA9
D109369669C1D55752F749F5EAB8B7D8B6AE6D443A2506A8
06AB58B92E51329D8B383630C72225B3A824298E68A93668
501F6C871B2993DBFE49BC7ACFE2081669DE0779214B95DD
4FEE743C75AA62C7C4E710564A888FBC68734C3EBB51AC78
666824D2F45A1478725580217760DDDD9171F4BFCEA7B03B
21610D09E77E1D33CADD6929CFEF6BABEBF4917DC0041E9C
240ED42DF691B1F437C9FE5F07B6909C1B940BDC25B9F6B4
A90FD389A5D88072A5BEBAB60D91A1BA0421074083CE7648
5E69573FA393FE1F2585B0829CA86D8B581689E227672C34
EFCEDD516A19F6AA8A413D2BD5E8941ED7ED0C9B1BBD17C9
5DE1C83EAE58E1C466A61C52DE60636E752A2AE8886EA060
15189D1D71E190D04FE1ED5FB133458E8DBFCD4914020CC1
8D7B0E795486890391B7F003310D119F6B60503E9C6E9248
495F1100050A35B4C45A06A00CC1D267C21438BD2946AC33
3AA0996C4B8EFA4DB3AA116412BE4E8E7F5F2FFA0797B50F
2FC4118D871AF00FF507EB5A8BC9AC43F26CDC62B937A4FB
B617A8177FFB0E687E16698275FA1728D5D2D1C5E157BB92
47C16CDBFAB49B2DC822B254B24BFDD26475853CC655D1B5
AD9453677F03FFC5C50F8C51DCC175C458DDDDD55FD63B2D
4C4932BB1CA77261DB48687F8BB6C8468C1D01D71C01CFDB
D58F0D9086DFE6837D288F14E0A4387BD2580BC5CB3B17EC
8911D34EAAA763C213B6658D6EB832FD9FD4BDAB1D5C2EF1
546341AC8EDC2C4C4540087565A0CB394EADA1D3FDA277E5
FFD385915ED0686BE4CC57FC68B1925995EEEC3F70F7483D
7633F31FBB37A4E6B435DECFAB377C98A4D08078F9C137B7
99CC147DCCCD70FAB7062D27D009326B8B6EDC82E7F373EA
1197A083298178829B2DFCC6EBCBAC378C5CFCD9495B258A
3F7831E25332D13A02995F8DB2AE089C6F035AF1DA1B95F1
1A6A7B270CA4102AADA61F756A15452F9BBE7416D23A338F
4011B22590A35B6C2102A22C431123CCDB227920E98A5EC1
505731CD24456B4B7367FC410DB9FB2C55853B4477C56A01
773A93745E28599071500DFD2B999BD15F91C6FC36370814
1052D418660C1F56B3FB45F71920A1D7C3F778462187F8B4
4ED83DBB47F130599D40247C81B5158FFD33B4DDFB6B78C7
A0C7F2004A7C567B3F6CDF36FB3E1760C1378217D2E04E8F
1DE8974FA4CBE84A99FD35F7FAF6F12BA07BCD6E4D8E745B
F53611266C6AD1BBDC4E939B731694E0C3BFB6E03090E3D9
4D49C7448C606399D8C58E95398763650D7BF4085040E5D2
4FC98C602BC7982DFA29A89AEE0B6ED21699F45DD6AB1432
657B2C096100761E1A4E7E0D5413DF57A783FDD98B52B382
6DE790C74C92F31D9FC5E44A97E7722B29FC09C20EE4D2D9
3ADB1E120CB2F5B1365282F36CD964D06D9B3C10268554DD
4BF08D131D0C48E0694E87F7F0B7C8B2004EE348E33B1EE0
544991BEEFAAB669FA6AFBF8597394D7AD2AD303DAF85D73
AA213AC10783A937548090D856062840D20942D34F86FD02
3784DA0374E464B3D47D4E6858D9F0471C173A2EB05027F5
BA5897623B4D71DA82565322A7CBBC8FED928CD702EA08BD
79774F6556FD2AC05E4CBC63304CDB589519125A1FF38D52
6381B4B2B68A94B193D09E7A4778671025A350B0F7758B1C
A46AF31978BBC181E80CC8C70089B221136F31B17F85E018
538CC84C5100358BEB7C38D1748E4E5C3A7C322AF70BBF56
4BA213291AA1E44A2E66D24436B7856CBD8FF1C2826F3FE1
26862248CF64CC0A836529ADF7317B0D7162EE8EB351C350
D1461F72DDD7E6C7E09D12676FF2617ECD1FF087F710DD3C
C9806044F10E43299543554DC4A28F8C713E51E57B7247B7
1FD02C6037B57E66A6AE867EF89AA995D9F83540025E7FFF
A412D13A4561F4044706854D6C1E8A74056531E52122C059
B9C4FDC07E1230647238952AC409771A293C6658E3B3E0EA
D6FE100266A17B4A76672ECDDB412E34CC690C380FDB3111
C567973D56223CD9C0FDA8D0BC2962743D59AA041414F8B5
C9A4B883EC1B2EFF79D4C13ABACEB371B5560C3F7FCA96C5
793F7B0ED37BAB94235120DDA3C5F1C2A43B2F8E8EE55D73
A6AB424282D4E67C68AF0357FD809D5144B4A5AEDB5D1A7D
B409B68B4D643ECC70910103B4E1E6443DD98D7B1558A723
443196D6CEB517EB57F002C9BAB9EF7162E56F92F5BDC26D
85D24BC650188F24C20186547086D8276DCD4562AFB206AA
84C10CB20B99F0EEAA8A806E5D8343DB7C5952ECF8A94A3B
4206121FAE2FB5EE369CE8EB37120FEEFACC2C9844B189E0
30AAD1556DB3AE8A6FC6AF906D3A14EBB82B3CFD3FA2A234
BF0E9561588D7597BAFDD7DB92EA4076C02777ED83D90F7E
000BE6E61AB6C2BBDF516B9377017F5586CA833806032DFA
BBA3663C74F0D7F007A1CD00A08AC26FCBB76000F50CD051
F8D29374A0283E4A481D5EF1E0E971DF9B58A323C9EDB088
DCBCB61D455881E1DB8F34387913C7E0B9D8E139F82D2228
D5B6583FD47B7FCDC215D7175250B5A493E6C860462517E4
A2360ABEE1E8406365991104B6CB2E00839FD1C06B61C760
4458C5002278E67105212A1CC62F6588A7D781F4A4FC5357
6E2365698FFFFC9A4EB50BC33313B22C2DE325914275460E
E861B1C8A5F7C1F4F9DFCA3587F2D0E11006439BDDAA1610
55F9CE3E79BEF013E90EB3AF638B3F184EF593DE91959722
3E330F1879414F77635E3E1DC9FB9AA248D05EC75EC4E82B
DBD985AD9C37AFE843FB76A54CACE5D4867904D291254619
54BB5C8A6F717938C3CCDD8C9DA3AD91FC7F608C704ED44B
4FCC37082ED3D69441BE59C426C3599F1871534309C72233
8175B05D2D62459840531F2E037065A2495E6E66B85E0D0B
2B33BB4E43CD2D24CBFD8546A6E39817737A155944724BED
4B7BA51F8271B56FF1C408744ED32A29EA3A41E98C47E8EE
B60EF7685CD451EADDA3E39EB0615C577C7269E03BBB15AC
5E147DA7C94812A2F44F03974995FB2C4F2CC49CB9512D55
82F6131C2F9E98B9DB98E609CD784B820763015A80A53927
F57AD3219ABD060716C611087C85E84710D0BF3C32769CDD
449A2219263F74AEAD15C56640055EA25AEE053E13B767B2
D51CF089DF7C2C1E777FFB3DB0B5DAAF9603BB3FCDB25E26
F93AA1F908BA37730526768D6F32338A78899C3EE87A6436
15D453E439DC8CCDB25A6B067E45CDFE59A773481CD75E16
94165F1AA0410FB865924FA318065CD6F031041291CAC83F
48E07926DA9DF0336DB82C7642FE52F4F813FEAB80FA0C57
10C09822607DBC165EAD444865D2A8094E5C21F51AB80707
306C8F708264CE4C6F2D27EAAFAF6AF2766E68735B4B9976
74CEF5F3B0115CA02529B2CD1A36F857372268AE5056CD23
2C585805DC79CAC09C799F525C13269002E2F2963182E736
D3A299EFF78FE98162C605F5B19AD5963A520F50C6837122
4A3BF565B8143D582432739149B761A684F2F7C1F33CF58F
939337883304AD2B2C19368F30EAACAAE835337985C2DDD4
A04A450DCA5969A95AB6F8E121387839FC4144E6401A9C15
75E52DEF00B2BA76F1E5B19C331B7C6551DEE993C1B58F2C
13F032892F7E72D70B7946CACD576F93F0FDA9AAA050FE81
A711EB9CE907453BF8D5B3CFBF7ECEA152D65C1C5667EE3F
8E7F3ADFAE68BDD0CD6945A97AE2DED8C47F85BB6B021134
97CD2DF043BEB7AD4EDB343FC92709D6BC9F151831F87546
E4D194C8A1A4D8BE41059B5176A447237035B13C8D7C7D9B
575AD1DBD998D27FF111F96373017FA44A9943B4E7046E36
313095D4D7EC3016F8E81AB8DA1FA8A45EDBE3349B2B5A6D
ABA8F73CC5CDF08569F60FBC5CBF9E10B7848B0EDD3F7B37
9CB44500ED9EE92590705E47899943AD5D05F7D103EB159E
AA39290F5C02ED0A4D9F260D0CD59A3D675E2C566A9A1053
E325DAD4801AF5D7BA996D4DE0091B1765443F3EE72DBBD0
FFA7F167C5D01B54BCBE1A4E31A0237D256A7E22AD28991C
0D16F17DEF0303D30B74DBB28696766109FD7264FD94F4F6
903854D0F96C8FF68859E113C0A9F6C16F3EA5BAF5464DDD
C20F7925DAF742B096F486DEA6D7EB5E13E575AEF2A0EA1D
327963A9CAF6FB5AD5949191A94F14A0B4472B3DF9E14C5F
61BDB1D9BBBFBD745CCBDF51DA5633784900EBDB4B709CA2
419CC09161FCED7E02D7B6F48252214F5DB436FB11AFD4A1
5551F22789448A4589E47A6E97461B21570805C3FB511044
9AD93E44D1237E333ECD64EE51330F45BFB3C3F47EF924D8
E6C5C6DEA989FF0C1CBAE51286DFE7E16A1429C93AF3602D
904E4BD380714B2A1817B68F201D0673FABEED63E07CC6DC
F9898C1BF60C050948F4B3D5A0B996B7DE031ED49AB8B547
D562EED38F1A5027CDB181E4B1B795E725156DB09788B5CA
689BE52D1A7B840A5F627C9443F7967B576CCC22EEE31EEB
C8E90705B01620A4DE226F607C0FACBF94AC731E78A311FB
593B63EC440F3354BABAB96D845C80CB819F68148E36A15E
5BC5822776DBD257C6FAE453439DC7A2BD84F1700444B04C
B819F76E41D7697A76F81A2A4F55456656687B43ED44AB39
E3B092AB1D68B91DAEA6FAEFD92A015660291875C41147B4
BE36F446EB8A28F2A8D66443C3654C488B5237E6C2101F8B
76B2FC9C4FB3339667BB3C33C17A120AAA620429AB6052BA
28A495D6D48B871CA78CE5FC335547EA9735B7E093799116
0AA175C019C0900C385A67D03F284A0A5759B230DAFA3AFE
737363B68D085F08839DA8DB4D4F2C8AD513D25F6EF092B6
A77D48BF658D85AF14968B7A44B80234708E5628C7F62998
3C875A4B529B50AF90694B934C19FC338575FF5DBB220412
13E4EC4877020730B69688949C9597A3AF8A6F091D4E1972
CFFFBA0915F9E2B05764D2310CD42A7D1807D7FF0057557A
2DAE5B66727A21A8644E8AE1BD81E93712CDFDCCBFDCFBC7
302A80049B4276804E9360BF942DF27E580D219A613A45B0
6767417B4771682EE98B85F81BDB9B4DBB309EF543D6385A
27AE6CB72491E65F9B41EFF3355BD29D56972484C38D99A1
7817F33927336CF0FFE645044EB6E3DDD84A8957EA743BBF
211F9B654A8AC5EF50C74AE8DA08D2CABCF5A693C71B112B
896E88FD28DFB0EFA2262C1D8F150E5A5FE12ABA0C7E5483
FE622B64DB74C0104A945C0EFC3B9174A70EAF76109887B3
6E821D7AA0EAC270B9958BADA4B1C950D7319AA515088514
67900A2606D880D17707495E7D70010CCF5755CC36F821A6
F400B5793F874CD8F4746FF967EBD721449BF15E2490678F
36C370DC6224318A9ED4EDDE86E81593E79BF9474CE2392C
DD127705882D85377DE203B28FB7B3947F79E8961FF86B5F
9FA6EE7BBDBB3E84239B553FFFE4102042FEDE0ED463EB2B
A83B89D62597BBF95A819279C818E8F1C94EB523E608D651
59BBF43D8496E58C9897CDA1A9960082B4B0D9637B9D3B4F
ECF36C83E4661883F9EA695E329C7AAAA043D4B3215048A7
8356601A595BF5A06446BE891CDC03F351DF9CA632A62B58
66650CD50761126B13AA900050B775F317380006ABEF4817
C1557F85C858BDDB1BB9A8660406897D8E6A48CACB1E617A
C0E63D4BB2C580A3C37D267693FE890673CD04647D107326
83911D217025C7F997867E63581F9DBDEE0508CD13C7615B
5EE8596673D9CBF9E36672BCA4691537AE41E65152F16223
57363FCD2C59C33289427433C5C8A975579BD977629B4AFA
DF90A828ECAEA1816930C4DBF7019A75EC0EFC21FAB9B4FF
DCF6103B1A9DFAB621CF6807270DE451BD58D0ED6BF78CDE
DCB0BDDE24C4D3E4AD852DC722BF32EEA460762D98D4BF56
28C665270D4396A8EB212DD5DB474B6BC398591B5C7BDB36
28F809BCEAE2BD0EAD360A460EAEBD2283EF6422F60302EE
E1DDDD8B7800DEC27D12916A1129AFE8D188A7F6729C6B56
2899BD74B23F3D65BDBA622B1E9FE2868BA7C4A09309EE5E
1E09A7619ED5A076873349EF33F73ADD65E766B43596B396
B1992E661EA24FB731B64E3FD1B6CB616D82C17DD48D1334
670D679F7A3AD6B4EB2D912B395ABFE430F12F9C2AEA65C0
D7F12D7B2E0C1812E3CDC3E8CEAE7F0551DCB0A363EA457E
A426CADDF790A1818622D879A7134ADA9A099756EFDBFDFE
E3EA0CC5C5FE6BE3ECC715177333AD443CCB51ED517022FF
995554D9F6AD2FCE7307A48A3313CB803F803FF2C0DF488E
E8119B88500060B9C6820800707398AF55424402690B17E5
09621424757EB7CD87B5F8FBD46A7314FF3A87A037778AC8
2A3FD9B5D499F746A0EA7A581AB2D707D603F868703A61F1
47254177F0E6AE0D3F6D5BFE62BB7F8821265DDB06FE6D94
23103EEA06D04F75BF15FBE372CEB215A5111CD3C9AE9563
B6BAC80FBCE5774E6C2778E9D83FBCDA1369D891290649C2
BD7AC55A72E3CA1658388F3FB5808E05C6752A57F8D085B1
A4D39C09FB7AFFC053E621D9BDDDAE9993AB7FAE0DDCDCFD
DB9825720A0E10BD4CB2A543083BA8F7A689686F43CCE1CA
3923FD802CD67E2DBA4F72BCD4503EFCD9EFA2B5A6A79D48
DB8E2B073D4DB0D9885BBC2A58B3C2C0D66D54CF08FC8886
B067B4D54895340CEA63F63E282AA27D493B10A6A1498A93
7D87E57C0CC8066F20B426E9FECA4ABAC8B72BF71D973A6F
4D1F6295CFA3178C59D924967611DCBB3E29264BE9173E19
6719BDA196E15A59EE23919C78878132EE08224A30CE9F5F
36E1DFF1DD1945B1D50ACD98046DD83DFD09EA6D42F240A6
7A685317411C52AB89676AE230D1ED86D757F73C8CF1AF5D
7F863B01D8C6BB6FC5739465CA972D38821233156DF5CDA6
5B8E6A93A7F5479F85CC074D9D05F9808B514046B2D6F762
BB9777B1EB1E08D2E7E4836C2574A53590C21879D06379CA
ADFE575245FFB4D1E3E46D4CA1CF7E13D0432297C6902ACD
0200D881F3E76A07B6702D4A790FD38264199107D5AC37E7
7C3AB4D69EBF607F08CE6510147748F86D6F6A454442A784
A03661C27E84875FF87DED9F152B5805F57E24BB57EE21F9
2038AD18FD416AF2CDCB498B916A0FE76291FDEC4B7E15EE
E64FD3B6834FCC8156D76AAC540F22C985AE780AF30F169A
A0F34383EA3D5B05987E1DB82B7479C2AA21AF4DC290D3C6
9B196CA91C78C771E9E06687042D01A79BE60AAD44B7C649
2453BB2A4D8B518D80688CA68DEEB8BFA8287DFEEC42F4A8
B29C4A433E7CFA19F9ED63A377298D46674FFDB92ED7819D
98D79BE15EF449166FEE668D54A79F07CF33A7463E6794A9
B2B36700146902F145A86F1C6B01C8098580F17573AEEB9F
468FEF8A17308692039802950378D3F4F695F2657E0C76E3
8DF2BD909B2F7264DA1F8ADD428E1048F805CF3232201F5F
D77FF43A1003BC31C4A37BB0FFD738641F05DF0130169479
06E286B033041D10D0EB80F5FE5891565563922AD0EA2744
943AE9857EFAEFDCC7C79CC71151C060FF0380C9D5F7B383
037CE6CCFF2CE4ECACDBE21CA38F724F00D4E812FC111270
E6F0C092722744F60669018563955E82BBB1D040519C8214
56B91DAF0650260EFDA7BA80504831A9B0D770FFD8DD0BE8
D102A750F7FCDEAAA97CF4D489DEE2E2904DFDA9F668C64F
76F5A3447F84ED06A11AC34762D1386329414821CAB51989
DA3D2FC8133D89CDC21655C86C3B11A5849225EB323025D3
90A55A0580E0628B33D7DC7FEE2546B0B8E37DD18505162D
6EC5F5F6FF63146B3A916258F1F6F496F2102D1350E9ACB1
86CAEF1514A54643E20EFF3A02C05DADE441D821A0EB5C55
1B20FDDBDBE458EF513F10CD1A60583E82BC705758474766
6882660A9FF628A70C2E087E27F623362A1CCECE22493E19
BF235CBD62EE40A1CB76A7FA485F0F84189C3F735ACE5D04
B30EEA56A1A13D504405FBFA424F452420F59DB6309CA399
641D9C62AAA10B1A6E089C4DECEA60D21970F25B69740473
0F0A7B1C4F2D6DEA12D47D3620D8350AFB9F7CF935A3FEBF
3685C2A29EADC567F970E2E653D217234B6A684C737253EA
C5EB2D5A8011F8A17DE40A1606B85BE699D53403991FAF91
0765BC0EFD21225E6AB9A18B74780C01D806B52714BBEC11
2079F42B2BCADCC57E8B7602F6DF44443ABE2D654545C769
F825921EE75260073DA9CE1D44C75E5F007E25C392900693
CBC6D534F003B11C4496451093DCEF01AED2CC52B566FAF2
634062F137181F5489F42C6679B3A8098FF72AC4FF9C4B5D
B55545EA3C753C5C1227E9B1A2FB9A88163D06AC333B2CC8
2B302CF9A13CF45BD409D9540873B6EF51561374E045BA8B
D0FC4BAC6FE1ABFFA7C58D2E1749C2E5DD8CA45E837B4F63
A1FBD3675DF058B7A10D0D148378C26E1BC8159CAF71DD49
779CE2B99B35F7EFA013189FCABD68AA0C65F0DA1C932EA6
E258D907E2962E61496132DEDF431A2C32D7CC3D51F14A6E
D85FFADA074D97BAF56D78DE48446224E35455051FD15891
2A98763F3AA449ED2E9BB09CF0D75732D6A617C50A4C0E46
FED5F0A6D8D79D3192CF976E44D91BBF650F6D72AB464E3A
81E798F5F0A167348EFC5DB29D2BB79989B2F3550C3F3C91
7ECEBBF2DF1FA77814018D2597272111A21E99CF34D1AF80
DD22EFE7FEE6D92FD2060C367220FF3BA630EB22DB5DE822
756DBA3AB397A06AA948064636BFCFDD7E722A066F2F2120
E37769257811BB9977F8918C2E14AA8B2F932A371BD1E20E
0A38815A7E611E4A0919C591A019436AB87A1EC2131CF259
4098043868BD31D1023981C105419593669676219050412B
EFC844B0E81832926F6D9F1C301850D5AD927DBD1E268C97
4D5B01D965FC43711CCC14B3E0D04FA621F664BC04112A69
FB7FEAD2B57CF6AE6E1B0792C32760FBF2BCF7182EA7745D
8A680FC5F7837123C720DEC5DC79A97450F937B3F5F666F2
E67C25F05AB4A64AD3569ECC17E922E1A8D51DE84FA9FB00
32DC0F3687FC59926AB778EB30741628C072C9DEBC36DEA5
D8DE570C7318930E3BEC6DD3DDFA80F56813F7103673B4B7
482604D0186323B0700E1CB475A32B6C71F40D31450D2EFC
3D6C63F212FFBB479F42C006AF63133B77D90F94A2522A93
027D91FB4A4CC6FCB40CBBDEF286A95574153ED34B052D6C
9FF943A8D939755BF72B1600B77D3869764D73871318EDCD
62813900AE9CBFC07E0BAB83C5397D270316A91247A75564
C63E0AC76FFDBF20C7C5DD2671157C09398985637B34E334
0662B2D85355331B226776420E94442E0C713B6CFD243445
EE260303B9AD3654A870357A69BF3F223E9CE182332E5BBC
F8DE547F08B38BFDBD662F618ABE3444FECC960DEBD7B968
AF42B89F5D5464F9139935B459A09E9A85D8210FCB67F0A3
44501323BE786D3C812DF2F2C3DF803554E0EB6213E55F6E
B86424D6C4AD8A92238DFB938F6178C2C55C7455611DB16B
D65DE3BD08AC458C7A39059EF1E80F95258E23D8CC4DC8B4
65260BAE6CA139C71E43AF6A2181775F88FD1C07E7F45DF7
5106929B64499232B94A572D3D0673DDCF90D84E09E00AD1
D7E913AD9432790ED3112AF77901396367EA2401A7D40B3E
2042FF47129A18BFEC710C3F4D2CFFB6F752863DD07DAC1B
F23221631955084780EDC100284166420D2882D83C327FBC
F67D481B1789E12356FB1A503BDC210E1E9AE3A6C07DE567
497F2ACB06365B3CFFF59068A307F54E8088A72C2CAEDAEF
552C3A00CFCDA651C45BF9930D25FE2C1F4455379A88A221
131174CBBDF97A130CDEDDFE33084BFD71240AAC9DAD2841
B10063B3480DE9DDDC23D828CA571E0032D62F641E568C63
5E8BA3453AEF5D1C1ABA2D0CA544887821E51A11BD987539
B15D5ED5169AB04AB50BE3A055B75489ACF7E4EB98C962A6
C0B7DD732E540786A3443E7C8C7960715B3DD4E2608EB0FE
CC839979E0F6A04D6EF6F9DB3FC74FACD5DE9C654F3FA2D6
60DBCE3248168F1F2B7ECB9D8AA8C0EEA67BE2DAA1E2212F
E60F50A97871BFEB363EF94F7A4AE23FA9BCD236A19117AB
8CFE636997249A3B0A8CFFAF3DEC9FFFF858C6A2D4377809
0ADD4805D00968919FB43E4187BE18E21F01F23E8BB585DC
AB51C82B1A2F20FC8BCEF38AC14DAD02B20FD5FC0AF6A4BA
278F4596ADB38CF3F016703E0E4CB65EBE36B6B0F27A3B6B
AC97B54C2D296A95D062ADD99FBBC17EE32DECE99ADFF5AC
B94A1D2BBC6F381709186812522AA17A76847FBD277DC3DC
B07283CB767EF86B4AE99E90675D6F2EF5DB3FC0C5CE2FB0
3282D98D623C3DC1FCC8222E6EB0C7CC7D5683DCEC57AF76
8E8183A5DB177F01EB117CDCA46BC754FA85211AA924C19B
70244FCE95124E243A888A4AFAAF3CCBE937C360380DCE31
3E235C25ECFF7ABEB7BC23D1D576AFBBA9FE6E3D0C68A6C6
2725F994099BB547E121DCA88D2EF36500C29EA9B7E80E06
303E03ACE384C5C060F6FC650DB9F7007EBE43339D7A5BA1
74AB99C566748BEB26CD30C4E4339A63BD6906FABCB8E47F
A6BD5F51B96C2F7F8DC1D8A6B88B75D1B21724AC0E69A6B8
AE47F5EF0ADD9DC6B378E0D5E1AFFD70FAD9EF696E1AD4F8
CCBB772FBB83E68F5796D84DD1067FEB0465F2236C3D7CC9
8BFF7EFC5F0E310CD7E2A2C412D7B149E5B194EA9D206761
F0474A3196C04E36145C95E1C5B57F947B6B9D75EB508927
60D101AB937E022EB07BD0FCE23F665896767359FF6521DD
26A415663A83CB286AE231DD53A32C7903238D8C5F38C7DD
01CA39010817307F22A30C4EDD0B9A3A96406CDC121D12FC
D64F708B073673862B85FADE8ED590CD479AB4B2A0FA8422
B53DFFC76EC59B282592F647D479EEEAAFDC7F6075821E7A
02EF5378FBC6F1A4F3AE08E9E6845977A8F95C31580977DF
ED0D9E59E36CC7FF9FC7D15E28EF07C1EA7A35F80AAE3CEB
C9FA39EA3BA9AD5F4DDDEA99DF1C3F09744AE5AF07674B41
620C6D4907782B3EA2232D6AEA70602F303D2EEC0A8447E7
B82FCBEE19B90942C84B0F69981B729B4E50DD9779384F77
5E1429D2E56A48C8D371603ED62F0D09FEB673A2D1F98F07
5D3E852103DDDC74A1A87BE69661985E21679D5BE7274568
D0A9C21B418B23244C0D242D29344D4609CD208E1B60E58C
AAFDACA95029356B5BDA0687E6073DB27020FA41A942EE8B
2492A251A0F18E548C6CE6F1C7BA5E75D7C3936F08B96AD2
6561E05E0A8904E18931B982DACA2115BC65CACB11550D36
F3AA0659BAE2240A82003E881BF801F6E4662C3FDB876110
62B954FA53C97210DCBC8772EA3B895FB1BE91EFBEAE9D25
37D46FDD719A7C7F1E2AFC182A0C21DEB9CD233F54D2EA30
494C4786470627AAF6699F99A5289C5C30420A08D5DAE20A
5A6A579179CAA0DFC46A3437AF5FFF194558368740CA8420
D930C66E75E4E49BD6292CC3645AA5171FB1968F9EEA3E81
CE11947E7FE651EB029B9195D0FB7E2AE77B650024350C46
B5D64F63C33E4A093EB3BE09701785E76E6FDD8B1466FE2D
49B0E6476321A588C133630512B9804F162C2BADA745EF0E
B30E0DA0FB74C17D8DE6416E2C0BBA28B9D8F49F9DD9D7A5
8D94836C9D6839540BDB486FAD9DB7B8B514812473E5967B
ACA570479CE86546CAD1FE33FF4A4DFE0DBAE9FB66C3CE20
DC44E11CED47B0B6921597300CF42D972E841CA17A30F748
177621B2EFFE4AEC75175E90FD9D7BD70B0CA773263F3C4B
ED7A85F3BC00383EB476263F673C94ABE21A07B37F4B42FC
6B536C68B65BA624FDAB64DF5D171CE2711CA542C04D76D0
3D3B75E30BA9E779882E524D540A8204A30FB2488E7AF168
5F627BA00C4C68493239E64C1BFEAC0B82CC840335B52C5C
25B9983C329C8278136D6AD92D88A1EA53B11842FFEB93C6
DB069A84D5C03BDAA3F73DC1D27C82D174EE044CADC3F4C8
4F404949A8030B9900A98020CABBF73DD45E6A0E8BC89C8E
60EB7B075FDDEF5741949C741736ABC5BF2850162F4A9608
5434341D1DED105694D7F54A0D89EF68085448D7CAC9FA59
E435DE128C174CB5109B9D77C75ECBE475A197CD72B93061
A01ABB66FA232A923372A5D2DB5D12F627F2425F1DBB3337
60BF5D8BEB28B66C206207AC86380A1BE16EBC51C9417720
9AE268E25C6F98E0D8AA9859AA81681E2AA1442708670F73
404B84EEB90C9A05C1F6C23F142C84F1CB392B24E196E32C
F258D27D1FF3B87832B37AD0B1BB91831C401489D1481F65
428F2E8F7B450032FA5F229755289505013E10D68EC60839
4D4E35A1C911B0E05BD29B5321EDDC23FC22D9F52BFD0142
3B4CC117660B10BA9B073DC25DB66683102F99E305EF8F50
562492E255F9DED16D80D6E23612B8EED7008171EEEA036E
FEF2AF534DFE6ABA8E9BD15AA183824176437AF06387B0B6
AB22B5A209953935B7F351B7DBBA9CE30A48F2C6337D7BAE
CC2623AD34C108C318B0AE7AB1BD1947A013E54139A4C69B
6CE5232B066CC4595B94A1B2413149AC44EF34BF948769A7
58A46DFA7BE2B3E77BB49974074F5EE464CA9251F04B8A99
2783D137C1B42509C9C72710F175298815C8BB92C1ECC17D
F512126B776D02C3088462C5CBE0C80CDB435609D077D744
D1782561B0FFC7692AA3457BF09C29C66806E25CE946A41D
0C64D73EB58420E216B35BBD6192C0F988690DF53F05596B
18F83F06A5390B5A93B55DC46970C08034A2D5CCE36BB2D6
28325759501E155FFFCDC30CAA9AC84C29790D7F7562A9A9
9144A43DEF617309C24F34721D81C9A983810BC649378B75
021DB9AA9884C184EF0A75290D2C38317680780ACE47F307
18F029C15081B9447831E91E982B83A29B900625FBD7D2A7
66A227C1DE36EBDE58F06416F930172758B6703748EFD4D2
8629726056FD13C9C97F676E397981ED602DA4F532E67EFF
4B568DBFBB826AFFEA463D713EEECF887AFE9068550FDAE9
3E8389E30CF6E76CE8B558B6C028C85550632CFEBCE8CE6F
A93A2646085B5E167868CF8CFD11D9441181C94374A09BE5
E6D14DBCC0352EC064826FDDF9B1985C3E9C2E1796A602C4
CD9979C29381C4B2A6B6A8E4BAC1818FCFA4E571B521F47E
57CE8530289EE23591198DDA716FCF2477506133EE014EB9
4D1B5B2710B6B2ADB817AD5159223F6E56AC86BECC50247B
3CBA8D8A9812542535CBA146FC2316B9FCA5EA9D6265EC8C
30D78262C201E10186CDEFE2100EC15DA10EFE29F114EA68
75B8306297A5B827D694DAA19B4107FC1576C68B3EA4077C
D7F7BB6FCEDD0D19C2AD996C34AD8CBA19010E54CEA4C4E6
DED93B50A2CEB50CFE5DC1F66FAB1DD5CEF3F4A36483CC2E
D651BAB98C347EDC8E7E50D6E798E0EF4D1D51CD4618F949
ACE5F7CA3562AE620DE68D57DB0E6B7E9D66F987C2B756D4
5E25688ACCE60F14EE598A30CA2F292BEC832066E3568C50
386512267EEE4F46A8474D0D17C41849DC0BB479A3F93D83
53D8F5E9B59206E700FDFFE2644D217CF6AF182D65E6CFDE
3DF6F716C474577BA71FB85F448DA57A6B3D20778921D683
877065865128BBB66774D4ACA70459B2371CB7AD850B0963
107D0FCC076EF12003BA6CF99148DC6B86FB245027527D1E
F986AA0EBDAD3EE932D9D2E96809B835BB2B671D01D2B58C
82A1C505543ED5C137FE805C0C2A30F03C572A09692DE88B
30098BFD1BA49FA2B94CC97ADA33DC10B75A9BFC29D3E5E7
14FE638FC183EA6F3AD033A0B3F02EADAA3659C687FDD236
879F46839D937DE3D323E7A28EF33D1B8C1B4FAC3FE559AF
33A8C540EDCAA0FE04E9B2EE87D81214CBD131A199FE269E
9D403D27521E9680A968C30380786722174ED2F5CBFC2C8C
8760F48B2F494590D5BE57D1414E877A16B15BD769439B0D
D0D6071C6A31E314D45E9A0E12317D27BE9D6C9E0B673463
4071AB344C6B2F934189B6D1BCD75FA2F2AED7A90C395280
4A0EE0786B11CCBA9C06D77312CA7E2274621BB7E425E1DB
7EBE0885C9006FB258B02EAD7F71660E959C5A375806C0CA
230E6A653065D424B9328625FBCA53DCA538185B3E6FD174
8D122827AC2EE1B94131B5C8197C4A39C5CB1F290C3A89D5
DEEB6EB8F544F5A5FD6752D2D8048944095858F53367CF3B
9B8E3F571CBED90DD44396AAFFE18A436E7447C6AB0787F6
E533F8E772F067F67BAE977E2746469D1DAD9F9DE4773AD2
986F0154A303A3795D8D40BDBFD35F0714F638D58DFE9ECA
5DD262883C2F4C7D4B40F60016DC8FC818CD1AA7D9570BBB
836A948A23EBEA5751C391E364E7065EAD3F25B0F904920F
40E54D5019629B31A7C0B8A362A1BB55DFBB8858AE824FE1
BE2504779D4D26766E54C731134464A108034C76FCC0853D
F291A874B0DD74D23FF47C0181C718E8E8D9F3AE39A8104A
A7D48803005345CFC410201E78D5483560CFA8AB056DCCAA
BD6965921458BCB92C251F951C1D1DFD19F07CBF8E8D3F1A
B71BEEE5BBEEC0E6E75BA0BA10BC64A1F217399EF3C1080D
EF9CDAE393751711776FADA1EC78E7D55D6B9CD7B958AB46
5E40381E3D813B19E4262FC56D4BD0689B1212C18C127781
A86E435FA254D8091ABCE1029F43593CDDDB08573FA8F345
FC1CD576DD600CE96085513DF3D7BEBE6CFA7F9DA344F920
79D71DB6E16FD313EEE79C78573ADA33A36ABE5E2E47F157
EE0C47D5B9B94FAE3257752C5A60D28E8B53AE1C4AF3427A
304E48D7BA2F5CD7121F0EADA23F093D093D00F7EC0B7130
856957885F0F6C13FB16B8DA52E3BD7760E733F5F1A9DD0B
48F60BC2191C08E72B914CE9E6402AAB8DF614C2FAB09F14
13DA2EB3FFD6399E86CFD37A489AB04C25C7E59E0A811BFB
C2210100963F9935E67B8C015F29EE13C0696DB185CC5292
693034E12AFE8D500607C27840C4A3344506755B3D104622
9565466084A9AC8E67CB3E4A406DE630207472F0AEB037AA
92CFB0434AC61AA64667F1149F9D6074C844EA6367C63556
3090255FBC3ACEABCF0D5E16FFDDC7BCDD8550D2F302955D
15E33E11E708F7397C0A2839FDC62D76C3540BA01FB126CB
681F06794BACD17C21A5195B53069178350F974B6A65711E
386148B9EFCB4A0FD6008ED3B3ACE8E2C7ACC19F9094EFE0
94EB4DBDC0F0ED5F43F1DF9765597612B48FC3DBDFEF9ABB
1853B3950B48D7B5825CA1B7308465FF35979E7C3C9D7E54
416CF8B6F6D2C31223D60AF9538D87D3D50DC2D7293AE4EC
D00E1DBE4865ECEE196F2D048D4B4A5857A46561701B349E
AAA7DAEC409BAF6E2A204BD71855F50D3E96028C5036709F
620914535236C3DBB7D6B6258F010202DEB069FC92DD16DF
D5C576962EFC4B411D8DA728BB60E4DC1D697ED9745F67C8
359DB6C560D608BA981E1ECEF18E754BA899E7471A01D6EA
E21DCA8835C0E334D4EFBEAC172A2B9B3D411BBBC291491A
28CE25981FC0ADFA4220279A44A4CEE9C46D4CDD24D78D3F
6D4D81F9673FA812E93C6EDF62F968A39AFBCF0F03CC961B
059D6A1C5A29842DF3172071B50BF697B0FC79A5DD116867
DF0361080471B1E441512F07C83EC650C994D27EBA678AFC
A7F20A1C5C86EFCC34659AEEF651426DF27CF9250A9759B9
F5F0F2758C4614F2ACC068764CD86BC871CCD2166D84258E
7DD6A58510E7E25180FF1456FCD8EB1B461BD77FCA5B4A47
CB13E2C5C9E9AE27D31E2985BB019957F8C5B15F14C52607
0F8CA3DC424F4C17437AF5331CD46B3014DEE2519683E7E4
FB6AA37F96B9ABD0ED7705A66316B6563B83F5A5ABCC8A05
7E2EFC7218433054C4EB68DF1D56396EDB473DBE040343FD
E86C1FDD039F62B7A2748204C2AF156E1B9E31B7EFBF17EC
B9714B996D04D50995B8F307F7BE1D5428FD36B382CBAA02
68751F5C5411D1E71D658577D41ADF0D2AF3314AF7C28D26
A2D77A9818D7C815077EB97B8F6C17B9013D7CC637457ECD
A110E266B0CCE2E4311D5A06F5AC16CACFEB58B4520F1C89
92FA6750D57C08340C98743B2E0BAE08FA3FFEDF93686301
EA161797246295EDB08739EAAE0EACF225DCD83D6319A4A4
4684782AC77DFC848DE4A56D00926A82205BF4E71E0293B0
C5C9DEEA436D854822F119D47E6EC9940F272EBA559E833C
967A5189C34ADAEF953E416DB8B6A520416DFD89709ACE9B
7D797327DB3783DDA5B89FB63FBC42F3E4D0083EF4B30C32
67601AB66C165D3E17985A8528B4B31164913ECCD1BA51CD
12B13BFCB4A0B1A69A7A489DCD0E6D690F6FDE63E6FEBCC0
C00BE46AE2F0263CF34E803BF5FDC4AA3ED231FB94118354
BC04AB3D7E946FA4DB9744A2BF1F911538704DD28A439A26
FA9B98DE73E840CD62DB331FD5FCFC754F3BFC8606EF8CD0
0A6CE115AD6D8D8CFDCCFD37628D1AC51CE9212691D634FE
8669ED357BDA914D23E7662DCBA114EE3B03699B4193DFB6
D20F515078D09F58F91A752CF64E30B04B6A234FCE29F311
C44B9BE28D0C75AE5BA38135CA3094C4CE220AE8463DABB3
CE7FFF047DB468FF729D7CF1422E97E0588A9C9AFF171A31
247B4E3E34094900D5C000062603E09898674420AD9974B7
0D5284031B0362B782242EBEA60A38376CE710579B433F2A
465B5AD43D4B5680290C93B38B04182514EFD98234387EAE
83C2B11D8AE8E4A0519DF63A8045F43A4386CF70DA1D268E
15168E8C7648A0198853D620E554B19AEAF7491115228D94
1E61AA55F81068A750C66487CF8E91A97848B7F97501D0BB
66DD222903F8A5C4867A7D3759E88AAEC5440C9DDC008374
08B6FFC7531D93B1F4E3DFC17DCF8135645442D08C320EC7
F79D1B25B5F1B304D58C6615100627689B15FEFF23CA8A53
0C600E4EFFC1C2685D458E705B7ECB5E8A2C544F780EBA1C
B967C183F743E274C7FD8F0A42EEFAB591EE4E766869BC25
07610F801D76C304BC9A359CE3ACC64B6D609DDFAA72F9DC
6DBA68F7474366D86C60832190FD575A066D3E87F88D8F46
0CF8A848511F605615B3BD8B65679AFA725E8BE6647394BE
96B17A484223E12256DA4595949C0F72FEA034789B7DCDFE
AB54C1FEB88D6A35D0CFFE5EAF271021AD60939B84C3175F
B1DA920E24C0D7D85ED9314E088B04B0858CAA32434F975D
00C7897A14B25357466A0284FEE987E86077BB8840E5F267
28B7E089A25C5998EF4BF8FA7152DE2DE1943DFAFF1FC97E
46C5894950CF5E48B29F90F183DD4F704945F2DDBC574327
986257EA8FDB34A426E23D670E487895E08BCD03621ACA73
0B9A7FD03CD15A1611D73A972F3091C09A1BBDCA4E337D2A
692B7999E3D40312BAE817E4415DD2BF65811CBB885F4C4B
2ED3E62914EF358D6FE26DB587435747263FC1FBC18B7772
7A125E45A6FF84EDB75A83C2E4083BCCC97CDAE60B381EB8
F4BEB704C896EEF31AEC3E3956352C0BE6B369062D0E49F7
7E6408F9BE7328996F6541E1EF15E50E960DB72F64D53F24
C8507293CCCE352758756FE3D489771CAE4E6A708B169539
EE5D92F507954DA0D98FA1C0D5AED00C481CE94AB204FEC0
837D7C24584D67851CBE6B1C8E63486AC21CDF75F4627E17
950DCFF833E9D48EC764F315DAF3B0D49AF29AC85621537A
64B98C619FA3E23E12FBA1F9CBB91F1EB48B00BB67D277FD
1104B961E3B8419AAEE4DEEF255F444E01DBE4237F12C1C1
3650D0ADBBD4551B5DFF38CE0860A6771D42AF04908B50F1
BFD5BEDCC1119CFC8395756737213BC3A4CFE0603D355E69
52B8D00760C49BCC95ABCB9201A1FA23CD0B6E2B1EEA729B
A3B57D0D4E3476598280A7F9EFC2791DC1D8F44E6D0FE459
D24559970E699FF64C6235230D1319FA07A33EFD3A4609CB
FF2926C8B25E92F80561E77F4493EE7E0DBC61776961431F
D83F669E197075D96A6F114FF1F08B5DB0BDE3FC93469FC2
F552D82C775606941F3A7D5BCB6074DF1B286A2A5FDBCE81
D86D7ECEABB77FCF865D9763538FDB726FCF90C3554EBC16
099ED273C2355745B6A9E1DD9E6E0311C0CEC005F2F93DAA
1C41052C3ADBEB1B4B63D957E27F4863FA0A5C359A5CAE35
A18F9D388B011644899F775DF0AE296141230A9655BCC65D
29B1C797BE2CEAB1B76ABD80030CCB2C9B4251DB2A7489EE
84DC683966EC230A0B216FBD59111FEB0AB4DF2F4F7DBFA5
61EA8F46E7975D9A3CABC44C9C3BBA5627BC7064C0AB6617
1F7442E04BB38926D578BD7AD90AAB2D63E16AB07793BD69
3CECDF9010E2385AA5202DAE66B43A825F96609594947103
D8DD2E5EAA965D74710C5447F249EB40A3A1E3BCE2B426B9
AD4621DD802ED04FACDD42C124E9099F53928B762CEF48EF
3AD6504A54F9FB7EAF8F0AA322BE12B754F46B754967DF4A
2271C68E1AADAF54CCB4057AC3A5EFA8AFB955DF1DB4FD5D
3D9A8E1A6863D1B6275D0EFF9E446D7A1126EACDA91E7A6A
E03380FBAC8072CD850E5AAA280E5AF7F03D41C489FB205F
461636663FEC9E11651350BC4A526F1E4008636DBACA2961
EAC455CAF0C29FD71C88A06AB5607F0EA6F358BFFE5EDA18
3E8C2D8A48C257D21A6AC8EA3011C0C76D0F42F8B08F7451
4588324242B06BC5244AE0DD743C67F2D36AC0718A4D45AB
998635B6EE45D221924462CBE6F5753BE3F69C292176CA66
63979FAFE5BF6BFAD65DBCAB94166DDB83E99E21B20FEDEB
674A6D722B7E20CFF12CB26CD361EE983A91EC124B6327B0
BE3176D9B27D56E563159DABFE5295B8BD2940E296C0E0A9
2217A8C2EF6E32CD1AF2420139F2021FB9112CD0C1AF2126
C26D8DB56973AE1BB9A17EE81B2F5E677AA30653FE676A62
E9C4C5744AA8EBE00DB53432C482C2BDA98CCF44AF6D46B4
49DA138435BCD931E6704FA4D1385573F1A10F661A050BF4
697A2D3E442BE4B88F6406A761DD4F7BD4C75DC5FD40364D
7DE09C31BCAAEEA54B71FE4A957114EF8715A5716741FC5B
C7D935D73F9D4A6D6114F2680DD0432F8FFCFD4EF978A932
A7230EAA6C5265DECAD20DCE2EC36996749331C9BC70F770
4DDDC973794411C00149EDE24149BF09D80F94D55F015D6C
E7EDC61E2A21898D4F3FCD2F4EDC831E0B838CFBE92BF048
7E1239BFEBC04912EE1A09E20B78EC98455B2375656F3195
8D4B43C62BCB73DE2CF33BAC01CCD2F86F8EC47CB2381453
53EC42E2694F196059F6C597E679A10F8B3DDB05A5F071CF
B6448E60EC3D64931ECF114C818133726C4774914E3F950B
F7E3C7969F32E5AC91C0FD188E36D2C407B2A248667E7693
7B011E3E081990E0429FD5DAF2BA546F3C3AB53CCD363FF2
0844887C98BF4433562145B4E7DAEED412D6B6F924F8F857
4F1777CF638C2EDABE1E711BFEC6FF3922119D95E08176A2
A31E087954A9AD12383F23C18E1B1CDFE30F83C1716F57A5
B537C7BA53CD6F226A11F07A9FA1B200CB7D736BED618D9E
0F992BA871D5212EB017A4883A83EA8F8DBA617C8E804355
3BEDFE0B0049C9E912D40CF6C0510619CEED22CEE7B9708D
BC78A35B4FE18CC749DB0D132AD3965E40E375B8C82A6854
C417A89CE78B1953DF60A65B067639D5E56B8507716E9468
E50860250831373B16AF3F034D9D82BF0175FB88224D84C1
10AD95B0B5550AEB557416141942239A8BD0D969EB826064
7D7B5F760F154DF9A6000667EA1214C3DB2A7F999A6534AF
256BD0EEA69D53E7D7D4C6C9A0BB5E58C9A7045A078268CA
4B7BAB74EF3A4F9E8CE9AFE189CC47B28900BF38BDEB9A6E
24C5CEA514BDD9EC3CB77F84C241DE7FBA53FDE5D23DCAA1
B53FFF5844B8BA1D78158D0965A1E8D3CB509674AC80DB44
53A50A17994758508C0AC3E0BA5FFE889961B42850361D5D
5893AF49A5A09D268EF99D5AA5C76EA69D204AE8E8E29344
2ABEF4BD816F7BAD3BE5AA455B3B9C80ACABA6547F49DB43
0B42168BFBA91557BD5A81F8BB4BA1A22A9CBD49AE8D0D8C
0FE7F91624C3786F7789381DE96ABD90A199D1E60FBC646B
BD89D1B1111CDB484E9D081851F154AA3185B08E85398CF5
5CB89F30410085613099B16C514F7F56FF25BAE823972FCE
9BE26C847092CBDCBD86D411949DE6ADA77DD7E0F45BBD25
47B7AEE6D602E7C60F5625A196A54C19EF3760F8A67FF201
5E12D49B53CD6BE5B7C2D316C11240368D55ADBAF2F35166
16A2A157E13D9D5D449F48F047AB288A43F0596BAC5D9AEB
60C85F4F2C5EA34E2A4FADDBF0102E3010D84089300FA705
BE3CC5FA2BCCF68320B9E21284E15FE10640D66A7CCA0CD4
D1C58300A9EE289A14162AE0C09722FDA1AA675C16FFC261
62A7CDEAA101F623DF6007C4E06210C145AA25DE894CE177
78B67AF19D3F9B284AA9CE1D7BE1628A8AB667D8D294659C
A0E661DDC6ECD6F7B386E69028D4161D698A90A3EBA6B891
3450614C7B5DE94FCF3306DD7BD8A5B9EB8B65119C1736CA
D0574D8932E20058A9DF3F3F0AD4D4DA633D30114F6ABDF2
BB96414EC94D18627FEFCB39097414A17FFEB35368DDD40C
2D098CC94D038167621FCCA88FE6907838704656E3F41F6C
49ACFD5FC899B77D630E47A2B750BAD21D274FB92E594722
16EB22A8AF69AA2126F98B6782560424DBC51415992667EA
45323CAA746EEBA389500281F6FF5631899D99B3A4148D45
B622825FD1CCCF9FFDD91F31E0D5F1FDA5F6F6FCF5B0CEF9
7028C1E2ABAD42D552DF6BB086559EFF79AE6EB6E127690A
BEFC5DB019640F0B37BD5ACA843818DDA4838A2EE8775140
CD24D8343A12CD714490953CE2B082E3A3E769BC3D24EF34
FF91D293D213E3CB164B2276C725E42A1842C3E219AB2F4D
50242B40DDC562B8AD60352F25C5FC3A2FFE1E868B4B9AEB
B8BBA1317D4FA22AE313A12501009EF2EACFB09569A47274
93EB1478F8EC06D32A68B2FF5708A4B31A489350FE6FA98C
CC05F943DD0B56E3FF96F573BE4D7C6FE607C7C31EC1FC92
8AFA59278B1C5E8DD4FB3E212FC5BBE5A2C2FA05A73316C2
171523914589C9D0F4D078AC7ACC686B9DF079CA5DF6E6D7
4EA6AD42AAC357F04574DAC70AD43D9D532B9AB245870089
7149F1F047680D0B6537E7B5F1A9D0198872114B23336BE0
18CB647C31B5DE67CFA25F408955150B56B1D93ACB08525C
6D4BB191A60B6E7E5BF91E49581ADAA76C3BC5468CBB0CE2
0E0FBCA84BC767086F8438244CAF9C1E62462E88578C1726
E0512EBA658C948F8D9523C0052B8BA446B108B7BBC8B709
E3839A68A4703352ED48FA858143139E205573DEFA9DEA59
633C6D13C88160FD30A6363EBED85C14914EBFA0E507C3B1
FEADBCAF8520B2A06E5374439F60D451DDF73AF7E3C582A5
16DEF0ABA54096C373C3B73A42B743A037B5FA4EEE8331C7
BEE0886D4363783862F009C12EEBB5483A4268CE99142028
2F96427D61865CAF3D858290D8496026F72947507572F87B
E5776BF8876EB2C96B2F6C82489CCE7482BC64BE828CB19A
790891B5A3B5AD55B125CDB084ED7551932B7046FCF91B26
BA0BA09E11B0ED91BC02DAA80B382CAFC6C8A523B24E8E9E
85DE280A047B1A275639A8494B1C210C99F3142701482DBC
AB74C8A03E0E11199E4169C7515C7A41EDA18A0AEBD3AC8C
88F82E26202C2E0102AEDF45BE38068D108DAF1131EA3AAB
B37CB50AA474A00E3CCC6B852E576E8E2AD0F9E5CBECC155
1FBF47BD0FAF8313AB7B8C2F0BC3D9CFBF0BC1F94021FF88
8D4BC12DE6026D784FF38B13ABFB9F67C3AA2D00D5D055F1
E428C2F471A1673F2DDBECEAAE360E5878F589F7A2CE554E
72137C49D16700EB0D05601A65676DC2F589722C3659F467
90F489E0FF07217EE2C54FEFB3ED2A2A73512E9C5D3E6299
A7458E1EC8702F3FB8C33F99943F7B8012036109E72789F3
EE10642A2721EDA783FDE3A2D6F85D03B1A204CA3B2CBAF4
64ABEF67F62D01263B68D6D1345B558790B8FB1B2A02FCF4
4F72F8D5A0F5EC7D517B2FF8F7F961977565F56375584FBE
54EF7DED9CD5A0DDC64B335CAA369FD58B9222D7B907D324
9773204A2A4E66694125B4DD4DEC7E691E56EC6B9269EDF3
EA9A3E39816A58EE8216EF274C0D5350BEE9CBCC279E2D9C
0D200B182B138B57846D7F4A2701649709BD2E187A84D9E1
0D937959A43594113CF8CE7A9CCD8211926D6B4FBF71A61B
4941FF204DE2162FA8803B761AB23C9281B4F11917215BB6
70ECC753283D3C815126FD4AAFBC0D8D4795AE7FCFC22D22
EEBA953BB629161147429445C269BE0800113E4B7722504E
9DAD91D82954EB36F896CBBC818320D306EAC7D8ED25F460
81F0CFCE1D94FE0FDF11BF6749A7ADB2D863C321E85AECE7
A00FA3989E684423F0840E6B483CE522210515C324293385
3F091A48CB56C688092DDD309F63FDA9EE54BD7C6AB8AC36
FD76BF5A7646BB73DAB9C0D91B95E52B79EC0796854882B0
49315218A8D6F4A161D57400E502D9A4868E0241D345D9BD
AE95D7227877DBBA24531B05F2C15C3395407E343A51D08A
17B3716AD76D22F6F4E0BA253CF3CD38D5967B6E14D8AD27
3F9940F0ABECE92267D252FBB3F4E2E433AD21209CC2CE2C
BA549FE483786E16DE58B2E16C90627EE41C4F8789FFA726
EA177A2509C47DFE6C38CC30CC09B50922848C289C167AB3
7D09CE7CE6F7E33A8BFED588530AE2362B139D8E62E90DF9
AF847E10FF0C329742EC191D82FE9F6A373D43A133BE73A7
49CCCFDD9792CB531EF285CC61135CB7D753777046A903D1
33B62292C0A3E1F71B7F22C7D8C54C71E4E7B7D4F9D96AC2
5BF988CF339D9A8B8F15CF9037CB3303163B36C616DCDECA
263567F5CE66A582B101EDD7BA81E0CC7EBD18D1D9A3FA50
926E43940718B2FBED3DF628F5CE30B43DCD1C28904CE605
934A7DA3A193B2D9EC73B3A52FC923D335723490696CB92F
19B091C04E72BDFC0A5ABD92DB3FD97A58B243A291C2B760
DA28C8C01E5F1129CCF848CA657DD07806F35FFF3EF52CBE
A30AB8C71E24C7ECE5AC4C5ACAF56FB2E5602FAFC08CA1A0
85F34D49BF4CC0E4604603EE319C7A14FE1E26231EEBB69F
EEED6D1250C7F38FAB75F204A165B85F22B4083BF261371E
716493F85069089A38A1929CB7DAF5862627B8E13B9443E3
D97FF69EB741A887EA2087FD31826A12E4ED6FD5F12E3D9F
275A1982549D9D0591AF26DFC7719290C9E96D32AA982E28
3A85DB607AAAC837AEC4FEC3275061E10DF3F84E2443AAE2
D4099E2C92CEBC83C9B74F0F813C78D09E7DB9684279AFB3
91BC152AF8D56C78FBCB4BA882582780273155C7AF52FC81
3A21BD09F4071B3CB67067E98991A22345965248A36FE11A
3ACAACC835FD7621539CF587A10CC015E25D2FF2E9343588
53CCFDA2598A0D5307A9F049E5ADFC5E7432E925D2FA5813
34A6527C3FEB37477CBB021A62A21FEEE1A3454267B2AB0D
C361EBB6B7232F3A411C8C3760CC6759D05A36DDCBB2E564
42D9BB7690D777F50284AFEF2AC83532CC19290051152690
8CD2F976ACC7A46EDBDD29CA1482A0152206E56F88A51D5C
D183316D654BB652B95BC3E52215FBB3E2DC7DD6D8306F4C
32559D48C0C417BA7AB4C3EB8203FB4EE321098CD1565073
DF923774B0D8094DD648BC3EB1F7443ED46E94F0F400E28E
8A5FE7EA3E174447E0E810FE969237E836EBBBA5C6E9FED7
126D320ECA1235159BC16FE7C779D0880971FB06F1FEFB45
C7F0016EDFDB30BCCF19DCD7392D77FA4A5387062C2C8E30
51E58305B713BE160F773CFFF162A167DFACA6DDCC82E25C
1FB89278BE30B64319F37D439B1A20FCCFD166F80B702C20
5A21E0A5CC6CD7726E20CFA194F52ABD231F5C8BA327DC27
EDC7D99619C628B3004465A70FC1182B595C630CA29C50DE
3E96F8AC4285F30F908D6784BF362C81E52057D15CA490B3
D859758456C0E7A3560B566CECDCB00D24361A0F2E742FA5
09CC28F5631AA82685D8B4F28BC4EF6CD4EB3BCE4C8E23D3
4DCE8997EDD60954A92F38847320BF4952944595A1F708CA
73403CFA1E942A96FE18679F79481F025AC1F12423539AB0
16C36423827B210BBF112BAECF0886D4B614DA42EC16F6B5
817B6160FA35603BDDA9F0BA675F715579434FFD2592E722
800E3603228A0D94CF6789E625AD8F57A6EA61AFA1BD912D
CB7282A5779A7DC2F6C0E0F5B932102ABAC5BE3655062CEE
C9B30D93F64E48F6962D1E317CEEC6153699DF47D20DDF82
724F618C194F3DE392BD56A6FA72EF6013B17F551EB33556
6791472CA5FFC0505A5264132359C36F3340F665E490061F
D0F6200C73DD9C364ED8C8706E9052EC6EFFC8A0DCD50984
2CA854AC5D9F6B3DB4355D2B9B85F802F70F43D236F9B964
A27029A534E71327C951B8E2C2FBA9B1B4C62555A06A92EA
17D21BF581A6EC2E9CDC9149EDCEC86965C838F1F6099C35
74ED1DA4DD0B10D4FED57652C8B1FDDD31CF0B0127710FA2
B585BA7A4FCF8040CCA28C144003AE88F54E9B3100FB9EBC
F2E38282AC6DC7D1000E4841BEAFED0ACB71AF8EB74AFAAB
DB9C74D2147063FE2FD3710BAE8D8FFA6B1BF6D1D0681F0F
ACE8E305B076211825D00B48A40C481AFEF27C83DE2C05E0
F43B91CEBF2A2668BA31F2B6D1D1F397348A892A7024A5B1
629324306153AA6268F72FAB176C7862858FBFFE2D91806F
F72F9A499E5C26267BFC0065701B78C52ACA777F28385D44
7D8EAA12479433A160526ED4EC4C2BD10A253B21E4C754DE
D7F187C70AB97BF043B63E6842AB18319FF9C14AD4CAECFF
4039678B8B4A7F86834779415BE05C87D398956538E56E51
110DC24D81FDBD52D9C9ECF4C5EFA96109A28FFDA729AE10
2F0B4727E8324C2B799A97F1B450DB998E0597C78B1ACDA6
3B5BD11151DA60AED7797EF29B47BF032D5B57007F02BD83
2C9759D13BA7F6CBBE54DB975A7B8BE6D035960EC0D38E33
4FE2412E88058F264F728FF1C5BA7D895A357BAA74FC4697
F2110128A4E9FAAD09A0C340AC8875475146996C39B8217B
F7CB3E0A4870DBA7A552CEDF93BFDC3F14DB2AC11C65751C
317CC5B7D543DF90CFEA515E363300E5303FED971CA2F104
ACF0195BBD7A1CE2FF3DA616F556F690820E68D227FCBBAF
6CA3B8DAA8D13592ABE8C3C58ADB3932752C27252997E623
5043C9153ABDEF5C6DB7EDADD86501DACB81D2B8FE056E0D
5F9199C638DBB40B35092B96144AFAA6A4BA19F96E1705A3
A39A09DDD4A358DAEA110A099BEE31F48553296A3A5B0869
3EB2796607F0852410734A88B64D94D6F8FB398C49FBA572
658522A23E553E4F67EE8FE88A4D26ADB7F043C788F9B23E
12D97EA391409AC26359C590DE485629B72EEAAF3309F643
52467EBF1FCE58669B3784C4158476AC3AEAFF1BCF015A83
29BF549BE442F001249862DBB3182BCC61859C0D8A1F52A3
7D1A7A058EE6D35DE243336FDD423E77C27498FFA82C963A
ED95439C515E14F87FB1EE6C067C1F20C61EE6B1F60E0FD1
FEC6C6514D7F286233714F281BF6A6BD5588E69C16ADFCE2
6D07595B8E4BB079C2DF5C0BDB5438FD4C96F8F7BC2FC3EC
711927B5685717ED7CA6AD8FF096120064DEB81A2CEF45F5
C040846186E6D7A7E3C6762309997530F999ABD6FB98C9E3
26BEAB9E77D8AD648ECE31D1EEC980298093DE13B1F9DA74
07E024670D42F58412EF3AEF0E4B6566DEF0155D97E6C5DC
ED5FA0058079760E821F303295A9C9AE2A257FEE0D6E8571
70B2D2FD8E338D278DD3846EA3BC5D67CDEE612595BF3AE8
E75372AA8B699E8601B92999BB2DEEF66F6DD6C9B4232892
0A89D1175CDA2E329762675021767E84A73025E18F521512
51A8A090A8006286A1F213C7A7A675D9D89E7FDD2000EDD7
9C7351A194D3055CDB3167F4FEDE77F1622618293A052C66
5C0E60F874FC3783F1C4AEDA0C61904A4BD7CA9BCBB648FE
F9010581910188D1CA6EF6416FAF6ABEC3AE486236007103
652693578A063D2BF76B589066359E86E3D2D8664F4C430E
AAA2BC5E07F5FD7EFEB5EED49E222011BC25EEFCE1920114
70DC29C76D8E9D627FFCEAD5AA5A24CA466662242436C88A
319C132D7D08591EDF18303BE01192947FBFEF0F4B5DDE25
02822EA323E5F1AF29EACD9550A3FC5BEB5D2D78465B7A43
9937BFB1F735F5A7363B912DBE487D0B769880785AC55276
B17E13DE69D992B0633BBDA2D6AB71466053C56F94A7DDF3
51201CD290BE4E32CAB9C50E646F9830492FBBD8975550D4
19812485AEC0EB1B10682672CE0A601B457AE1E1B284A7F4
CBA939EEA18D44752803E6DD9EE318E0449F0B7C8484735D
F25372E706D136EDA08E0423C23FEF937C8FBED98787748A
44C015E3958D1A3EFECABF941ED8346FD7A049130D884CF6
F893CBA2809C510B055817C1ABDA27E266E156BB35AFB3A9
87EA6FCF6CA767C6D048A5AAF799F940444BBC062E000A91
F6BFE234218242CAF538EBE5E131ACD92CD479ED87CF569B
63C296797D2F244D561CB497790EEEE97072C5696B9D15C5
69FF03B97498B76B3399B9EE77AAE1BDBEEFFDEFB4D3B5D3
8BB846D0CF20177A45CFDBECA2750D9B3FA743C65356C190
1C63C172D9DD7E587F593BB686EA4DA0EEDBAA5236890D23
9BCADD9188FACAAC656D5444435206977F252A3C22697063
C332A2C91C3FC3C187CD6A7C1B725DB2C025FC5253876829
D2844CDA270775BE69A29A6BE3887A5E6BD8A6E8217494E3
4651A5F6C18D7FE33F62A08A197113B284FFF02A7E95A8A5
70652DBF91A48A910D3D5E7B3CA97562975288E1DBBDCA9D
BC8979A31443EE44130D8EEF81509AD11818D3CA80A28046
C45CC3A04CB73A90849EB5AEC8C92C612523E2583E408390
D12E230F22F5DBAC95F40DCC7C3F1EEE146B4E7C0405C4E2
004F20FA2C7A77FE30C00E01C32CCCC30BD7FA6D691154E0
2E36697E17DC8D08A30B5E49C78AC6ADB58B42FF91DA0A2E
8E718C13DDEF82783754AEC7451099E9B987E364D28BD2BD
B70D4D7D88395D6EB0D32869FE3D3A2B7056DCAF4945D532
937EC722C36638423CB61FA699DC3B2EC5F570AD68D8149A
122DE90849301B0397D42F1C8AF110CA7F505E0E35EC5B16
B80F90362554228C627714DFE9B85B5A76BF44B7ABFAF688
2BF1CB46ABAE2A2B0A2B4A59853FB2A38D001899DF612666
9427D0C5C794D769FC84CF798599EB3594E974177F31009A
32D955FCD85E03B9C71D429923E64037C267307AA1F1E924
DE891471499F650357700F5EFBDB2C4D5846A27A85576034
4C9C2E16C528D065840D5C24E66AB46A021619D9662157E0
4B2B08FD94E85F6F541182AC4D28B10CE0DCD775C26274FA
3CA1D440E31C21A1A84DFB7C176C598A7BC9608BFCCAC256
3244BEB7C9501B7523971AB423E990C68A2EDA9DEEEB1CB9
40DA3FB52AB5AD317812DD98BF6582AB6E3DA8EB3E34A6FD
F2CEF44E515403232730D8BE93B396D204B44E34004926FB
910EB7D54DFB1782BC3E91D68B75C7B5AF480E9DE2C88F4C
E26A19997A60B7AF3A2FB197B764D329E1C3A99F287A7E87
D95F7DDAF0982072AF928168301BB867921CA8561C760592
D8EF90FA2870B289C7EF4D255FFE8D92EF028974680D582D
43F16D73457E4F7FFF14FCF554D8DC11B85B460F2BF403BA
A24AE67DB2230AF0609215AC7A1138D425EBFBA52851CA53
9BBC214A301C345063AD9C9EFF57DB6E899A2865FC25E902
E442142BE7B46EEE8D19C740CBD916599955CC70871225FE
2541BDF5981DE370AB6037935602D05006C63FD0C2D920CB
F3F338A8A394B0CB86E1639595844DE854335D43C5E49AAD
4A963033836262B146F04190E059F141D7C17E781D3E26B0
57AE085A9DF9AFEA3317EF779F65CD938B53AD08C31EF869
AE8C430AF52EE350642F3B921F01BBCF5C0102AEA3A69E54
4C903FB8088D5DD297424D3D760495FCCC03ECB8478B4A0A
602C69D548CAD29382B2DB3D16CE3C3CDBA9A98EFCBABF55
AF50BAA464DD7341AFA1B63C68AB4D26747336D9BC0FFA7D
1D1013BAA4A7DD92D5FDB401C2074A31DA812B662EC733E8
9E784890A21C6C067CA44A0517128FBD6B4BEC997CDAF56A
8DFA1E6611CF5B7E6BF797B06B1B251EE6A9F03A91FB07E5
D5EE23DADB65C827B2D906388AFA7A4A5D7E463F1D3D8312
BDA60A01F5A5DE0D5B01A9D8E6DAAB4F68282251768CF95B
ACA1C67C6839437FF596D17C44FBD92C98ACA589C92C26DE
3A7CCEF12B1367E1B6DCF53AD6C62A9CA43352F65CCDC05A
FB374635DE59C0D9FDE1B2FF81ADEDC3D7E7F4E0EFD03CA2
273125B8B107E50A0DB5D276DCB2B7D8BA8F863CAF84B079
23AA24942265320DBAA6FFA4AA92B237AD73FC0AE9E3B089
0085743D7F3854B87377D48CB10A58C3F878A31E1F3E5E75
AE8B900DB6BA1234EF3DC59F9238E63F6533D170B0967A99
E9F9F1A09D9190901F97EE63DB0222DE3F6327E25A94F13B
C0BE113A107026CB289456CCCBBAAF63DE3C502AC37397D3
A346F3333DC6836FE1F61CCF8D37892417A94362A90BFCB1
0C587C0547574984789B5BAC9B73D89F0B6F5B551A16390F
9D7630625E1AB10ABE5EB1950EE7CD3F12BAC415ADDE39F8
B6166012C43CDCA7917C03B2C2A86EA77939F983E590F605
568D4ABEA3C2D6FCDB3E1C4A967E83C3AC26AEE9A552936F
302C7313CF82A7EF4E557C2F29E095487F372F6FAF39B65F
C8F156DAC85456C4544368381A7ADA96242FD803F4C918D8
F4F115A42FFEA61D4A2CB21123E919D6F1271A424AAEC7F7
F37A73D81A5572340F88C9287A1D6E045CF7F9895B80E3D6
D6036302AD55B30BDCAB58750E4C3EAF4262471515DC3146
336F147F3EAE4AC6575A80D89905874BFA592B931B05292B
F14EF3DFA46FCCB9BE58425C3F7DC0E1CE3A0E3C7BDD66BD
09D578332A403B63D90B214A08C1AFC2847F8EF018ADD4DB
224B13842DDCD1971A3E63EDFE03F86ACB1CCA3C299EE266
2C287E687089ABF23D3227BFDDE6EE42124612E1519FA558
7DB32955EFD4A21A9C93E0E50A193D41C787B176D213E40D
3148991D8F6E48C1E71704CDD2BA96E5207D4899E27EFB59
72A0D6BCB34AA146A5CB923D3DFF7E11D69EDE42102A4C1E
294A1A61DFE2D7F20769C5765D600B6877E00B8F50DE9D6E
76C40B2031A3AA8B8D0FFB124B02FEB88E10E2EAD7878063
6A426E9FCCFDBAD448AB03BA51CEE3FAFBAE3CFFB008666B
D93738040F3B239161DF7BAAEAD7D7969FC428C6CB846B68
57221E604E689F3F8047D41AB4127B82B959423C7FDD3DFD
DBA313085383E5735F4931860B317763618806C099663531
AFA18F92583819E143A333EA74F8990E44155BAFC8303535
A55A3C61F43D818490D42EC422DD4BD08CE67592FB5B1D63
E326F43C8F8C251E6B2B625C65846DD883B1F50BEFABA6AF
1410B0FF4A1BA7D44E6AC667FDC26B9AFCF06D0C92066430
CEE6DC7C1E10F6F87702070A37895B6F87A9D3EEE5261097
20A878F0A2CD5687B85A369408A9A4EC1C59C69E10F83D35
AE17932BC89084E24BCFBDFAD907B8E2E609B86F26F808EE
C16ECF242FDB04873F41644FCEF2BF6D0C86F4C65ADE88E0
707EBEFD57C2101B5C6D47802CA835DAA52AEF84BAD3F905
6DE8A8C95AB235D4E19AEF6E52254837F14B97CAC912D222
CC8DC5FC400914E75F1461F007B4EFE0561D9C2D8CD17C2F
024B3E496F4348419603448E03EEF17B356B51CF5F7510BD
»;

for sort((0^..^($testvecs),440..449).unique) {
    is +T2.new.finalize(Buf.new(0 xx $_ div 8), False xx $_ % 8), ("0x" ~ @t2set2[$_]).Int, "Tiger2 Set 2 vector #$_";
}
for 512..^(512 + $testvecs) {
    my T2 $t2 .= new();
    $t2.push(Buf.new(0 xx 64));
    is +$t2.finalize(Buf.new(0 xx ($_ - 512) div 8), False xx $_ % 8),
       ("0x" ~ @t2set2[$_]).Int, "Tiger2 Set 2 vector #$_";
}

my @t2set3 = «
C380A44FC98DF30033706B205994915E9605B7EBF4AD981E
6DAF682E0C6F8CD811385C55994515D92D83CF333C230CCB
5EAC9623B188335114079B31F2E42FA61AA99F4292432ED0
DA08EF7413890F21A5617AA1F5AC4FA56074E4088A5A03E3
359B88AE41E9129AAA8123140A275051A75E4EBF610C1E91
6BE1A899EAAF9A64343EA090CA4C075383D9CF002AFB2A46
2BB0A501CCED454FCABFBF466FCF2093BD0A8CC2C149FD13
4131B3E9B62853E1FC7D8A75DB492FEEAFB41294570C81BA
02A8873995A199D8CACF9B44BD91F046B427676695445532
9E416238C958F87CB13708D31684A334D82EAC2991AFF1E5
7263ABA832C9B21BA476BACF4C1C95BDEC6EB181705930BC
97A0D194744A49F6E04ABF9351FD94E6B734A2C490C561A3
839A73E06A3A0FC747703E22101D0B695F1EC1BF726721A4
8C814F36B9DED5983E179B241DEEE39F7DEFED83F1EBBDC4
DCC16DE4EABE089C89B07B805EC428F8DFECE231764860B7
20BA1F647A7887CC02F06D64E834A15262A7042852F312DA
80AE8A86954EF0A299CF3F7F3400145A0417DAC67C92C6FB
44E4232B997E7137AC97FF7BB5E58FCBE40DDF4CD222468F
90AD4AD7DBEB578803FA8C725F1804D33694D5EA6FA9AE3C
1A5654B84EFF7058A47982C4FDA9FB598B3E44E3D971A94F
C85BD3189602CE476F22C441F350BEBB08E57A9226C6B51A
58D4109FF56C9C10004BAE84EE2BE80ACADB67175DF660F1
79390183C3D1D7B6EC927A8759112E575F7B43B187C9A93F
E911C0E03D25219EABC9FE47A929022D69E925FB30D2E356
8CDEDC61148D1C2E0DD5D20316D6AB38D93079EFA7516AF8
DE165157C167D0D3158D52F244322FCDC7747887F22915B7
7E96E1662EFF0AD61014EDA3732AF77675450F237ED2AF0B
CFFA4E6769F67745FA64A2F51B74F1C035DFE466A8B14A95
25100B5E8A19165D90FD21C3E0F9916475B836EE0B02F1A2
CF13C84C7A3F2D4899F99E4B2B75A1AA54AFE91C8C3F549B
63685322B6131B47F86313EDFDD5DFCA37AF4EFC7A3F1368
D5DCBC0A2D9E7D64BB7793F3B7268F39D3D62A209CBA9B0D
BB2161C68A3A241D23EFC8864E0A37B1549C2A78D2836031
315CA9D1696A7A2117DFBB7E9B5FD2ABDCFD25668962D72A
B01A2B0FCD1FDA89034C85BB155A0EF21AD698A26723D795
243FDD0FD11057AC7F03F33077F5DD8958EB46E5C579C66A
78B0871FFC0A492658BB51C6AB500D8522D74CEB1C050DFE
88CCBB5572E258BF0D7C9FF62CDF3AAE1530960C96B0026E
7876126649540BD6E46165DB4F0515338589E7143F5FD8D8
62B5F33A355AC15FA89D90718206C8D49EF5214C4A2AD466
B5E6ABFEB6F33FB8BC5F265F5B8302633E7D96D0361C3A58
3D792597F8A4BD6C85A94C1C66BE8D498E1F20370F6C6F24
FE95ABF59FF1ED8DDB0EEACF030F2CB6CB42189218F67D50
9D1442740470C3F518711FAFEF7008A5B03CEDE3809E7C87
90EF07D70AF19DF6751FCFF10D32456211CC623C8428EE69
C357AF878389E3E0597BBB553C64B23DA8D0707F249AD2DC
C237ECF550B2E312DABA9C448C1A3D452AEFD027B3E9F5B0
28463CC20613656FACC68EAEBBDEEEF46553358BB1FC27A9
431C759CF733FCF3964E1F5ED2F3AA313821A1E8EA8CC2CF
AC5953C1ABBBF0603849305AF61CD6AA677C0222BCA91B84
30001C9069CF971C84B89E6E43489BAA77292C46A7F13989
AEE5D23C8329B662A3C81802D982E78E59A959A436F6E7BB
CF89B6ACFA8808A54B683C1B44D757E4D4AC527F50D3B612
419350C3FCFDE508CE57964C3FF62DFB15BAAB89511C40EF
94670C89B96DEB96D14D6D1B26A325716594AB46FA574DE7
CDA37F4E9B64E7A2410484D0DA5297C22FCAAC2C5720106B
2CBA5727513040BD7EFE765BFD0951FF85F927798FCD834A
96F139A3BE593EB0E9D935915DD46ABE8889638D207B9314
4CC17346C5D26A729CABA436E902BBB727E70D13B6781DB6
945300E27729D6600EAF4BFEE582EF63F088DAF8ACD798CA
44BC363C91DF843A3E22AECD3F84699E71E1E087A4782AB3
2DC73BF710A09D80496A007147E4656D5820C5C87A52A799
06CC489A4603A66F72E2905080435A359DAE2E3E1A4A8B66
492B9F375B0BDC56930098A3249DA440CD88F32D4B764164
86FE60BD3793605D5433F98A40F007330F621BC2A43F9D68
B7E7CE62686741453C031B2911C4B6E680D08B254922F439
8693E5C08CCE33191CE512EF2E74C7B5859C05F03580C8D0
D261BD8853B89C2F8B66CA2A0F9DAFBA6C6A0BC39BA78398
20B6940DEB4D67F15FAE5DDDB404672059D20A792F7A2566
3D6934CCCDBFB70EDEF30045F72E2C0CA979B4C09188DBF9
86A2BBEBDFFF7E8F01AE22DC011FBD6D6389EA11A80D1537
C14E2386A6249CFB34C1CE8D7D7153829AE0C045202AC0D0
7E16F92B55BBBDDD78CF540EE68F422495908B471513CA52
B31560161C6BCEEAFD7C434D26ADC1380F542C5CE23A7F33
3E42966752685D42D0D58B62AA8E9287139E0494CF2DEFBF
496AFE2B4DDD78E46F5392CD417CA452AE9D63AF76D65F5A
856656E89D79A12DC7E4E241995346156C4EF4CBE879F671
D50B717E963FA39AC6D9C0EE59A401EA2054A62D269839D9
CE4F0F50BAB7F475A83A9E5B3F5AE253EEBA779433F93D0B
82DDC7D82F9D9CCAEBCADDEA245B41F6E426C85B03BD8771
6B12A7F8162DF67023A4C7E8FF61EF42B7634A15D19F9BFC
28B24FD92C73B48216D10CEADB0484BC239A9644FD83DFF8
C6FE075B4DCE972D174D8F725AA0D3286FF9300794C41049
8582851E40B03AEBF3A28D0C3E7DB684D9144E17754C5A18
804A83910DAFD13552BE4D1FE2DB551E4F87791E40500B96
F98BAFADF2EC4711BACFF0C3BB849C4DA835B12400765EC3
2A6B55763FF9F8F9CA129A6DBF75A9E66D2B814D2E6B41EF
2E243FFB6EBDA647D1FDD917BCFFE24B6F93920EE1346F2A
A312FA39DFD63A97D7202FED841B8D6F96269FBCD3FF9802
DF81D94E8168A58EBC2B4FD14CA662F3753531958509BAB3
0BE96A3F41860C35F16C4F8E0FE41B135850472F33FDFBE2
E0DBFC489CE8AA55AD3DDD11C07FE6DD72DA629D2E5BB8C5
36B26D2F4DBD3FCE308F01843BCD637CA2A7702468381A30
BCD30B5AE7BBDBB6AABB33194FAFBB415D7D8338B1DC33A3
4B97B4BBB0B03FDEEBE860ABCFE2A9889D1E52A4267A8932
989F8D9AE297F830F517FE44B52A7D1BDA4C657092526522
C5D2106818DB6A49329C8EE743FBAE2C440B9D0812860AA0
19EBE76641DFA2A02B5F61EB71B7EE53F1F142E35F778F19
D171F033BE90DCBEDEAAD999A73B131B3F6BE0FA5443C925
28A2C8DDAF8CDE6A9879D1581AD51D287141CC2F37AD3FAC
E9EFB3EDF6F48BF905B2E4BD01F58D46CA79903D8465FA1B
4DC326DE95D445FF95ED69FDDF530ED2C5A6BEE172CE0339
CC99985DB8B98CC137F888AEA95B325E9E8E0C83222DFC8A
04A407F4D17E5A5C846DD83D250AC9BEFF60DC64A56CFDCA
7CFB2CF02EC9733562500AA1A02C8523F0A858AA4999CFD4
77E6EC62F20CF5C39898F7A3DB6D80693A0FA701CF0900F0
865DD9D1B81414082DC178BF17FE76772D9E721CC6BE564B
589C891F4C37E5DA4C500FFF6291C89DC23D1180426AB887
8DBF58CF70818194ABEF315507C8E9FE37E4247580311040
A517DB95D211CDA1269C85F97FD73C6653AE62599EE90EC7
87D6200832C67FAB81D1025224BD420CBF2A918C051271E8
C6BFCF0378B1944947AD7281C540BB43895B16053670D9EA
1790D006CAFB63A8F31E7FDC144A04C1641C241CF5D43554
EB0734B20B5ACDC9697920AC01501E289302A0488405D7ED
CFC48521089090C4BF3AAED1215A1795BD72B3B5676948B5
0073E90EB2AB27CC8C4167F6678A8544CD112521C20867DC
F2959D70D63C6561B2357E506D8750CBE9D9F36C876ABA8B
BC2E7DFED455CFB31AA311FF753182E46721B4DD6F208D2B
C06C4F1BFDF25CABD65AA080723140C5CA518741DF613189
CD3B9232DECC861DEF715C1938F6E8CBD985B7A7550CD710
85E2313133C60AA907ED2E0458528003FC7FC28F3A490750
A064FFAF65806CB4F3EF36FB8779BD02AC5461299F638A42
E4D34425FD0C150B608E5F7065DF36ECC5C86D88D72894BF
D31F276EC5108358E04BA0EA030F302CFE569CB22BAA61CD
02959F498FDC3459EC9E16A0DA05ABE25CBE297FD5BC5381
0FACCE20D81F07F79287B2E4E460EAFEAC7100C0EEB5CBE2
136A710B48FF2A7BD5F54BE705B1812F8EF3F66C1EB6B5FC
69818C16425833F9149CF73C3C635630AC83869A6302BD15
0FF2CD959C821E102EE1B471C7550448BD5590DE0FF30F97
0487CA1E646FF87F725B05158EAF97082E83AB1503B97D6F
C711E5FAED7A077A920580D86BB18DDA809D913F71A9AE7C
C5415A19B6383C6FF12F3264F3B98D305BAB15E4EB7221B3
78CC7F1DC62D4A9861D8860AA15A69DC3533073DAE07117D
F6961A51B11DD3A7430A5FE37E0CC3690F210DF6C0644305
CC183EF1C7945578521682AB94A0EBE74C374ABCD83EE1F1
5DD6AAD8D07ACA491339114422311EEB2DBDAF3FC356434F
3180BACA96DE5AF8D32445E0B210FCC505474821A62F9A04
502C605E12D80C216F9B77B5D5E47B4C69967E7980CB4B1D
E474CCCD58631EA47F04E3B8D2924B4FF64A08BB851B5FFB
608CFE91EE9078918BD83198B0F1A81D07433D35E21B6BCD
E31E62442A53784DC6C14136DEC031678A395B8051007169
8FE1C5B713CEA441AFCBD6AA5DEBDD9627CF6B40B7254659
435841A95D2EBF7A81B89C9EED57E4AD57625B08EB834D25
8C03C9764ED1BB71A8666DBD964BC6BC1F35FAAF3A9797F2
F95AA15E95709A46E74A84B74DD182A0F312C0E94C54BC2E
76AA6C980D7E8A9B7C11CF704FFE8CE9C0F99E3D2E9AC586
B9794CBA18B503FC008508C92DA9FEE72D8975A4AB6797AE
0013542E0BA813A83DF06A9B7450900AD6565F39F926AABF
5DD624F23A25C23A6756E8932DE53BE7C2A494A5BC587C8D
AF42722169FB924C58D0F421850ECF118ACC122F570B5BC7
DFC2636235CC121402EA3530224273ED8EB431F2CE4AA45F
7ED95E018955EE7B6FC4E912CE43E2C1C3A6B2CF2EA553CC
3A98C7D4F74CADB37330503698F38110284610BD27073785
0A95E914CE07CC6784325627F5BCC05EE00DDCEA5665D7EC
C0001A053EDBC6FDC46DFA3E222E6787070CB632EF482990
5811E004C8855D1B6537D977C7F396C1B4EA06114121B340
92DB976F725DD45DCE1F8AAB20BC55A984AB6EF4F2FE4784
A0567F7BB633CB42204DB752A55A34201D88A310443CD8CE
332FD2BDBBE1AF01706E659C9E47C74903A3823196136616
6F13060C4319B189349004C5F5E471A3AC56C869A5D55C7B
5AA0B1CEC5BBC79A753A6688292411B83A142A92561CC5CF
4B9B8371C5775369F0F8DCC1E32370C49A37CEAA9E0F8BAB
E27C8E14ADF33C01041B4BC408E42F55E20FFE649E779C3C
4823DD09861D51686106E127957AB8EF6CC35577F330A259
A73CD8718AFF3E365C92FDF4386694CDFB926A2CA8EC7F2B
3D38036EC5B1483F0799DF9091ED7A566FDE2283C0717630
6F394DB12B792C6D5D45B6A1295DAD3FD37C1B0C0986C785
B6714F1EFE5670B81E6CF2AC9CAC29B52FC62E2FDD8E99A3
5A0072FD8DA3A3F2BA40D394C95525F137922946F67AF4DE
AFC3BE1417C7296B2097BA67ED0323714AC2CFB41BE089F3
D315963CC7BB5D39E1970B7BD04E004232DD905432E365B5
C44BC37D6DAD7258E99783A60F13F07652811218AA311F46
BA91AAC981099AAFFD9ECD7345B6C4FCEA7655E59CA1FF52
5BB3C99AEAA28728A0B158765032B6A5A15636D485A214D8
3FCF53DCF06A3D4F2C2B623A3C8DDC3970247F6D3DA674C4
CED1E4912DFE560EDDDB3F4ED6685F7EBB7F80DA7A2697C1
A49263AC94FFBF1C354E60F2836174B264C3566E2110D993
BB12F10A4DD0EFF484FCBFAD5D5F47C1CFD1EE6A53AA5580
767B546E9D4D3EF5DD28A0B9D0D3797C011387B31523555D
AFC3956A77B24424DB59A9E12A9940585168AAD422B4F9B7
77D520A0A7E979FCC0B10D456CC418F5A476FC5CE9511316
5A7873522D4B60BA14CD761590EF76640ADF5320F97F1F80
851825E515D24985D553320E1AFD4C40959ED376D2C5FD65
2F54AB0464836182697FE909CEED14E1E73A07EFDB74CCA9
7FEB956B3BFE3A9867676B0F8AF2ACF7AED19C3D1D68D6BF
DB004072CE7B1D52E9801BB342847AEB5F9BE8CE36B5AF55
2761AAAFC471E83B037161A95766DF80C7DF46D98A0AAC1E
CDC26E17D3A3F04CE5B77618D4B83AB4C20EA0B203365D2B
29D1847DEC4CE7344AAA5AF6A75969C286D84A7CDF498CD0
F891FE788C50D81638433E07EB334DB18E8A0867F3FDC025
C33ADBBAAA8375FBD0870270CA428715E55E58926436EB62
0C5A703397D8AC69E7D6E5E3B062E450495B8C7574025FDA
FE6B5E067DDAABBCF3D5DB4998FD1A5E49E969DE7B8B47BA
59BFBEE333D37FB981624E064EB36462AFAD21EC6D80CF66
DF60D7DBAD1304EE5C033FAF8E199FA4B9346B72BF78F82C
341868C1F675AD22A99A1676A92A3E0BADA0764056B3D25E
C223233C7004526B23920B0E6DA5A3C334DAF319C6220597
45CEE5758EF6B73C6DE869AC5B81DD573258CBBA38E20679
14266D228FF4F04C82602A2377C80974CFAE4704CD968205
E4DF3E9FC213FC93E6D1FF65CF60730BF39039E58907289A
5309C4C10882675636C95B9C5CA7B4E14224FC8D65DBB002
55993D816EB5A710B11D9C4D0E36F45C02435C3EA2D2CF5C
5D68A136AB8A0A4DD63D807CF08F36B32CD15B38D051CC8B
A4A958017124CA7B062206EA758784E005922F98FD7F96D1
3D81540AFD5E22AAD96FEFC6FA18B49ED457EDD78EB7FEA6
17480099FBB9FD417953FD80A59E4B4A9EB82A63D823D40B
F7297F5601957814FFDF21638EC3AF675091D89615566A4D
6F8276BB3DFC50651C53CE64F7D91AB722E9DF2ABB38B23B
1AA985D633630611AA77EB67EFBAA61237D647C5359E20D5
ECB58DF9D54A0ADA99600656139008AB9B9AC03E363098E4
3C21CA6CABC8384BBF4F8F1EBE2620AF74ACED384FAF73C6
ACE5C17532453AF5E13475FA6B09266B932D42462B467A70
E6FA68E56D1FF50723A117F22193F4498598C08F25194FDE
91D11BBE454CF7B02E8DE61EE84E2A092CDCEB685D9F4766
E478742A60A30A75364E61391BE2A3C63928062555942F2D
1DD52CFFF84F91243502EAAB9CDC68F057AC117C5DE79CC9
5933F960BEE3FAB749259CEDFDE664387904CF51295FBE79
441332A4815C852600E26F344EC76BF7B9599F1A62ECD889
586139B17EA03348433FBC4837642AC4F9558DA3C51FAF7C
FDDDB851990E1F2830F0CFC1F1D09055A3D175DF53B2C9F2
D8C95C4A0DF35EDEACE105B8737358298B09677E5978DA1B
19DCD573B585960B5B1899715050072CE52E6CB407ED1024
6EE225F075A59474F08398B489FD8848CD19AC2F42B9C92A
8278757F0C336F9A59D0B2EC50E53DF1AC1ACE736B9D0395
B8DFDB3CA4A0397F6F932BC2C7BF466FDA92898363F5A81A
C2F878B92C24B71A2871CC6A2B2EA23F5CE3A60394475994
9D0C169D08D7889FA90324C347420FC6D9564314598E375F
9F1F33D21228617897B71E7D7F361C8C691872A7E21A9E64
6D4460C4217E565F10619038A0C23F387206F26F29D9BCE3
20329E664518117DD9DD30DBF3DE9A1B5AA9EF8293FA9B78
EAA7AFE52E12F18C1D4B5AD65CDE3CC301725DF3CAC012BD
1B0D7A1011980B2A4DAB9E8E2EC8ABCE617307EF379848F1
F06E21B047169F9A37520E0FF99648BFC94BEEE0EC4580B8
98316ED5B18038F3D5711E3C3E9D9892DEF825D8A533D82D
60A64CD42814030EC6C1D673C026A2C345457E7D8B6F4501
FEF228E428734BC4F819097DF45D5DFC1C5A304BB1F50879
CBF60612326EB2615721736DA7326DCBDA0FF0FB542B0468
0392F96C35D4E66BA4ADF7FA3DD4E70F8C110BDAF86B967D
DB8FDE5D1E415811EA9E9BEFE6965E29A00C2C58601BA701
B0324F3B62C4443033A81329D96C42105B1758F46FFBE6AA
7172682636B549B1C486431C7363141AA028EF3603735A2C
39DB9995783CCD9604E8E0BF975D07E3E56630A1BCFAE6AB
0C85F996543525B7E993DA0D92F859F065686214C782FBE4
64F633FC78F40378335994EE02F448C789332D83E367F815
16A372A2B097EDB3B0F5099151F67B7CA660D8C1A1B7DAD6
A4C7B9B4D400CEB1D011ACBEA8341D2AD3B8AD48627BDB34
9BAC39B684CD2B13BD46BED19A2D44DF294AF9DE1764A82E
F4A2A73B7E2250C8FD5290AF3A3F0DA77535721B9490CABC
99389CCCE43CE77D38F43D300FA8E2AFD6C1E6E16A885127
88D752D8BE0129F6AF2C9251654433BD208B612D3F8E61FF
4ECAFF6783B1E80FE6B25A10BA4797749143D81B4C6665B9
9234F26DE12471C99824C889E652BC57126BBD3D8C6236A6
29BB57ADE50EC02C789B1D514F0B8243E1B5C4ECA551E44F
830B1016126D10E92E2EBC2374BA35DE2BAA61F4767A70D6
85088EA30BF20B96AD60F189103AFA82E54B0C3CBF822FD2
80E5498ED448EABB2C83226A3F2C34CBF1D2792703E21146
8359592563A74A329124F52BA5EEDC37895D920B35764CA1
4B4B958E8421D349FDA3493E1BAEE3B43C57B22321D27911
DA4FA78E5B95048AA7B1FB21E12088A4F32C69BD19CE5351
0637B526EF445E4FF44A8895C560BAAE44AD8ED3774D0902
D29B044A65BFD0D8964AD7FF338D248417EE0A3CF438074F
E96B74EF31BE692A396A36321F2BB5684D82ED25D34C93F2
29F2C1AEA05A842FE8D952E0F8643253C39FEB3A459F69A1
BA3CBF9E1A77EE59A2C6F71D5E191A4C19C473754D53AA61
714C8E84AB51A15B486F860A13959D8F7CFBEDF33527EF58
2F6D7BED6C2438C9747A9FC64F34AD117FD5E9F7D64B0733
EB0382044E03278F8D6E941EB66B0EE48905DF57EBE1DD4F
D256F340CF2527581073311FDAD4D9F9568918FC0559E8F8
525DBCA2FD2ACAF4648856BD7853CD1EA6AF7A0BE8E773D7
0A3847F01271DF90294415ED899703CFF0EB76539F70CBF3
ED3CDA480F51FDB9456DA9F310A31D881C83843562B90F4F
FD477BB0ABD3ECB90BEB79605677034BAC49899DD4415F38
19482921BB0BDC61523914C62B1EF6BC35B92AE0E52EAB90
29FE36A4D34FCAB71F8E86901A8B5D8946163C9718F96F44
5663450B524565B27D4E64BB2FF18B19BB8884DFC7AF84C0
F6D24A216E91C8A3B4CF14A2378A0EA93D2596FB1C7EA22B
73D3169EFA4E55787D5B6A2B301416347B527FA0047B0E48
194C11FC723EE0DD0631AA4FED937827EB27D8EE69FBE40E
D7FFDD9723C7E30F3AE301BA4BEAB7B4A601D8471D48ED54
30F8C838A8E94BB4B1BD945744013196E87A2206845B42BA
5AFBD2C5F0AA042AEB9E07B21861ABC81C953F366AF3467C
4144C4E94BE9E96F7ECFB6320908D877517BDF31E76B7FEF
EC38AE508B01FE89D26CB7CA026BF3186DD41DAAA396EB7F
19A09FDA9B99C7F0122AA99D4A1F965BD9E302873B07BBA1
540D82CFED7CA081A33D1DF820E62CA377A5A1CB1C8B367B
91FD511A3B39C736912C94F0F8C914C04094E34348EC5574
5AD7DAA0D5D2AD49668E1D27CCC9F8D5A58B1DCF55FF5E14
B85E6AF5DCD4C999A2BC1A1D77E05C532536C641C928E17A
A94B3D37FC8BA71A70B6EDB6C61C76A92716ABD43A4095D7
CB01E4B1A8E596E6499761A3A1C576BA773438C7DE8FF798
51AD4B8005A83DD3DA7CC432C2277077527970D67F366FC1
2E6E56CCF1E4EBB28F6F45B336F7B9DD7D8E05B4539CE5A3
3CC1D0192A7679BC1B45F0DBBD42483A6DDE26DBA05C389C
AD19DC4A104D3810AD1AAE01284A495C3D98D4F88BDF3D53
A81C5D28568550FD6DD9B63EF9C742F4895AEFFC09959D4A
8A56C45EBAEA868B3AF6CBA6B38C48F0D2603B3129E271B2
AF5BCEAEB4AF4A811C41B8DE730702DE75FB7C3FFE3D23BD
DF5341FD396610FEE6877181FE24A51B0D63B2EC56B1C2D4
2A0F413A4F255D6AEA619FD24E6EDF9154009DB39D7C87C8
961C365CF2057590992989BDD26786047156FFF293083480
67C296CF1A1CC669683433FB9245892058172DDF2C9B203B
2D792515338F2740FFDE35C3F61872CC58D11B5AFCF63DB4
042B6F3312660A93F0A15086678B3679208F9AD75FF9B10C
A9D35DC92CE3DEBBF780E6BEF1A1A464A9C33AF1E328D8DE
0B909161B6975627A3071BAC7FD1C0ADF383003A549814AC
70AF80DE5E72F1DF361472258B9E3D2314FFDDDCC65FACEB
5916FC13E409542889553BA8A2ADD5A65BD9B3E4B593A217
3A05D2070A206C93DD651B741011B0E4602A1CE1C4BBE4D1
7CEC93056E97DD3F3942760CC451F7E64DEF5FD4C35516A6
F287F9ADFE6C58719B5650187093608518213FD26B49619E
DF975A134C12A5AC12A64724EA2EADED50AA0BC662017FAC
2D90A401CB47093B0F2915BA1E525462F7DAA542C0BF28F0
22A953A038C248D965E0BE5A32555840B195C8DE0B4A720E
E0B9D2400DEE0A8E8C2A1989654DE9CB11EB9D0263CDE218
CDC8A58C6A77C11851CD678368C318892D46F68AC6A3EB19
C06E1184FF68CED8FC131AE4603E77BD80AECF1C3A270440
9A0FAF2DBA2FCF30A13D2125B14A03A01587339DB040039D
4B0EFBDAA5B85075BB1BE7A20C79B5E0D593DC013B44454A
6683FD52EF0664F6A5F5515CCBD125D8F2E5EEAC98DD6069
3A70D220838A88042ED18AAFE8967FE8B4B812BE20C22BAB
AF04E702E5EFF0211D6409888A17430B620D3A7B907F7591
FD83AB6509083693614D721076D2CDE86CD06C930B864F5F
76C15FAB324F666625FD2E55F079FFCA3A80B9C6E07BBDE5
233E46CA0ADBD559908C1E28D2F9236AE9E0534CDA7E2DF3
D963140B678EFDF92F46BD672B71BA20E717FE53D19941B7
0B9FF6902A081C625C01F7A158CE917594BCDC76C90FFF9D
EBEE6BDD2429306153038D807D51EC7ADC24415195797479
78B4AE3D26BBF0321751430B0ED0993A61CC1D86B5362F11
ABD9CB67D5A6651D604CE0DC77847009A53F7CEA59A927B0
D2D0C6072736D5914DC327AC10C7C258EFEC573A02BD905E
3B16848168CA217AD9897073082FA137DE8F60C934681065
46AFD7A1662AEB9126DAA6D2C019274D0F59B9B018AB322C
EC6DBB67374440E91C3CE1365589D3305D3AA4C73471DB85
181899E5FBE6FE855737BB4293D4EB4635C2B8939CB03F98
6695946FEF7CFD5E5334C63ED0CBC64088F6214C32DDB404
111AC413465A26923F47F31130578B343934B9C363A163C8
4B231A0630EEC2E355F456C3CB0E55E845EE912290A5F1FF
D9BA1DB405D19649D086B56957392F24623131D831C9F2E0
C52C6FA31271F9535D26692A8E0DCE28D009B199372AE9FB
FA9F182795769F892366FBF2F88EE3613439C42C85629EBB
88EC5197FB3667BF84EA8A4061E101D91B8F95425E5F3EDE
731CFD787E951F2325E6E4A441952B82891A4A14CB0E4287
787839C42BE250F2AF2CE151975288DCD7E1B4609F6F06CD
0AC720641547B1C697FC1C1601E39D2FD19C73DAE2057983
92AB64E95C7F75DC9BB93404739677D2ECB1CC1D25BF76C0
0D8EF41642D5AEA094CE8FB69037D4BC9863D1F04D096E54
435FBEA59D86F2B6FC71ACB661B02EA8F972498DDE49FD08
85CFD78BFCA7FA15D876E39A797E7A0BA6E2B3A1FF8A02E7
C9E82939FD6F2B5EB487B9D72312B1CB5BF44A73D37AC33C
365AD2EFB2C6D9CD6E0DBACA51F9F945A99556EACD7E3283
A10F003C7EFDDE6C262B71A83E21B760221FC95879037204
D3C1B609C66123AFE036E9485769552092565EF4EEF3E755
D774ABBABA70CEC3762D4154C44041998489E85257ABCCD7
5525CF78F7571FA744A57DFB13EFA9F785FFE4B4DEB0DAFB
B983B456FA34AB6E445D33CFA21FCAD11DAAECF86F701EEF
E4883982CCA7BA5E9CA4D7DC8DC855FAE6504FA7D4C8F35A
17A3715F8C58ACD9B7A5B9C86349263EF048A7DC4F9F5C1B
368633A26D758DBBC3056BDC3B1C030864034EDB49E92ABE
B7D0ECEFED0D1683DC8563B63C6AEDEA6A0D5CDB2B68D4DE
9041FBB76324459950969AE2629F3C1DBC5DBB489FFD1163
3943C62EF66F975CE0CE4DB1ABF79E663F408E4700FBFC18
D5794F349A4E421923D017E00D2406D239EBAC5DB702348B
D981362708381920467FB77B8096BCA571E0092C15D10E1D
87215C58C7E26FD4B468112093BC6ACD2FE08DC8F73A7CDB
6D9D7325D9DA999D969B0ACC93C4EEB24BF8B16F3A1B0698
AE2B15DF4CBAC24BA38194BF50AF066B9FBB96DD0A9077E4
D37359F60763EAEEA15AECE475C9D377562D68BB9BFE84FB
8CB7D0480838D9923DE15B52F0F9018F1380E92F86E0100C
166228F4A5261343A6388CBD80C47950202060ECE1140AB6
897BEA78CAEE0E60165E7966557B70C9CC0F6363C5A65018
2EA61DA750566938C938992B1C31E282BB49CBD65A681A37
7CB1DA380AB9687491EACF474EBC46A6EDFDC4CD3904A96A
0BB073F53375ED5B74009CAF29993A47BDD35840486650AE
D2C84B1839EAB9143FC520399C02F33B65ADD2D337E84946
816089E1382F48F1814856141B5B7A04D0E97F63FB735937
6D9D1F0B5698718A946B18896F098ABA28C93AF436C43093
FC5FA753216C236E55987B3CDF34D03ECB05BD2C38396EE9
79C8740446C2B091F398309085E9F27AEBE538BF10D1EADE
D023979E21AC42A5A51E9C0B891AE4DBD491950EEB7CF3DF
59E2A2F38F2C1ED8627C399EE98DEB79DA6F24F691A778C2
EB1E1687CE062C04404672DC9B99941A7CFB00204CE64DD0
E4A17244C78E210D0A7955C2863DA22D6FAB8D8222F3874D
1DCFDB5B0CC796AB5753C20E3C4F1D5258F3725F73667E40
FCDC55912F1E13DAE3078C45116FD8C5CBBC4D4DD4CE5C4B
7093B5AA5BA856D6585D2E28FB016C2AAE329380EC136F35
4D4B8986A02BAAD0A5EE9B6DE13304A86C0BFA8A6529FBB5
906B29C35ED6847EB4C3AA266FEE0A82F73D43BFDAF90910
4A53902876F43E0798B63C2F94C31F2EEBC75717BD335635
DF7BE2B8B49A7E4B60D661CD60B60C3965C22685120B28F2
CBE0085F43ED13AF5503761A9609EADC449F8E99CB7AAEE6
3EEA2A1FCD0BFE5FDCB64EEE1FBCFB1B7038FB0DFBF1D732
52F837A595A0E5C4F3E3DE9CC67474D8792ADBA803D89AD7
FDFA3C8E78655E93A78E834D9852375177D8186E4D0149F3
902BAD6123CEB3426B1916DC5D161305A7F0A1B4980D3AAA
AEA7B706B51B46A373FB9C4317F2647D603915B1D60A53ED
3369777BD3677337D2DAD8FF564CD7ABA0D25081AB783D04
7F65AEAA8DC56EE4BA0DD19D586263D54508E0800AC27EF2
3CFA71DB478CE64B0A1F58C0035457DD31B29165926AE102
A4688783EB428D51AC25F3823025F8F1ADAD0DE091679F30
491EE7F27E50EC5A886C9242A8B883331772B2F7E972B9F4
5098ED5B5ABE33D873129A9E928E2BB20BDD6D139F63C815
118CCCC2FA8C9D3F6D7630CCDB072C5277EFE618D54956C8
37A2002F9271FBFA9642574497EED65995D9FE3FD3CF6620
0E4AAC44F7D974A8231CE0DDA49814C6C44DDB8DB40A2198
53E35A1EF2D10E7354B78B7A93A47859B536F5AA68F3ACDB
50A6E3DEDFEDCD505ABA301A55F1393BFEC30B3DF524F7E5
0B0BD4C7BCE4DF69CF4A714BE3A967E3C523EC588BACE9F7
E08647C942C341198A17C487C06998A63A977875653898BF
D94FE91FADF1EDBAABAEDC1EC1C046057196E47AA04AA1A4
0B186E80D86E65E93141530DD6744D13D88E876CA11A4432
A93288BB32673AD9E98604AFC67BC9A4DA49ADDEAC9D0F29
A0D7EF062549C43CD517247A7C9A3A9C0C4870D8277EA579
905CBF887BE1E91BC5D46CD84BC40FF66870A190E6359796
E83FEED8CAA8514CD98503C9A5D227A585162CAE8C13AA85
2CD25EE28DB5D29AA5563081E1DE0BDCA2230F4023881CC7
D32CBD62565CAFBF0A9F803D38F4D228FBDDA9A49E96BA81
F97910076BFBFAE0E6883D213193E9B7AAC68BC19833239D
DB4ADDDA17530EA1ACC1D948D6E4A5FABEE357725DE03082
68F56B1C8797B5ADD1E60A370A5B77112D6399E48E8BDDC3
85197DCBAE654074C80A0D80C8CEB22039EEFA8C190D4F07
E1A4002E2182A82E1F0CA87886DF36E7A960F9722C28F391
1AE5D84C3747D5F7936257D295C96408506EBC89B18D3EDA
2EEE023E182DA6CA9656DB8ACF494F08AC52A4B9AF04D74C
5098F7068A07A085E11F5DE9559A80F895614CFA754B08BB
B75E545AB80756D57796147DE93FA5CCC4D9E147DC17149A
54C743E9709468D27848BCBBD0CF55FDBFD665E956571E4C
69B34BFBA379EE5BAE4914737B0BF63AB490E4CE08060ED1
6DE72F5FE8A3FC41A7E76453D28624AC3F1E60AC7D45E8BF
765CC7DF0BF893481EEA7791AD31C57C22495A822C02D414
F12184AB0E2FD89CB6A1EEAFDEBCE7472ABB1A25774A9C64
05840E354198D6C292783B330AFF492AE4C03589D6B90C4F
B084ECF197DD0D632B855E4769D6E7FDD3B7EC28438C8D06
B32792215F1914F69531D355B43BDF7F73181205EB2EE448
5EB98AF11A3805AB7AB3CFA76E7B26A8A017B079E313FD9E
A7637114E92D788D0C59F76821FFE13773580C9A970867FC
7ECDE04D98A448466753E0E0FE788FFAAB0F99C645797E35
1CC41121392E425E8B6B2923B42E10E8B0458C606613FB5C
DC15114C3C4301B8B83B93C94B3AE303294F1B6A1C6EEDCE
BDE2B01D165F9D8A23DB9E37AAE800596F7DDCC9AF801B2F
C90148053E2F7B466305CFAD4671F42464022A8281BF20C2
461DA265E259FECBE8A6A868159C98E7CC54A7F57066C61C
4C853E7D3E3452C6FE8AB7DD2335F2B909C74D827551A2C8
79F5CDCE0543C0887B6C395AEAB898019E04C896C10D7BEC
FF01D7CD5DE636228245303128955CDEBBF1F9BCFC8EAA75
D61FB9842C4DAFCDAE7FFA311F27C6C588E6288E6F9DF317
F3C8CA152721EEE1A37AD593431C2274BD5FDD7D92E5C03E
06819D5441A9931BC9F4144207703464A1C1C9388986B5F3
7EC464B5394C538BFEF13ADFED99CD6F4653977AF7DCB42E
8D6A4A42BA74BB799997A93DE95A23DA2E671B3C05D37345
FC276A165070079322B0070428685AA123ACF09A8E2776EF
11779AFB63DE9681422E70A7943A2665B27DEFEE1588024B
5E9263C91612944D486707A87890EA72FA223744E3377A06
34723F5581E4EA43165E543E334A2C8A8BEB179C1562F6AC
94931AFF6B0B7C1F98714F26FFF150303CC7015C05805BA3
3AC096C54A82B77C81A7D2CEC99BFBD0B027865B5B69C355
EF10991EBCF73A2D52B8EA4A1E4B9C1910E6B1381DE720EA
9EA92FF697F71F7914DFBD68968378E6A9D92934BDDC345C
8F65512C4A462146B49FD5E529DBEC31F2C5F50D17C748E9
8388826771FE379A66EAB5A0CE650651B59E86F5DB75AE76
55509D3B84D576228BE5EC68AB5195128736DE5684B9F291
2FE71CF54869C9D580A5892CA17D203B78A6D754047405D6
CD19808D9CF3055B8A0A45A5ABF137C444BC25E4FD0CB038
5A6F82A6EF698AE7251D0568DE3B36D681F484BDA17599A3
09458B38AF55B5DCA0943DC2C06C59BF72D965D990D4F31E
2CF55F432E5D7A78E7758252D5823D7BA63938945AAFEB90
7A3EA92B93057E37EA5F15F81ABB42070C868D0AF699BCFC
78A7AA0EF7E673511287729FE8174B514BADE9D171DF9A90
DF99878610F2223313D3B361363D2EAAC0457F84C3ADC945
1ABDCEDDA4292F9B6274B11F4A212DBF66495FF3C0159835
49C59D3982D0540092E5224DE77FD587C2B8C7B50FED988A
57025E2C16E6A58C3267852621FFFC470A908A4626ACB124
4BA406FA4B3F2A449F72A7A8CF1343689B9856FF53A4A1A6
D05654740A2E305B39D6B02A20C0C57CF94DEB1E0B128A43
120AFB71DAB95AC0C8CE8CA25EE41E514873E0CA0C6C76D6
A3C64CB10C15C58B3BE8A8204AF9083AF966A5386B84E94F
32F5B1FD13659E0D952E2790D3F87E86E3FAA9E62D8E772E
FFCAEB301BA6A4D120FB4918A8301E4DD5FB4A199F507466
E2CB80F3C7951DD771A0FA928BC03FF5ACD8F4C8562CFC84
DA2DBF43F4C2E0F9B7BE6E7CDED36ABEDA6C9E75B8E1F8A5
1AED9FE7F48524CB478B75FDB524F21C0492A15481907834
86A2D3E2904B73D5085415C7BF00DF8F3E862E2506B6078A
66E817DD374B3EADC51846153B5737EB0E5B3CBAB8FF2B8B
1ECF1E8C6061868A2A97C8E352E2C55FA740862335796467
A078DE9DCDA0C8C62EDAAC4546DC1F644623A8978C238BF4
D99653B1B1223D1D8C92A3AD442541661F0C1F9BD71EE55C
23D8A9F9098CC6F5B31B537E363E2DD5702BF7395415B01B
55F82191817FABE8D1F3024A13EF0D3825C6A3EBEFEB94E8
D5CA5F8B803E4C8A9124672F0659EA48D2710E4E3E25FDB8
52B854D7FE6386DAF7D801D6FC44D16EA9A6830B622FA0A9
9DE1792122D4C58BE62AE588DEF8FF712859D1D0917F007A
662889AA693CF8D9706593A06F16EE0F7B728672CA49727D
0D6A2F9319A49B1D24186A61CEAD768DBCB0BC9E0C2341A7
49ED631DC6E86459868E52731232E0343BAD2D5C4CCD8732
99B9954947221B86C525724518648380425DBEE4CD41FBF9
93F8A55DBBA5722DB9B5721D0081796AFB17F7CBFDF2116E
EB800CB0C7F76EA5A69C50DFE03A3BB56E76BE73824D9FED
27E6C1A7275D51AC93CBCAB6D4C1B7A3015525B7DD1CEE75
CE31310771CFCCB8800DBE43C017EF3C8A9B99E7690DF9B7
4A720A2F5B25CFA549DE5FE1CD8DD21C30221E6D847DCE8C
C7578513006EACD938778D0365235EA1179E160D283D3AA1
D2DE91D0158B2E63E768E17403E4184A2B27E8E2CC8BFBF3
26A532EA1D835A792F77DB574A7DDD13E1AA4ABBFDB0AA39
96DF3BD657E0B6F6F6381ACCD223C24BB25931B689C56A89
7D4503CAAF66AF13A89A935EB7E8160F824B58366EE922A9
04A3161F590621ED766F2E7B9C06E8861A15F9E8589C4B43
669EDD653A522B124AACB668B02743279A0281A2F059BE6C
5409D5AE9D4DFB198880BA001BC71FC235D3CDBCE440ED15
367067AADA914E6B3BD43EEFCE16C2832C684756B3B0F4CF
D8B0DAA9F3C806BE263961D12A6B457ED1CC2E7750560EE3
362204625952AD8EB255B029A5C073AA8EFDF1C67BCFDFB2
3CE31FD4459880C9DE83EC89B425746DD9E1085A0E9660FF
0F77C92A36269FBAE744FD34A9C68F21548F5DF6B27EF2AA
»;

{
my $msg = 1 +< 511;
my $vnum = 0;
while ($msg) {
    next if $vnum > $testvecs - 1 and not 439 < $vnum < 450;
    is +T2.new.finalize(Buf.new(255 X+& ($msg X+> (504,496...0)))), ("0x" ~ @t2set3[$vnum]).Int, "Tiger2 Set 3, vector #$vnum";
    is +T2u.new.finalize(Buf.new(255 X+& ($msg X+> (504,496...0)))), ("0x" ~ @t2set3[$vnum]).Int, "Tiger2 Set 3, vector #$vnum (indirect)";
    NEXT {
        $vnum++;
        $msg +>= 1;
    }
}
}

#$s2 .= new();
#my $digest = $s2.finalize(Buf.new(48 xx 0));
#is hexify(digest), "1DE8974FA4CBE84A99FD35F7FAF6F12BA07BCD6E4D8E745B", "Set 4, vector#  0 -- initial";
#$digest = T2.new.finalize($digest) for ^100000;
#is hexify(digest), "F7948E9054736D824E1F2807122F020AA270903F82C611CF", "Set 4, vector#  0 -- iterated";
