use v6;
use NativeCall;

#-------------------------------------------------------------------------------
package BSON:auth<github:MARTIM>:ver<0.2.0> {

  # BSON type codes
  constant C-DOUBLE             = 0x01;
  constant C-STRING             = 0x02;
  constant C-DOCUMENT           = 0x03;
  constant C-ARRAY              = 0x04;
  constant C-BINARY             = 0x05;
  constant C-UNDEFINED          = 0x06;         # Deprecated
  constant C-OBJECTID           = 0x07;
  constant C-BOOLEAN            = 0x08;
  constant C-DATETIME           = 0x09;
  constant C-NULL               = 0x0A;
  constant C-REGEX              = 0x0B;
  constant C-DBPOINTER          = 0x0C;         # Deprecated
  constant C-JAVASCRIPT         = 0x0D;
  constant C-DEPRECATED         = 0x0E;         # Deprecated
  constant C-JAVASCRIPT-SCOPE   = 0x0F;
  constant C-INT32              = 0x10;
  constant C-TIMESTAMP          = 0x11;
  constant C-INT64              = 0x12;
  constant C-DECIMAL128         = 0x13;

  constant C-MIN-KEY            = 0xFF;
  constant C-MAX-KEY            = 0x7F;

  #-----------------------------------------------------------------------------
  # Binary type codes
  constant C-GENERIC            = 0x00;
  constant C-FUNCTION           = 0x01;
  constant C-BINARY-OLD         = 0x02;         # Deprecated
  constant C-UUID-OLD           = 0x03;         # Deprecated
  constant C-UUID               = 0x04;
  constant C-MD5                = 0x05;

  constant C-UUID-SIZE          = 16;
  constant C-MD5-SIZE           = 16;

  #-----------------------------------------------------------------------------
  # Fixed sizes
  constant C-INT32-SIZE         = 4;
  constant C-INT64-SIZE         = 8;
  constant C-UINT64-SIZE        = 8;
  constant C-DOUBLE-SIZE        = 8;
  constant C-DECIMAL128-SIZE    = 16;

  #-----------------------------------------------------------------------------
  subset Timestamp of UInt where ( $_ < (2**64 - 1 ) );
}

#-------------------------------------------------------------------------------
class X::BSON is Exception {

  # No string types used because there can be lists of strings too
  has $.operation;                      # Operation method encode/decode
  has $.type;                           # Type to process
  has $.error;                          # Parse error

  method message ( --> Str ) {
    "$!operation\() on $!type, error: $!error\n";
  }
}




#-------------------------------------------------------------------------------
sub encode-e-name ( Str:D $s --> Buf ) is export {
  return encode-cstring($s);
}

#-------------------------------------------------------------------------------
sub encode-cstring ( Str:D $s --> Buf ) is export {
  die X::BSON.new(
    :operation<encode>, :type<cstring>,
    :error("Forbidden 0x00 sequence in '$s'")
  ) if $s ~~ /\x00/;

  return $s.encode() ~ Buf.new(0x00);
}

#-------------------------------------------------------------------------------
sub encode-string ( Str:D $s --> Buf ) is export {
  my Buf $b .= new($s.encode('UTF-8'));
  return [~] encode-int32($b.bytes + 1), $b, Buf.new(0x00);
}

#-------------------------------------------------------------------------------
sub encode-int32 ( Int:D $i --> Buf ) is export {
  Buf.new.write-int32( 0, $i, LittleEndian);
}

#-------------------------------------------------------------------------------
sub encode-int64 ( Int:D $i --> Buf ) is export {
  Buf.new.write-int64( 0, $i, LittleEndian);
}

#-------------------------------------------------------------------------------
sub encode-uint64 ( UInt:D $i --> Buf ) is export {
  Buf.new.write-uint64( 0, $i, LittleEndian);
}

#-------------------------------------------------------------------------------
# encode Num in buf little endian
sub encode-double ( Num:D $r --> Buf ) is export {
  Buf.new.write-num64( 0, $r, LittleEndian);
}

#-------------------------------------------------------------------------------
sub decode-e-name ( Buf:D $b, Int:D $index is rw --> Str ) is export {
  return decode-cstring( $b, $index);
}

#-------------------------------------------------------------------------------
sub decode-cstring ( Buf:D $b, Int:D $index is rw --> Str ) is export {

  my @a;
  my $l = $b.elems;

  while $b[$index] !~~ 0x00 and $index < $l {
    @a.push($b[$index++]);
  }

  # This takes only place if there are no 0x00 characters found until the
  # end of the buffer which is almost never.
  die X::BSON.new(
    :operation<decode>, :type<cstring>,
    :error('Missing trailing 0x00')
  ) unless $index < $l and $b[$index++] ~~ 0x00;

  return Buf.new(@a).decode();
}

#-------------------------------------------------------------------------------
sub decode-string ( Buf:D $b, Int:D $index --> Str ) is export {

  my $size = $b.read-uint32( $index, LittleEndian);
  my $end-string-at = $index + 4 + $size - 1;

  # Check if there are enough letters left
  die X::BSON.new(
    :operation<decode>, :type<string>,
    :error('Not enough characters left')
  ) unless ($b.elems - $size) > $index;

  # Check if the end character is 0x00
  die X::BSON.new(
    :operation<decode>, :type<string>,
    :error('Missing trailing 0x00')
  ) unless $b[$end-string-at] == 0x00;

  return Buf.new($b[$index+4 ..^ $end-string-at]).decode;
}

#-------------------------------------------------------------------------------
sub decode-int32 ( Buf:D $b, Int:D $index --> Int ) is export {
  $b.read-int32( $index, LittleEndian);
}

#-------------------------------------------------------------------------------
sub decode-int64 ( Buf:D $b, Int:D $index --> Int ) is export {
  $b.read-int64( $index, LittleEndian);
}

#-------------------------------------------------------------------------------
# decode unsigned 64 bit integer
sub decode-uint64 ( Buf:D $b, Int:D $index --> UInt ) is export {
  $b.read-uint64( $index, LittleEndian);
}

#-------------------------------------------------------------------------------
# decode to Num from buf little endian
sub decode-double ( Buf:D $b, Int:D $index --> Num ) is export {
  $b.read-num64( $index, LittleEndian);
}
