# Bio/RNA/Treekin.pm
package Bio::RNA::Treekin;
our $VERSION = '0.03';

use 5.006;
use strict;
use warnings;

use Bio::RNA::Treekin::Record;
use Bio::RNA::Treekin::MultiRecord;

1;

__END__


=pod

=encoding UTF-8

=head1 NAME

Bio::RNA::Treekin - Classes for working with I<Treekin> output.

=head1 SYNOPSIS

    use Bio::RNA::Treekin;

    # Read multi-record file (records separated by single '&')
    my $treekin_records = Bio::RNA::Treekin::MultiRecord->new($treekin_file);

    # Iterate over single records.
    while (defined (my $next_treekin_record = $treekin_records->next)) {
        # Print current file name.
        print 'Treekin simulation for ', $next_treekin_record->rates_file, " \n";

        # Print population of minimum 42 at the end of the simulation
        my  $init_pop = $current_treekin_record->init_population;
        my $final_pop = $current_treekin_record->final_population;
        print 'min 42: ', $init_pop->of_min(42), ' => ',
              $final_pop->of_min(42), "\n";
    }

=head1 DESCRIPTION

This module provides auxiliary classes to parse, query and print the data
generated by the reaction kinetics simulation tool I<Treekin>, as well as
multi-landscape simulations as generated by I<BarMap>.

Both I<Treekin> and I<BarMap> are being developed by the Theoretical
Biochemistry Group (TBI) of the University of Vienna. Note that this
package is I<not> developed or supported by the authors of I<Treekin> and
I<BarMap>.

=head1 CLASSES

To process a regular output file of I<Treekin>, use a record object as
provided by L<Bio::RNA::Treekin::Record>. When working with multi-record files
as produced by I<BarMap>, use a record stream from
L<Bio::RNA::Treekin::MultiRecord> to retrieve the individual simulation
records.

Each I<Treekin> record comprises, beside meta-information, a time series of
populations of each minimum. These data are stored as objects of type
C<Bio::RNA::Treekin::PopulationDataRecord>.

Please refer to the documenation of the individual classes for more
information.

=head1 AUTHOR

Felix Kuehnl, C<< <felix@bioinf.uni-leipzig.de> >>


=head1 BUGS

Please report any bugs or feature requests by raising an issue at
L<https://github.com/xileF1337/Bio-RNA-Treekin/issues>.

You can also do so by mailing to C<bug-bio-rna-treekin at rt.cpan.org>,
or through the web interface at
L<https://rt.cpan.org/NoAuth/ReportBug.html?Queue=Bio-RNA-Treekin>.  I will be
notified, and then you'll automatically be notified of progress on your bug as
I make changes.


=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Bio::RNA::Treekin


You can also look for information at:

=over 4

=item * Github: the official repository

L<https://github.com/xileF1337/Bio-RNA-Treekin>

=item * RT: CPAN's request tracker (report bugs here)

L<https://rt.cpan.org/NoAuth/Bugs.html?Dist=Bio-RNA-Treekin>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Bio-RNA-Treekin>

=item * CPAN Ratings

L<https://cpanratings.perl.org/d/Bio-RNA-Treekin>

=item * Search CPAN

L<https://metacpan.org/release/Bio-RNA-Treekin>

=back


=head1 LICENSE AND COPYRIGHT

Copyright 2019-2021 Felix Kuehnl.

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see L<http://www.gnu.org/licenses/>.


=cut

# End of Bio/RNA/Treekin.pm

