use Test::More;

use Module::Generate;

my $windows = $^O eq 'MSWin32';
mkdir 't/lib/unable', 444;

subtest "start" => sub  {
	ok(my $start = Module::Generate->start());
	is(ref $start, 'Module::Generate');
	ok($start = $start->start());
	is(ref $start, 'Module::Generate');
};

subtest "dist" => sub {
	ok(my $dist = Module::Generate->dist('Test'));
	is(ref $dist, 'Module::Generate');
	is(%Module::Generate::CLASS{DIST}, 'Test');
	ok($dist = $dist->dist('Testing'));
	is(ref $dist, 'Module::Generate');
	is(%Module::Generate::CLASS{DIST}, 'Testing');
};

subtest "class" => sub {
	ok(my $class = Module::Generate->class('Test'));
	is(ref $class, 'Module::Generate');
	my $test = %Module::Generate::CLASS{Test};
	is_deeply($test, {NAME => 'Test'});
	ok($class = $class->class('Testing'));
	is(ref $class, 'Module::Generate');
	my $test = %Module::Generate::CLASS{Testing};
	is_deeply($test, {NAME => 'Testing'});
};

subtest "lib" => sub {
	ok(my $lib = Module::Generate->lib('t/'));
	is(ref $lib, 'Module::Generate');
	is(%Module::Generate::CLASS{LIB}, 't/');
	ok($lib = $lib->lib('t/lib'));
	is(ref $lib, 'Module::Generate');
	is(%Module::Generate::CLASS{LIB}, 't/lib');
};

subtest "author" => sub {
	ok(my $author = Module::Generate->author('lnation'));
	is(ref $author, 'Module::Generate');
	is(%Module::Generate::CLASS{AUTHOR}, 'lnation');
	ok($author = $author->author('LNATION'));
	is(ref $author, 'Module::Generate');
	is(%Module::Generate::CLASS{AUTHOR}, 'LNATION');
};

subtest "email" => sub {
	ok(my $email = Module::Generate->email('thisusedtobeanemail@gmail.com'));
	is(ref $email, 'Module::Generate');
	is(%Module::Generate::CLASS{EMAIL}, 'thisusedtobeanemail@gmail.com');
	ok($email = $email->email('email@lnation.org'));
	is(ref $email, 'Module::Generate');
	is(%Module::Generate::CLASS{EMAIL}, 'email@lnation.org');
};

subtest "version" => sub {
	ok(my $version = Module::Generate->version('0.01'));
	is(ref $version, 'Module::Generate');
	is(%Module::Generate::CLASS{VERSION}, '0.01');
	ok($version = $version->version('0.02'));
	is(ref $version, 'Module::Generate');
	is(%Module::Generate::CLASS{VERSION}, '0.02');
};

subtest 'synopsis' => sub {
	ok(my $class = Module::Generate->class('Test'));
	ok($class->synopsis('This is a synopsis'));
	is(%Module::Generate::CLASS{Test}->{SYNOPSIS}, 'This is a synopsis');
};

subtest 'abstract' => sub {
	ok(my $class = Module::Generate->class('Test'));
	ok($class->abstract('This is a abstract'));
	is(%Module::Generate::CLASS{Test}->{ABSTRACT}, 'This is a abstract');
};

subtest 'use' => sub {
	ok(my $class = Module::Generate->class('Test'));
	ok($class->use('Foo'));
	is_deeply(%Module::Generate::CLASS{Test}->{USE}, ['Foo']);	
	ok($class->use('Bar'));
	is_deeply(%Module::Generate::CLASS{Test}->{USE}, ['Foo', 'Bar']);	
};

subtest 'base' => sub {
	ok(my $class = Module::Generate->class('Test'));
	ok($class->base('Foo'));
	is_deeply(%Module::Generate::CLASS{Test}->{BASE}, ['Foo']);	
	ok($class->base('Bar'));
	is_deeply(%Module::Generate::CLASS{Test}->{BASE}, ['Foo', 'Bar']);	
};

subtest 'parent' => sub {
	ok(my $class = Module::Generate->class('Test'));
	ok($class->parent('Foo'));
	is_deeply(%Module::Generate::CLASS{Test}->{PARENT}, ['Foo']);	
	ok($class->parent('Bar'));
	is_deeply(%Module::Generate::CLASS{Test}->{PARENT}, ['Foo', 'Bar']);	
};

subtest 'require' => sub {
	ok(my $class = Module::Generate->class('Test'));
	ok($class->require('Foo'));
	is_deeply(%Module::Generate::CLASS{Test}->{REQUIRE}, ['Foo']);	
	ok($class->require('Bar'));
	is_deeply(%Module::Generate::CLASS{Test}->{REQUIRE}, ['Foo', 'Bar']);	
};

subtest 'our' => sub {
	ok(my $class = Module::Generate->class('Test'));
	ok($class->our('$foo'));
	is_deeply(%Module::Generate::CLASS{Test}->{GLOBAL}, ['$foo']);	
	ok($class->our('$bar'));
	is_deeply(%Module::Generate::CLASS{Test}->{GLOBAL}, ['$foo', '$bar']);	
};

subtest 'begin' => sub {
	ok(my $class = Module::Generate->class('Test'));
	my $ref = sub { 1; };
	ok($class->begin($ref));
	is_deeply(%Module::Generate::CLASS{Test}->{BEGIN}, $ref);	
};

subtest 'unitcheck' => sub {
	ok(my $class = Module::Generate->class('Test'));
	my $ref = sub { 1; };
	ok($class->unitcheck($ref));
	is_deeply(%Module::Generate::CLASS{Test}->{UNITCHECK}, $ref);	
};

subtest 'check' => sub {
	ok(my $class = Module::Generate->class('Test'));
	my $ref = sub { 1; };
	ok($class->check($ref));
	is_deeply(%Module::Generate::CLASS{Test}->{CHECK}, $ref);	
};

subtest 'init' => sub {
	ok(my $class = Module::Generate->class('Test'));
	my $ref = sub { 1; };
	ok($class->init($ref));
	is_deeply(%Module::Generate::CLASS{Test}->{INIT}, $ref);	
};

subtest 'end' => sub {
	ok(my $class = Module::Generate->class('Test'));
	my $ref = sub { 1; };
	ok($class->end($ref));
	is_deeply(%Module::Generate::CLASS{Test}->{END}, $ref);	
};

subtest 'new' => sub {
	ok(my $class = Module::Generate->class('Test'));
	ok($class->new);
	my $ref = sub { 1; };
	ok($class->new($ref));
	is_deeply(%Module::Generate::CLASS{Test}->{SUBS}->{new}->{CODE}, $ref);	
};

subtest 'accessor' => sub {
	ok(my $class = Module::Generate->class('Test'));
	ok($class->accessor('foo'));
	my $ref = sub { 1; };
	ok($class->accessor('foo', $ref));
	is_deeply(%Module::Generate::CLASS{Test}->{SUBS}->{foo}->{CODE}, $ref);	
};

subtest 'macro' => sub {
	ok(my $class = Module::Generate->class('Test'));
	ok($class->macro('test', '{ 1 }'));
	ok($class->macro('test', sub { 1 }));
};

subtest 'sub' => sub {
	ok(my $class = Module::Generate->class('Test'));
	ok($class->sub('bar'));
	my $test = %Module::Generate::CLASS{Test}->{SUBS}->{bar};
	is_deeply($test, {INDEX => 5});
};

subtest 'code' => sub {
	ok(my $class = Module::Generate->class('Test'));
	ok($class->code('{ 1 }'));
	ok($class->code('test', sub { 1 }));
};

subtest 'pod' => sub {
	ok(my $class = Module::Generate->class('Test'));
	ok($class->pod('add some pod'));
	is(%Module::Generate::CLASS{Test}->{SUBS}->{CURRENT}->{POD}, 'add some pod');
};

subtest 'example' => sub {
	ok(my $class = Module::Generate->class('Test'));
	ok($class->example('$example'));
	is(%Module::Generate::CLASS{Test}->{SUBS}->{CURRENT}->{EXAMPLE}, '$example');
};

subtest 'generate_dist' => sub {
	%Module::Generate::CLASS = ();
	ok(my $dist = Module::Generate->dist('Test'));
	ok($dist->lib('./t/lib'));
	eval { $dist->generate };
};

subtest 'generate_use' => sub {
	%Module::Generate::CLASS = ();
	ok(my $mod = Module::Generate->class('Test'));
	ok($mod->lib('./t/lib'));
	ok($mod->use('Foo'));
	ok($mod->parent('Bar'));
	ok($mod->require('Bang'));
	$mod->generate();
};

subtest '_make_path' => sub {
	ok(Module::Generate::_make_path('./t/lib/path'));
	rmdir('./t/lib/path');
	if ($windows) {
		ok(1);
	} else {
		eval { Module::Generate::_make_path('./t/lib/unable/path') };
		like($@, qr/Cannot open file for writing/);
	}
};

subtest '_build_phase' => sub {
	ok(my $code = Module::Generate::_build_phase({
		BEGIN => '{ 1 }'
	}));
	is($code, 'BEGIN { 1 };');
};

subtest '_build_subs' => sub {
	ok(my $sub = Module::Generate::_build_subs({
		SUBS => {
			test => {}
		}
	}));
	is($sub, 'sub test {


}');
	ok(my $sub = Module::Generate::_build_subs({
		SUBS => {
			test => {
				CODE => '{ 1 }'
			}
		}
	}));
	is($sub, 'sub test {1 }');
};

subtest '_build_pod' => sub {
	ok(my $pod = Module::Generate->_build_pod({
		VERSION => '1.0',
		SYNOPSIS => 'JUST A TEST',
		SUBS => {
			test => {}
		}
	}));
};

subtest 'generate' => sub {
	%Module::Generate::CLASS = ();
	Module::Generate->generate();
	eval { 
		Module::Generate->lib('./t/lib')
			->dist('Bar')
			->author('LNATION')
			->email('email@lnation.org')
			->version('0.01')
			->macro('self', sub {
				my ($self, $value) = @_;
			})
			->class('Bar')
				->abstract('A Testing Module')
				->our('$one')
				->begin(sub {
					$one = 'abc';
				})
				->new
		->generate;
	};
	ok(1);
	if ($windows) {
		ok(1);
	} else {
		%Module::Generate::CLASS = ();
		eval { 
			Module::Generate->lib('./t/lib/unable')
				->author('LNATION')
				->email('email@lnation.org')
				->version('0.01')
				->macro('self', sub {
					my ($self, $value) = @_;
				})
				->class('Foo')
					->abstract('A Testing Module')
					->our('$one')
					->begin(sub {
						$one = 'abc';
					})
					->new
			->generate;
		};
		like($@, qr/Cannot open file to write/);
	}
};

subtest _perl_tidy => sub {
	eval { my $kaput = Module::Generate::_perl_tidy(q|kaput[{]}|) };
	like ($@, qr/Exiting because of serious errors/);
};

done_testing;
