package Schedule::Activity;

use strict;
use warnings;
use Ref::Util qw/is_arrayref is_hashref is_plain_hashref/;
use Schedule::Activity::Annotation;
use Schedule::Activity::Attributes;
use Schedule::Activity::Message;
use Schedule::Activity::Node;
use Schedule::Activity::NodeFilter;

our $VERSION='0.2.2';

sub new {
	my ($ref,%opt)=@_;
	my $class=ref($ref)||$ref;
	my %self=(
		config  =>$opt{configuration}//{},
		attr    =>undef,
		valid   =>0,
		built   =>undef,
		reach   =>undef,
		unsafe  =>$opt{unsafe}//0,
	);
	return bless(\%self,$class);
}

# validate()
# compile()
# schedule(activities=>[...])

sub _attr {
	my ($self)=@_;
	$$self{attr}//=Schedule::Activity::Attributes->new();
	return $$self{attr};
}

sub validate {
	my ($self,$force)=@_;
	if($$self{valid}&&!$force) { return }
	$$self{config}//={}; if(!is_hashref($$self{config})) { return ('Configuration must be a hash') }
	my @errors=$self->_validateConfig();
	if(!@errors) { $$self{valid}=1 }
	return @errors;
}

sub _validateConfig {
	my ($self)=@_;
	my $attr=$self->_attr();
	my %config=%{$$self{config}};
	my (@errors,@invalids);
	if(!is_hashref($config{node})) { push @errors,'Config is missing:  node'; $config{node}={} }
	if($config{attributes}) {
		if(!is_hashref($config{attributes})) { push @errors,'Attributes invalid structure' }
		else { while(my ($k,$v)=each %{$config{attributes}}) { push @errors,$attr->register($k,%$v) } }
	}
	if($config{messages}) {
		if(!is_hashref($config{messages})) { push @errors,'Messages invalid structure' }
		else {
		while(my ($namea,$msga)=each %{$config{messages}}) {
			if(!is_hashref($msga)) { push @errors,"Messages $namea invalid structure" }
			elsif(defined($$msga{attributes})&&!is_hashref($$msga{attributes})) { push @errors,"Messages $namea invalid attributes" }
			else { foreach my $kv (Schedule::Activity::Message::attributesFromConf($msga)) { push @errors,$attr->register($$kv[0],%{$$kv[1]}) } }
			if(is_hashref($$msga{message})) {
				if(defined($$msga{message}{alternates})&&!is_arrayref($$msga{message}{alternates})) { push @errors,"Messages $namea invalid alternates" }
				else { foreach my $kv (Schedule::Activity::Message::attributesFromConf($$msga{message})) { push @errors,$attr->register($$kv[0],%{$$kv[1]}) } }
			}
		}
	} } # messages
	while(my ($k,$node)=each %{$config{node}}) {
		if(!is_hashref($node)) { push @errors,"Node $k, Invalid structure"; next }
		Schedule::Activity::Node::defaulting($node);
		my @nerrors=Schedule::Activity::Node::validate(%$node);
		if($$node{attributes}) {
			if(!is_hashref($$node{attributes})) { push @nerrors,"attributes, Invalid structure" }
			else { while(my ($k,$v)=each %{$$node{attributes}}) { push @nerrors,$attr->register($k,%$v) } }
		}
		push @nerrors,Schedule::Activity::Message::validate($$node{message},names=>$config{messages});
		foreach my $kv (Schedule::Activity::Message::attributesFromConf($$node{message})) { push @nerrors,$attr->register($$kv[0],%{$$kv[1]}) }
		if(@nerrors) { push @errors,map {"Node $k, $_"} @nerrors; next }
		@invalids=grep {!defined($config{node}{$_})} @{$$node{next}//[]};
		if(@invalids) { push @errors,"Node $k, Undefined name in array:  next" }
		if(defined($$node{finish})&&!defined($config{node}{$$node{finish}})) { push @errors,"Node $k, Undefined name:  finish" }
	}
	$config{annotations}//={};
	if(!is_hashref($config{annotations})) { push @errors,'Annotations must be a hash' }
	else { while(my ($k,$notes)=each %{$config{annotations}}) {
		push @errors,map {"Annotation $k:  $_"} map {
			Schedule::Activity::Annotation::validate(%$_),
			Schedule::Activity::Message::validate($$_{message},names=>$config{messages})
			} @$notes } }
	return @errors;
}

sub _reachability {
	my ($self)=@_;
	my $changed;
	my %reach=(min=>{},max=>{});
	foreach my $namea (keys %{$$self{built}{node}}) {
		my $nodea=$$self{built}{node}{$namea};
		foreach my $nodeb (@{$$nodea{next}}) {
			$reach{min}{$nodea}{$nodeb}=$$nodea{tmmin};
			$reach{max}{$nodea}{$nodeb}=(($nodea eq $nodeb)?'+':$$nodea{tmmax});
		}
	}
	$changed=1;
	while($changed) { $changed=0;
		foreach my $nodea (keys %{$reach{min}}) {
		foreach my $nodeb (keys %{$reach{min}{$nodea}}) {
		foreach my $nodec (keys %{$reach{min}{$nodeb}}) {
			my $x=$reach{min}{$nodea}{$nodec};
			my $y=$reach{min}{$nodea}{$nodeb}+$reach{min}{$nodeb}{$nodec};
			if(!defined($x)||($x>$y)) {
				$reach{min}{$nodea}{$nodec}=$y;
				$changed=1;
			}
		} } }
	}
	my $triadd=sub {
		my ($x,$y)=@_;
		if($x eq '+') { return '+' }
		if($y eq '+') { return '+' }
		return $x+$y;
	};
	$changed=1;
	while($changed) { $changed=0;
		foreach my $nodea (keys %{$reach{max}}) {
		foreach my $nodeb (keys %{$reach{max}{$nodea}}) {
		foreach my $nodec (keys %{$reach{max}{$nodeb}}) {
			if($nodea eq $nodec) { $reach{max}{$nodea}{$nodec}='+'; next }
			my $x=$reach{max}{$nodea}{$nodec};
			if(defined($x)&&($x eq '+')) { next }
			my $y=&$triadd($reach{max}{$nodea}{$nodeb},$reach{max}{$nodeb}{$nodec});
			if(!defined($x)||($y eq '+')||($x<$y)) {
				$reach{max}{$nodea}{$nodec}=$y;
				$changed=1;
			}
		} } }
	}
	$$self{reach}=\%reach;
	return $self;
}

# These checks ignore any filtering that might be active during construction; these are only sanity checks.
# Recommend stashing the reachability results in $self for later.
#
# Here are the tests and their defined orders.
# 1.  Activity that cannot reach finish
# 2.  Orphaned actions (no activity reaches them)
# 3.  Dual-parent action nodes (with more than a single root activity)  NOT(item2)
# 4.  Dual-finish action nodes  NOT(item3)
# 5.  Dangling actions (cannot reach their finish node)  NOT(item1||item4)
# 6.  Action nodes with tmavg=0  NOT(activity|finish) (this is only a problem if there's a cycle)
#
sub safetyChecks {
	my ($self)=@_;
	my (@errors,$changed);
	$self->_reachability();
	my %reach=%{$$self{reach}};
	#
	# Be very cautious about names versus stringified references.
	my $builtNode=$$self{built}{node};
	my %activities=map {$$builtNode{$_}=>$$builtNode{$_}{finish}} grep {defined($$builtNode{$_}{finish})} keys(%$builtNode);
	my %finishes=map {$_=>1} values(%activities);
	my %actions=map {$_=>$$builtNode{$_}} grep {!exists($activities{$$builtNode{$_}})&&!exists($finishes{$$builtNode{$_}})} keys(%$builtNode);
	#
	my %incompleteActivities=map {$_=>1} grep{!defined($reach{min}{$_}{$activities{$_}})} keys(%activities);
	push @errors,map {"Finish for activity $_ is unreachable"} keys(%incompleteActivities);
	#
	my (%orphans,%dualParent,%dualFinish,%dangling,%infiniteCycle);
	foreach my $action (keys %actions) {
		my $parents=0;
		my $terminals=0;
		foreach my $activity (keys %activities) { if(defined($reach{min}{$activity}{$actions{$action}})) { $parents++ } }
		foreach my $finish   (keys %finishes)   { if(defined($reach{min}{$actions{$action}}{$finish}))   { $terminals++ } }
		if($parents==0)    { $orphans{$action}=1 }
		elsif($parents>1)  { $dualParent{$action}=1 }
		if($terminals>1)   { $dualFinish{$action}=1 }
		elsif(!$terminals) { $dangling{$action}=1 }
		if(($actions{$action}{tmavg}==0)&&(defined($reach{min}{$actions{$action}}{$actions{$action}}))) { $infiniteCycle{$action}=1 }
	}
	push @errors,map {"Action $_ belongs to no activity"} keys(%orphans);
	push @errors,map {"Action $_ belongs to multiple activities"} keys(%dualParent);
	push @errors,map {"Action $_ reaches multiple finish nodes"} keys(%dualFinish);
	push @errors,map {"Dangling action $_"} keys(%dangling);
	push @errors,map {"No progress will be made for action $_"} keys(%infiniteCycle);
	return @errors;
}

sub _buildConfig {
	my ($self)=@_;
	my %base=%{$$self{config}};
	my %res;
	while(my ($k,$node)=each %{$base{node}}) {
		if(is_plain_hashref($node)) { $res{node}{$k}=Schedule::Activity::Node->new(%$node) }
		else { $res{node}{$k}=$node }
		$res{node}{$k}{keyname}=$k;
	}
	my $msgNames=$base{messages}//{};
	while(my ($k,$node)=each %{$res{node}}) {
		my @nexts=map {$res{node}{$_}} @{$$node{next}//[]};
		if(@nexts) { $$node{next}=\@nexts }
		else       { delete($$node{next}) }
		if(defined($$node{finish})) { $$node{finish}=$res{node}{$$node{finish}} }
		$$node{msg}=Schedule::Activity::Message->new(message=>$$node{message},names=>$msgNames);
		if(is_plain_hashref($$node{require})) { $$node{require}=Schedule::Activity::NodeFilter->new(%{$$node{require}}) }
	}
	$$self{built}=\%res;
	return $self;
}

sub compile {
	my ($self,%opt)=@_;
	if($$self{built}) { return }
	my @errors=$self->validate();
	if(@errors) { return (error=>\@errors) }
	$self->_buildConfig();
	if(!$opt{unsafe}) { @errors=$self->safetyChecks(); if(@errors) { return (error=>\@errors) } }
	return;
}

sub _nodeMessage {
	my ($optattr,$tm,$node)=@_;
	my ($message,$msg)=$$node{msg}->random();
	if($$node{attributes}) {
		while(my ($k,$v)=each %{$$node{attributes}}) {
			$optattr->change($k,%$v,tm=>$tm) } }
	if(is_hashref($msg)) { while(my ($k,$v)=each %{$$msg{attributes}}) {
		$optattr->change($k,%$v,tm=>$tm);
	} }
	#
	return Schedule::Activity::Message->new(message=>$message,attributes=>$$msg{attributes}//{});
}

sub findpath {
	my (%opt)=@_;
	my ($tm,$slack,$buffer,@res)=(0,0,0);
	my %tension=(
		slack =>1-($opt{tensionslack} //$opt{tension}//0.5),
		buffer=>1-($opt{tensionbuffer}//$opt{tension}//0.85659008),
	);
	foreach my $k (qw/slack buffer/) { if($tension{$k}>1){$tension{$k}=1}; if($tension{$k}<0){$tension{$k}=0} }
	my ($node,$conclusion)=($opt{start},$opt{finish});
	$opt{attr}->push();
	while($node&&($node ne $conclusion)) {
		push @res,[$tm,$node];
		push @{$res[-1]},_nodeMessage($opt{attr},$tm+$opt{tmoffset},$node);
		$node->increment(\$tm,\$slack,\$buffer);
		if($tm-$tension{slack}*$slack+rand($tension{buffer}*$buffer+$tension{slack}*$slack)<=$opt{goal}) {
			$node=$node->nextrandom(not=>$conclusion,tm=>$tm,attr=>$opt{attr}{attr})//$node->nextrandom(tm=>$tm,attr=>$opt{attr}{attr}) }
		elsif($node->hasnext($conclusion)) { $node=$conclusion }
		else { $node=$node->nextrandom(not=>$conclusion,tm=>$tm,attr=>$opt{attr}{attr})//$node->nextrandom(tm=>$tm,attr=>$opt{attr}{attr}) }
	}
	if($node&&($node eq $conclusion)) {
		push @res,[$tm,$conclusion];
		push @{$res[-1]},_nodeMessage($opt{attr},$tm+$opt{tmoffset},$conclusion);
		$conclusion->increment(\$tm,\$slack,\$buffer);
		$node=undef;
	}
	$opt{attr}->pop();
	return (
		steps =>\@res,
		tm    =>$tm,
		slack =>$slack,
		buffer=>$buffer,
	);
}

sub scheduler {
	my (%opt)=@_; # goal,node,config
	if(!is_hashref($opt{node})) { die 'scheduler called with invalid node' }
	$opt{retries}//=10; $opt{retries}--;
	if($opt{retries}<0) { die $opt{error}//'scheduling retries exhausted' }
	#
	my %path=findpath(
		start     =>$opt{node},
		finish    =>$opt{node}{finish},
		goal      =>$opt{goal},
		retries   =>$opt{retries},
		backtracks=>2*$opt{retries},
		attr      =>$opt{attr},
		tmoffset  =>$opt{tmoffset},
		tensionslack =>$opt{tensionslack} //$opt{tension},
		tensionbuffer=>$opt{tensionbuffer}//$opt{tension},
	);
	if($path{retry}) { return scheduler(%opt,retries=>$opt{retries},error=>$path{error}//'Retries exhausted') }
	my @res=@{$path{steps}};
	my ($tm,$slack,$buffer)=@path{qw/tm slack buffer/};
	if($res[-1][1] ne $opt{node}{finish}) { return scheduler(%opt,retries=>$opt{retries},error=>"Didn't reach finish node") }
	#
	my $excess=$tm-$opt{goal};
	if(abs($excess)>0.5) {
		if(($excess>0)&&($excess>$slack))   { return scheduler(%opt,retries=>$opt{retries},error=>"Excess exceeds slack ($excess>$slack)") }
		if(($excess<0)&&(-$excess>$buffer)) { return scheduler(%opt,retries=>$opt{retries},error=>"Shortage exceeds buffer (".(-$excess).">$buffer)") }
		my ($reduction,$rate)=(0);
		if($excess>0) { $rate=$excess/$slack }
		else          { $rate=$excess/$buffer }
		foreach my $entry (@res[0..$#res]) {
			$$entry[0]=$$entry[0]-$reduction;
			my $dt;
			if($excess>0) { $dt=$rate*($$entry[1]->slack()) }
			else          { $dt=$rate*($$entry[1]->buffer()) }
			$reduction+=$dt;
		}
	}
	foreach my $i (0..$#res) {
		my $dt;
		if($i<$#res) { $dt=$res[$i+1][0]-$res[$i][0] }
		else         { $dt=$opt{goal}-$res[$i][0] }
		$res[$i][3]=$res[$i][4]=0;
		$dt-=$res[$i][1]{tmavg}//0;
		if($dt>0) { $res[$i][4]=$dt }
		else      { $res[$i][3]=-$dt }
	}
	#
	# Full materialization of messages and attributes occurs after all
	# slack/buffer adjustments have been made.  Node attributes are
	# the 'defaults', which message attributes applying later.  This
	# means that all attributes will be applied, but 'set' operations in
	# messages will 'win'.
	#
	# In the future, message selection may occur during path construction,
	# to achieve goals of node filtering, but random message selection
	# here will only see the single message in that result.
	#
	foreach my $i (0..$#res) {
		my $node=$res[$i][1];
		if($$node{attributes}) {
			while(my ($k,$v)=each %{$$node{attributes}}) {
				$opt{attr}->change($k,%$v,tm=>$res[$i][0]+$opt{tmoffset}) } }
		my ($message,$msg)=$res[$i][2]->random();
		$res[$i][1]=Schedule::Activity::Node->new(%$node,message=>$message);
		if(is_hashref($msg)) { while(my ($k,$v)=each %{$$msg{attributes}}) {
			$opt{attr}->change($k,%$v,tm=>$res[$i][0]+$opt{tmoffset});
		} }
	}
	return @res;
}

sub schedule {
	my ($self,%opt)=@_;
	my %check=$self->compile(unsafe=>$opt{unsafe}//$$self{unsafe});
	if($check{error})                  { return (error=>$check{error}) }
	if(!is_arrayref($opt{activities})) { return (error=>'Activities must be an array') }
	my $tmoffset=$opt{tmoffset}//0;
	my %res=(stat=>{slack=>0,buffer=>0});
	if($opt{after}) {
		delete($$self{attr});
		my $attr=$self->_attr();
		push @{$$attr{stack}},$opt{after}{_attr};
		$attr->pop();
		$tmoffset=$opt{after}{_tmmax};
		%{$res{stat}}=(%{$res{stat}},%{$opt{after}{stat}});
	}
	$self->_attr()->push();
	foreach my $activity (@{$opt{activities}}) {
		foreach my $entry (scheduler(goal=>$$activity[0],node=>$$self{built}{node}{$$activity[1]},config=>$$self{built},attr=>$self->_attr(),tmoffset=>$tmoffset,tensionslack=>$opt{tensionslack},tensionbuffer=>$opt{tensionbuffer})) {
			push @{$res{activities}},[$$entry[0]+$tmoffset,@$entry[1..$#$entry]];
			$res{stat}{slack}+=$$entry[3]; $res{stat}{buffer}+=$$entry[4];
			$res{stat}{slackttl}+=$$entry[1]{tmavg}-$$entry[1]{tmmin};
			$res{stat}{bufferttl}+=$$entry[1]{tmmax}-$$entry[1]{tmavg};
		}
		$tmoffset+=$$activity[0];
	}
	$self->_attr()->log($tmoffset);
	if($opt{after}) { unshift @{$res{activities}},@{$opt{after}{activities}} }
	%{$res{attributes}}=$self->_attr()->report();
	if(!$opt{nonote}) { while(my ($group,$notes)=each %{$$self{config}{annotations}}) {
		my @schedule;
		foreach my $note (@$notes) {
			my $annotation=Schedule::Activity::Annotation->new(%$note);
			foreach my $note ($annotation->annotate(@{$res{activities}})) {
				my ($message,$mobj)=Schedule::Activity::Message->new(message=>$$note[1]{message},names=>$$self{config}{messages}//{})->random();
				my %node=(%{$mobj//{}},message=>$message);
				if($$note[1]{annotations}) { $node{annotations}=$$note[1]{annotations} }
				push @schedule,[$$note[0],\%node,@$note[2..$#$note]];
			}
		}
		@schedule=sort {$$a[0]<=>$$b[0]} @schedule;
		for(my $i=0;$i<$#schedule;$i++) {
			if($schedule[$i+1][0]==$schedule[$i][0]) {
				splice(@schedule,$i+1,1); $i-- } }
		$res{annotations}{$group}{events}=\@schedule;
	} }
	$self->_attr()->push(); $res{_attr}=pop(@{$$self{attr}{stack}}); # store a copy in {_attr}
	$self->_attr()->pop();
	$res{_tmmax}=$tmoffset;
	return %res;
}

sub computeAttributes {
	my ($self,@activities)=@_;
	$self->_attr()->push();
	$self->_attr()->reset();
	foreach my $event (sort {$$a[0]<=>$$b[0]} @activities) {
		my ($tm,$node,$msg)=@$event;
		if($$node{attributes}) {
			while(my ($k,$v)=each %{$$node{attributes}}) {
				$self->_attr()->change($k,%$v,tm=>$tm) } }
		if(is_hashref($msg)) { while(my ($k,$v)=each %{$$msg{attributes}}) {
			$self->_attr()->change($k,%$v,tm=>$tm);
		} }
	}
	my %res=$self->_attr()->report();
	$self->_attr()->pop();
	return %res;
}

sub loadMarkdown {
	my ($text)=@_;
	my $list=qr/(?:\d+\.|[-*])/;
	my (%config,@activities,@siblings,$activity,$tm);
	foreach my $line (split(/\n/,$text)) {
		if($line=~/^\s*$/) { next }
		if($line=~/^$list\s*(.*)$/) {
			$activity=$1; $tm=0;
			if($activity=~/(?<name>.*?),\s*(?<tm>\d+)(?<unit>min|sec)\s*$/) {
				$activity=$+{name}; $tm=$+{tm}; if($+{unit} eq 'min') { $tm*=60 } }
			if(defined($config{node}{$activity})) { die "Name conflict:  $activity" }
			push @activities,[$tm,$activity];
			@siblings=();
			$config{node}{$activity}={
				message=>$activity,
				next=>[],
				tmavg=>0,
				finish=>"$activity, conclude",
			};
			$config{node}{"$activity, conclude"}={tmavg=>0};
		}
		elsif($line=~/^\s+$list\s*(.*)$/) {
			if(!$activity) { die 'Action without activity' }
			my $action=$1; $tm=60;
			if($action=~/(?<name>.*?),\s*(?<tm>\d+)(?<unit>min|sec)\s*$/) {
				$action=$+{name}; $tm=$+{tm}; if($+{unit} eq 'min') { $tm*=60 } }
			$action="$activity, $action";
			push @{$config{node}{$activity}{next}},$action;
			$activities[-1][0]||=$tm;
			if(defined($config{node}{$action})) { die "Name conflict:  $action" }
			$config{node}{$action}={
				message=>$action,
				next=>[@siblings,"$activity, conclude"],
				tmavg=>$tm,
			};
			foreach my $sibling (@siblings) { push @{$config{node}{$sibling}{next}},$action }
			push @siblings,$action;
		}
	}
	return (configuration=>\%config,activities=>\@activities);
}

1;

__END__

=pod

=head1 NAME

Schedule::Activity - Generate random activity schedules

=head1 VERSION

Version 0.2.2

=head1 SYNOPSIS

  use Schedule::Activity;
  my $scheduler=Schedule::Activity->new(
    configuration=>{
      node=>{
        Activity=>{
          message=>'Begin Activity',
          next=>['action 1'],
          tmmin=>5,tmavg=>5,tmmax=>5,
          finish=>'Activity, conclude',
        },
        'action 1'=>{
          message=>'Begin action 1',
          tmmin=>5,tmavg=>10,tmmax=>15,
          next=>['action 2'],
        },
        'action 2'=>{
          message=>'Begin action 2',
          tmmin=>5,tmavg=>10,tmmax=>15,
          next=>['Activity, conclude'],
        },
        'Activity, conclude'=>{
          message=>'Conclude Activity',
          tmmin=>5,tmavg=>5,tmmax=>5,
        },
      },
      annotations=>{...},
      attributes =>{...},
      messages   =>{...},
    }
	);
  my %schedule=$scheduler->schedule(activities=>[
		[30,'Activity'],
		...
	]);
  if($schedule{error}) { die join("\n",@{$schedule{error}}) }
  print join("\n",map {"$$_[0]:  $$_[1]{message}"} @{$schedule{activities}});

=head1 DESCRIPTION

This module permits building schedules of I<activities> each containing randomly-generated lists of I<actions>.  This two-level approach uses explicit I<goal> times to construct the specified list of activities.  Within activities, actions are chosen within configured limits, possibly with randomization and cycling, using I<slack> and I<buffer> timing adjustments to achieve the goal.

For additional examples, see the C<samples/> directory.

Areas subject to change are documented below.  Configurations and goals may lead to cases that currently C<die()>, so callers should plan to trap and handle these exceptions accordingly.

=head1 CONFIGURATION

A configuration for scheduling contains the following sections:

  %configuration=(
    node       =>{...}
    attributes =>...  # see below
    annotations=>...  # see below
    messages   =>...  # see below
  )

Both activities and actions are configured as named C<node> entries.  With this structure, an action and activity may have the same C<message>, but must use different key names.

  'activity name'=>{
    message=>...    # an optional message string or object
    next   =>[...], # list of child node names
    finish =>'activity conclusion',
    #
    (time specification)
    (attributes specification)
  }
  'action name'=>{
    message=>...    # an optional message string or object
    next   =>[...], # list of child node names
    #
    (time specification)
    (attributes specification)
  }

The list of C<next> nodes is a list of names, which must be defined in the configuration.  During schedule construction, entries will be I<chosen randomly> from the list of C<next> nodes.  The conclusion must be reachable from the initial activity, or scheduling will fail.  There is no further restriction on the items in C<next>:  Scheduling specifically supports cyclic/recursive actions, including self-cycles.

There is no functional difference between activities and actions except that a node must contain C<finish> to be used for activity scheduling.  Nomenclature is primarily to support schedule organization:  A collection of random actions is used to build an activity; a sequence of activities is used to build a schedule.

=head2 Time specification

The only time specification currently supported is:

  tmmin=>seconds, tmavg=>seconds, tmmax=>seconds

Values must be non-negative numbers.  All three values may be identical.  Note that scheduling to a given goal may be impossible without I<slack> or I<buffer> within some of the actions:

  slack =tmavg-tmmin
  buffer=tmmax-tmavg

The slack is the amount of time that could be reduced in an action before it would need to be removed/replaced in the schedule.  The buffer is the amount of time that could be added to an action before additional actions would be needed in the schedule.

Providing any time value will automatically set any missing values at the fixed ratios 3,4,5.  EG, specifying only C<tmmax=40> will set C<tmmin=24> and C<tmavg=32>.  If provided two time values, priority is given to C<tmavg> to set the third.

Scheduling may be controlled with the tension settings described below.  Future changes may support automatic slack/buffering, univeral slack/buffer ratios, and open-ended/relaxed slack/buffering.

=head2 Messages

Each activity/action node may contain an optional message.  Messages are provided so the caller can easily format the returned schedules.  While message attributes may be used during schedule, the message strings themselves are not used during scheduling.  Messages may be:

  message=>'A message string'
  message=>'named message key'
  message=>['An array','of alternates','chosen randomly']
  message=>{name=>'named message key'}
  message=>{
    alternates=>[
      {message=>'A hash containing an array', attributes=>{...}}
      {message=>'of alternates',              attributes=>{...}}
      {message=>'with optional attributes',   attributes=>{...}}
      {message=>'named message key'}
      {name=>'named message key'}
    ]
  }

Message selection is randomized for arrays and a hash of alternates.  Named messages must exist (see L</"NAMED MESSAGES"> below).  Any attributes are emitted with the attribute response values, described below.

=head1 RESPONSE

The response from C<schedule(activities=>[...])> is:

  %schedule=(
    error=>['list of validation errors, if any',...],
    activities=>[
      [seconds, message],
      ..,
    ],
    annotations=>{
      'group'=>{
        events=>[
          [seconds, message],
        ]
      },
      ...
    },
    attributes=>{
      name=>{
        y  =>(final value),
        xy =>[[tm,value],...],
        avg=>(average, depends on type),
      },
      ...
    },
  )

=head2 Failures

In addition to validation failures returned through C<error>, the following may cause the scheduler to C<die()>:  The activity name is undefined.  The scheduler was not able to reach the named finish node.  The number of retries or backtracking attempts has been exhausted.

The difference between the result time and the goal may cause retries when an excess exceeds the available slack, or when a shortage exceeds the available buffer.

Caution:  While startup/conclusion of activities may have fixed time specifications, at this time it is recommended that actions always contain some slack/buffer.  There is currently no "relaxing mechanism" during scheduling, so a configured with no slack nor buffer must exactly meet the goal time requested.

=head1 SCHEDULING ALGORITHM

The configuration of the C<next> actions is the primary contributor to the schedules that can be built.  As with all algorithms of this type, there are many configurations that simply won't work well:  For example, this is not a maze solver, a best path finder, nor a resourcing optimization system.  Scheduling success toward the stated goals generally requires that actions have different C<tmmin>, C<tmmax>, and C<tmavg>, and that actions permit reasonable repetition and recursion.  Highly imbalanced actions, such as a branch of length 10 and another of length 5000, may always fail depending on the goal.  Neverthless, for the activities and actions so described, how does it work?

The scheduler is a randomized, opportunistic, single-step path growth algorithm.  An activity starts at the indicated node.  At each step, the C<next> entries are filtered and a random action is chosen, then the process repeats.  The selection of the next step is restricted based on the I<current time> (at the end of the action) as follows.

First, a I<random current time> is computed based on the current time, the accumulated slack and buffer, and the tension settings (see below).  If the random current time is less than the goal time, the next action will be a random non-final node, if available, or the final node if all other choices are filtered or unavailable.

If the random current time is greater than the goal time and the final action is listed as a C<next> action, it will be chosen.

In all other cases, a random C<next> action will be chosen.

=head2 Tension

Schedule construction proceeds toward the goal time incrementally, with each action appending its C<tmavg> until the goal is reached.  If the accumulated average times were exactly equal to the goal for the activity, schedules would be unambiguous.  For repeating, recursive scheduling, however, it's necessary to consider scenarios where the actions don't quite reach the goal or where they extend beyond the goal.

=head3 Buffer and Slack

Each activity node and action has buffer and slack, as defined above, that contributes to the accumulated total buffer and slack.  The amount of buffer/slack that contributes to the random current time is controlled by including C<schedule(tensionbuffer=E<gt>value)> and C<tensionslack=E<gt>value>, each between 0 and 1.  Tension effectively controls how little of each contributes toward randomization around the goal.

In the 'laziest' mode, with C<tension=0.0>, all available buffer/slack is used to establish the random current time, increasing the likelihood that it is greater than the goal.  With a lower buffer tension, for example, scheduling is more likely to reach the final activity node sooner, and thus will contain a smaller number of actions on average, each stretched toward C<tmmax>.  With a higher tension, the goal time must be met (or exceeded) before aggressively seeking the final activity node, so schedules will contain a larger number of actions, each compressed toward C<tmmin>.

The tension for slack is similar, with lower values permitting a larger number of actions beyond the goal, each compressed toward C<tmmin>, whereas with tension near 1, scheduling will seek the final activity node as soon as the schedule time exceeds the goal, resulting in a smaller number of activities.

The random computed time is a uniform distribution around the current time, but because actions are scheduled incrementally, this leads to a skewed distribution that favors a smaller number of actions.  See C<samples/tension.png> for the distributions where exactly 100 repeated actions would be expected.

The default values are 0.5 for the slack tension, and ~0.85 for the buffer tension.  This gives an expected number of actions that is very close to C<goal/tmavg>, roughly plus 10% minus 5%.

=head3 Response

The scheduling response contains C<{stat}> that reports the accumulated slack and buffer used for all actions, as well as C<slackttl> and C<bufferttl> which represent the maximum available.  The amount of slack used during scheduling is C<slack/slackttl>, and the same for buffer.  These values can assist with choosing tension settings based on the specific configuration.

=head1 ATTRIBUTES

Attributes permit tracking boolean or numeric values during schedule construction.  The result of C<schedule> contains attribute information that can be used to verify or adjust the schedule.

=head2 Types

The two types of attributes are C<bool> or C<int>, which is the default.  A boolean attribute is primarily used as a state flag.  An integer attribute can be used both as a counter or gauge, either to track the number of occurrences of an activity or event, or to log varying numeric values.

=head2 Configuration

Multiple attributes can be referenced from any activity/action.  For example:

  'activity/action name'=>{
    attributes=>{
      temperature=>{set=>value, incr=>value, decr=>value, note=>'comment'},
      counter    =>{set=>value, incr=>value, note=>'comment'},
      flag       =>{set=>0/1, note=>'comment'},
    },
  }

Any attribute may include a C<note> for convenience, but this value is not stored nor reported.

The main configuration can also declare attribute names and starting values.  It is recommended to set any non-zero initial values in this fashion, since calling C<set> requires that activity to always be the first requested in the schedule.  Boolean values must be declared in this section:

  %configuration=(
    attributes=>{
      flagA  =>{type=>'bool'},
      flagB  =>{type=>'bool', value=>1},
      counter=>{type=>'int',  value=>0},
    },
  )

Attributes within message alternate configurations and named messages are identified during configuration validation.  Together with activity/action configurations, attributes are verified before schedule construction, which will fail if an attribute name is referenced in a conflicting manner.

=head2 Response values

The response from C<schedule> includes an C<attributes> section as:

  attributes=>{
    name=>{
      y  =>(final value),
      xy =>[[tm,value],...],
      avg=>(average, depends on type),
    },
    ...
  }

The C<y> value is the last recorded value in the schedule.  The C<xy> contains an array of all values and the times at which they changed; see Logging.  The C<avg> is roughly the time-weighted average of the value, but this depends on the attribute type.

If an activity containing a unique attribute is not used during construction, the attribute will still be included in the response with its default and initial value.

=head2 Integer attributes

The C<int> type is the default for attributes.  If initialized in C<%configuration>, it may specify the type, or the value, or both.  The default value is zero, but this may be overwritten if the first activity node specifically calls C<set>.

Integer attributes within activity/actions support all of:  C<set>, C<incr>, C<decr>.  There is no actual restriction on type so any Perl L<number> is valid, integers or real numbers, positive or negative.

The reported C<avg> is the overall time-weighted average of the values, computed via a trapezoid rule.  That is, if C<tm=0, value=2> and C<tm=10, value=12>, the average is 7 with a weight of 10.  See Logging for more details.

=head2 Boolean attributes

The C<bool> type must be declared in C<%configuration>.  The value may be specified, but defaults to zero/false.

Boolean attributes within activity/actions support:  C<set>.  Currently there is no restriction on values, but the behavior is only defined for values 0/1.

The reported C<avg> is the percentage of time in the schedule for which the flag was true.  That is, if C<tm=0, value=0>, and C<tm=7, value=1>, and C<tm=10, value=1> is the complete schedule, then the reported average for the boolean will be C<0.3>.

=head2 Precedence

When an activity/action node and a selected message both contain attributes, the value of the attribute is updated first from the action node and then from the message node.  For boolean attributes, this means the "value set in the message has precedence".  For integer attributes, suppose that the value is initially zero; then, if both the action and message have attribute operators, the result will be:

  Action  Message  Value
  set=1   set=2      2
  incr=3  set=4      4
  set=5   incr=6    11
  incr=7  incr=8    15

=head2 Logging

The reported C<xy> is an array of values of the form C<(tm, value)>, with each representing an activity/action referencing that attribute built into the schedule.  Each attribute will have its initial value of C<(0, value)>, either the default or the value specified in C<configuration{attributes}>.

Any attribute may be "fixed" in the log at their current value with the configuration C<name=E<gt>{}>, which is equivalent to C<incr=0> for integers.

Attribute logging always occurs at the beginning and end of the completed schedule, so that all scheduled time affects the weighted average value calculation.  Activities may reset or fix attributes in their beginning or final node; the final node is only the "end of the activity" when C<tmavg=0>.

=head2 Recomputation

Any schedule of activities associated with the initial configuration can generate a standalone attribute report:

  %attributes=$scheduler->computeAttributes(@activities)

This permits manual modification of activities, merging across multiple scheduling runs, or merging of annotations (below) to materialize a final attribute report.  This does not affect the attributes within the C<$scheduler>.

=head1 ANNOTATIONS

A scheduling configuration may contain a list of annotations:

  %configuration=(
    annotations=>{
      'annotation group'=>[
        {annotation configuration},
        ...
      ]
    },
  )

Scheduling I<annotations> are a collection of secondary events to be attached to the built schedule and are configured as described in L<Schedule::Activity::Annotation>.  Each named group can have one or more annotation.  Each annotation will be inserted around the matching actions in the schedule and be reported from C<schedule> in the annotations section as:

  annotations=>{
    'group'=>{
      events=>[
        [seconds, message],
        ...
      ]
    },
  }

Within an individual group, earlier annotations take priority if two events are scheduled at the same time.  Multiple groups of annotations may have conflicting event schedules with event overlap.  Note that the C<between> setting is only enforced for each annotation individually at this time.

Annotations do I<not> update the C<attributes> response from C<schedule>.  Because annotations may themselves contain attributes, they are retained separately from the main schedule of activities to permit easier rebuilding.  At this time, however, the caller must verify that annotation schedules before merging them and their attributes into the schedule.  Annotations may also be built separately after schedule construction as described in L<Schedule::Activity::Annotation>.

Annotations may use named messages, and messages in the annotations response structure are materialized using the named message configuration passed to C<schedule>.

=head1 NAMED MESSAGES

A scheduling configuration may contain a list of common messages.  This is particularly useful when there are a large number of common alternate messages where copy/pasting through the scheduling configuration would be egregious.

  %configuration=(
    messages=>{
      'key name'=>{ any regular message configuration }
      ...
    },
  )

Any message configuration within activity/action nodes may then reference the message by its key as shown above.  During message selection, any string message or configured C<name> will return the message configuration for C<key=name>, if it exists, or will return the string message.  If a configured message string matches a referenced name, the name takes precedence.

The configuration of a named message may only create string, array, or hash alternative messages; it cannot reference another name.

=head1 FILTERING

Action nodes may include prerequisites before they will be selected during scheduling:

  'action name'=>{
    require=>{
      ...
    }
    ...
  }

During schedule construction, the list of C<next> actions will be filtered by C<require> to identify candidate actions.  The current attribute values at the time of selection will be used to perform the evaluation.  The available filtering criteria are fully described in L<Schedule::Activity::NodeFilter> and include attribute numeric comparison and Boolean operators.

Action filtering may be used, together with attribute setting and increments, to prevent certain actions from appearing if others have not previously occurred, or vice versa.  This mechanism may also be used to specify global or per-activity limits on certain actions.

=head2 Slack and Buffer

During scheduling, filtering is evaluated as a I<single pass> only, per activity:  When finding a sequence of actions to fulfill a scheduling goal for an activity, candidates (from C<next>) are checked based on the current attributes.  Action times during construction are based on C<tmavg>, so any filter using attribute average values will be computed as if the action sequence only used C<tmavg>.  After a solution is found, however, actions are adjusted across the total slack/buffer available, so the "materialized average" attribute values can be slightly different.

This should never affect attributes used for a stateful/flag/counter-based filter, because those value changes will still occur in the same sequence.

=head1 IMPORT MECHANISMS

=head2 Markdown

Rudimentary markdown support is included for lists of actions that are all equally likely for a given activity:

  * Activity One, 5min
    - Action one, 1min
    - Action two, 2min
    - Action three, 3min
  2. Activity Two, 5min
    * Action one, 1min
    * Action two, 2min
    * Action three, 3min
  - Activity Three, 5min
    * Action one, 5min

Any list identification markers may be used interchangably (number plus period, asterisks, hyphen).  One or more leading whitespace (tabs or spaces) indicates an action; otherwise the line indicates an activity.  Times are specified as C<\d+min> or C<\d+sec>.  If only a single action is included in an activity, its configured time should be equal to the activity time.

The imported configuration permits an activity to be followed by any of its actions, and any action can be followed by any other action within the activity (but not itself).  Any action can terminate the activity.

The full settings needed to build a schedule can be loaded with C<%settings=loadMarkdown(text)>, and both C<$settings{configuration}> and C<$settings{activities}> will be defined so an immediate call to C<schedule(%settings)> can be made.

=head1 INCREMENTAL CONSTRUCTION

For longer schedules with multiple activities, regenerating a full schedule because of issues with a single activity can be time consuming.  A more interactive approach would be to build and verify the first activity, then review choices for the second activity schedule, append it, and continue.  After full scheduling construction, annotations can be built.

Incremental schedules can be built using the C<after> and C<nonote> options:

  # Use nonote to avoid annotation build at this time
  my $choiceA=$scheduler->schedule(nonote=>1, activities=>[[600,'activity1']);
  my $choiceB=$scheduler->schedule(nonote=>1, activities=>[[600,'activity1']);
  #
  # two or more choices are reviewed and one is selected
  my %res=$scheduler->schedule(after=>$choiceB, activities=>[[600,'activity2']);

The schedule indicated via C<after> signals that the scheduler should build the activities and extend the schedule.  Attributes are automatically loaded from the earlier part of the schedule and affect node filtering normally.

Annotations, which may apply to any node by name, are dropped when the schedule is extended.  This is because a single annotation may have a limit and match nodes across activities, so full regeneration is necessary.  To make generation more efficient, C<nonote> may be set to skip annotation generation in earlier steps.

The final result above does generate annotations, but it's also possible to pass C<nonote> at each step and then generate annotations without adding activities by calling:

  my %res=$scheduler->schedule(after=>$earlierSchedule, activities=>[]);

This functionality is experimental starting with Version 0.2.1.

=head1 BUGS

It is possible for some settings to get stuck in an infinite loop:  Be cautious setting C<tmavg=0> for actions.

=head1 SEE ALSO

L<Schedule::LongSteps> and L<Chronic> address the same type of schedules with slightly different goals.
